/* 
 * Copyright (c) 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */
/*
 * Example mini-kernel that boots from a MultiBoot boot loader
 * and connects to a remote GDB debugger on COM1.
 */

#include <stdio.h>
#include <flux/gdb_serial.h>
#include <flux/x86/multiboot.h>
#include <flux/x86/base_cpu.h>
#include <flux/x86/gdb.h>
#include <flux/x86/pc/reset.h>
#include <flux/debug.h>
#include <flux/smp/smp.h>
#include <flux/machine/spin_lock.h>
#include <flux/c/malloc.h>


/*
 * This function defines our kernel "console device";
 * the calls to printf() below will resolve to calls to putchar().
 * Our implementation here sends characters
 * over the serial line to the remote debugger's console.
 * Since sending characters one at a time in individual messages
 * to the remote debugger is quite slow,
 * we also provide a puts implementation that sends a line at a time.
 */
int putchar(int c)
{
	gdb_serial_putchar(c);
	return c;
}

int puts(const char *s)
{
	gdb_serial_puts(s);
	return 0;
}

/*
 * This function defines how to "exit" the kernel;
 * the call to panic() below will eventually wind up here.
 * This implementation of _exit() notifies the remote debugger
 * that we're exiting, and then reboots the machine.
 */
void _exit(int rc)
{
	printf("exit(%d)\n", rc);
	gdb_serial_exit(0);
	pc_reset();
}

/*
 * The MultiBoot startup code will call this function
 * after setting up the base environment.
 * The kernel command line string passed by the boot loader
 * will have been parsed into separate argv option strings,
 * and options of the form FOO=BAR will be separated out
 * into environment variables,
 * which can be accessed using getenv() as in ordinary programs.
 */


/*
 * This is a sample function for smp_start_cpu()
 * This is the first "OS" C-code that gets executed.
 * It should never return.
 */
void booted(void *num_proc)
{
	int cpuid = smp_find_cur_cpu();
	printf("in booted: Processor %d; %d processors total!\n",
		cpuid, *(int *)num_proc);

	printf("processor %d halting\n", cpuid);
	__asm__("hlt");
}

/*
 * Here is the driver to demonstrate the use of the SMP
 * code in a remote-gdb framework.
 */
int main(int argc, char **argv)
{
	int num_processors;
	volatile int q;
	int num, curr, bsp;

	/* Init the serial line and wait for the remote debugger to start */
	gdb_pc_com_init(1, 0);
	gdb_breakpoint();

	printf("Hello GDB!\n");

	/* Okay, here we init the SMP code */
	if (!smp_initialize()) {	/* smp init succeeded! */
		if ((num_processors = smp_get_num_cpus()) > 1) {
			printf("smp_initialize succeeded: ");
		} else {
			printf("Uniprocessor smp-capable system: ");
		}
	} else {	/* non-IMPS-compliant system. */
		num_processors = smp_get_num_cpus();	/* == 1 */
		printf("smp_initialize failed: ");
	}

	printf("%d processor", num_processors);
	if (num_processors != 1)
		printf("s");
	printf("\n");

	bsp = smp_find_cur_cpu();
	printf("BSP is Processor %d\n", bsp);

	/* Let's find out which processors we have */
	num = 0;
	curr = -1;
	while (num < num_processors)  {
		curr = smp_find_cpu(curr);
		printf("processor %d exists\n", curr);
		assert(curr >= 0);
		if (curr != bsp) {
			/*
			 * Start up the AP's, and have them all
			 * run function booted(&num_processors)
			 */
			smp_start_cpu(curr, booted, &num_processors, 
				(void *)malloc(4096)+4096);
		}
		num ++;
	};

	printf("BSP has started up all the processors\n");
	for (q=0;q<10000000;q++) ;
	printf("BSP is exiting....\n");
	/*
	 * In a `real' system, it would be necessary to make sure
	 * all the AP's have shut down before I exit.
	 */

	return 0;
}

