/* 
 * Copyright (c) 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */
/*
 * Example mini-kernel that boots from a MultiBoot boot loader
 * and connects to a remote GDB debugger on COM1.
 */

#include <stdio.h>
#include <flux/gdb_serial.h>
#include <flux/x86/multiboot.h>
#include <flux/x86/base_cpu.h>
#include <flux/x86/gdb.h>
#include <flux/x86/pc/reset.h>
#include <flux/debug.h>


/*
 * This function defines our kernel "console device";
 * the calls to printf() below will resolve to calls to putchar().
 * Our implementation here sends characters
 * over the serial line to the remote debugger's console.
 * Since sending characters one at a time in individual messages
 * to the remote debugger is quite slow,
 * we also provide a puts implementation that sends a line at a time.
 */
int putchar(int c)
{
	gdb_serial_putchar(c);
	return c;
}
int puts(const char *s)
{
	gdb_serial_puts(s);
	return 0;
}

/*
 * This function defines how to "exit" the kernel;
 * the call to panic() below will eventually wind up here.
 * This implementation of _exit() notifies the remote debugger
 * that we're exiting, and then reboots the machine.
 */
void _exit(int rc)
{
	printf("exit(%d)\n", rc);
	gdb_serial_exit(0);
	pc_reset();
}

/*
 * The MultiBoot startup code will call this function
 * after setting up the base environment.
 * The kernel command line string passed by the boot loader
 * will have been parsed into separate argv option strings,
 * and options of the form FOO=BAR will be separated out
 * into environment variables,
 * which can be accessed using getenv() as in ordinary programs.
 */
int main(int argc, char **argv)
{
	gdb_pc_com_init(1, 0);

	gdb_breakpoint();

	printf("Hello GDB!\n");

	return 0;
}

