/* net/atm/pvc.c - ATM PVC sockets */

/* Written 1995,1996 by Werner Almesberger, EPFL LRC */


#include <linux/config.h>
#include <linux/net.h>		/* struct socket, struct net_proto,
				   struct proto_ops */
#include <linux/atm.h>		/* ATM stuff */
#include <linux/atmdev.h>	/* ATM devices */
#include <linux/atmclip.h>	/* Classical IP over ATM */
#include <linux/errno.h>	/* error codes */
#include <linux/kernel.h>	/* printk */
#include <linux/skbuff.h>
#ifdef CONFIG_AREQUIPA
#include <linux/arequipa.h>
#endif

#include "static.h"		/* static data ... */
#include "common.h"		/* common for PVCs and SVCs */

#ifndef NULL
#define NULL 0
#endif


static int pvc_dup(struct socket *newsock,struct socket *oldsock)
{
	return -EOPNOTSUPP;
}


static int pvc_shutdown(struct socket *sock,int how)
{
	return -EOPNOTSUPP;
}


static int pvc_bind(struct socket *sock,struct sockaddr *sockaddr,
    int sockaddr_len)
{
	struct sockaddr_atmpvc *addr;
	struct atm_vcc *vcc;

	if (sockaddr_len != sizeof(struct sockaddr_atmpvc)) return -EINVAL;
	addr = (struct sockaddr_atmpvc *) sockaddr;
	if (addr->sap_family != AF_ATMPVC) return -EAFNOSUPPORT;
	vcc = ATM_SD(sock);
	if (!(vcc->flags & ATM_VF_HASQOS)) return -EBADFD;
	if (vcc->flags & ATM_VF_PARTIAL) {
		if (vcc->vpi != ATM_VPI_UNSPEC) addr->sap_addr.vpi = vcc->vpi;
		if (vcc->vci != ATM_VCI_UNSPEC) addr->sap_addr.vci = vcc->vci;
	}
	return atm_connect(sock,addr->sap_addr.itf,addr->sap_addr.vpi,
	    addr->sap_addr.vci);
}


static int pvc_connect(struct socket *sock,struct sockaddr *sockaddr,
    int sockaddr_len,int flags)
{
	return pvc_bind(sock,sockaddr,sockaddr_len);
}


static int pvc_listen(struct socket *sock,int backlog)
{
	return -EOPNOTSUPP;
}


static int pvc_accept(struct socket *sock,struct socket *newsock,int flags)
{
	return -EOPNOTSUPP;
}


static int pvc_getname(struct socket *sock,struct sockaddr *sockaddr,
    int *sockaddr_len,int peer)
{
	struct sockaddr_atmpvc *addr;
	struct atm_vcc *vcc;

#if 0 /* add some sanity checks later ... @@@ */
	if (sock->state != SS_CONNECTED) return -EINVAL;
#endif
        *sockaddr_len = sizeof(struct sockaddr_atmpvc);
	addr = (struct sockaddr_atmpvc *) sockaddr;
	vcc = ATM_SD(sock);
	addr->sap_family = AF_ATMPVC;
	addr->sap_addr.itf = vcc->dev->number;
	addr->sap_addr.vpi = vcc->vpi;
	addr->sap_addr.vci = vcc->vci;
	return 0;
}


static int pvc_ioctl(struct socket *sock,unsigned int cmd,unsigned long arg)
{
	/* put PVC-specific code here */
	return atm_ioctl(sock,cmd,arg);
}


static int pvc_setsockopt(struct socket *sock,int level,int optname,
    char *optval,int optlen)
{
	struct atm_vcc *vcc;
	int error;

	error = verify_area(VERIFY_READ,optval,optlen);
	if (error) return -EINVAL;
	vcc = ATM_SD(sock);
	if (level == SOL_ATM && optname == SO_ATMQOS &&
	    sock->state == SS_CONNECTED) {
		struct atm_qos qos;

		if (optlen != sizeof(struct atm_qos)) return -EINVAL;
		if (!vcc->dev->ops->change_qos) return -EOPNOTSUPP;
		memcpy_fromfs(&qos,optval,optlen);
		return vcc->dev->ops->change_qos(vcc,&qos);
	}
	return atm_setsockopt(sock,level,optname,optval,optlen);
}


static int pvc_getsockopt(struct socket *sock,int level,int optname,
    char *optval,int *optlen)
{
	/* put PVC-specific code here */
	return atm_getsockopt(sock,level,optname,optval,optlen);
}


static struct proto_ops pvc_proto_ops = {
	PF_ATMPVC,
	atm_create,
	pvc_dup,
	atm_release,
	pvc_bind,
	pvc_connect,
	NULL,			/* no socketpair */
	pvc_accept,
	pvc_getname,
	atm_select,
	pvc_ioctl,
	pvc_listen,
	pvc_shutdown,
	pvc_setsockopt,
	pvc_getsockopt,
	NULL,			/* no fcntl */
	atm_sendmsg,
	atm_recvmsg
};


/*
 *	Initialize the ATM PVC protocol family
 */


extern int atmdev_init(void);


void atmpvc_proto_init(struct net_proto *pro)
{
	atm_static_init();
	if (sock_register(pvc_proto_ops.family,&pvc_proto_ops) < 0) {
		printk(KERN_ERR "ATMPVC: can't register");
		return;
	}
#ifdef CONFIG_AREQUIPA
	(void) atm_init_arequipa();
#endif
}
