/*
 *
 * Lan Emulation client header file
 *
 * Marko Kiiskila carnil@cs.tut.fi
 *
 */

#ifndef _LEC_H_
#define _LEC_H_

#include <linux/atmdev.h>
#include <linux/if_ether.h>
#include <linux/atmlec.h>

#define LEC_HEADER_LEN 16

struct lecdatahdr_8023 {
  unsigned short le_header;
  unsigned char h_dest[ETH_ALEN];
  unsigned char h_source[ETH_ALEN];
  unsigned short h_type;
};

struct lecdatahdr_8025 {
  unsigned short le_header;
  unsigned char ac_pad;
  unsigned char fc;
  unsigned char h_dst[ETH_ALEN];
  unsigned char h_source[ETH_ALEN];
};

/*
 * ATM LAN Emulation supports both LLC & Dix Ethernet EtherType
 * frames. 
 * 1. Dix Ethernet EtherType frames encoded by placing EtherType
 *    field in h_type field. Data follows immediatelly after header.
 * 2. LLC Data frames whose total length, including LLC field and data,
 *    but not padding required to meet the minimum data frame length, 
 *    is less than 1536(0x0600) MUST be encoded by placing that length
 *    in the the h_type field. The LLC field follows header immediatelly.
 * 3. LLC data frames longer than this maximum MUST be encoded by placing
 *    the value 0 in the h_type field.
 *
 */

/* Hash table size */
#define LEC_ARP_TABLE_SIZE 16

struct lec_priv {
        struct enet_statistics stats;
        unsigned short lecid;      /* Lecid of this client */
        struct lec_arp_table *lec_arp_empty_ones;
        /* Used for storing VCC's that don't have a MAC address attached yet */
        struct lec_arp_table *lec_arp_tables[LEC_ARP_TABLE_SIZE];
        /* Actual LE ARP table */
        struct lec_arp_table *lec_no_forward;
        /* Used for storing VCC's (and forward packets from) which are to
           age out by not using them to forward packets. 
           This is because to some LE clients there will be 2 VCCs. Only
           one of them gets used. */
        atomic_t lec_arp_lock_var;
        struct atm_vcc *mcast_vcc;
        struct atm_vcc *lecd;
        struct timer_list lec_arp_timer;
        /* C10 */
        unsigned int maximum_unknown_frame_count;
/* Within the period of time defined by this variable, the client will send 
   no more than C10 frames to BUS for a given unicast destination. (C11) */
        unsigned long max_unknown_frame_time;
/* If no traffic has been sent in this vcc for this period of time,
   vcc will be torn down (C12)*/
        unsigned long vcc_timeout_period;
/* An LE Client MUST not retry an LE_ARP_REQUEST for a 
   given frame's LAN Destination more than maximum retry count times,
   after the first LEC_ARP_REQUEST (C13)*/
        unsigned short max_retry_count;
/* Max time the client will maintain an entry in its arp cache in
   absence of a verification of that relationship (C17)*/
        unsigned long aging_time;
/* Max time the client will maintain an entry in cache when
   topology change flag is true (C18) */
        unsigned long forward_delay_time;
/* Topology change flag  (C19)*/
        int topology_change;
/* Max time the client expects an LE_ARP_REQUEST/LE_ARP_RESPONSE
   cycle to take (C20)*/
        unsigned long arp_response_time;
/* Time limit ot wait to receive an LE_FLUSH_RESPONSE after the
   LE_FLUSH_REQUEST has been sent before taking recover action. (C21)*/
        unsigned long flush_timeout;
/* The time since sending a frame to the bus after which the
   LE Client may assume that the frame has been either discarded or
   delivered to the recipient (C22) */
        unsigned long path_switching_delay;
};

int lecd_attach(struct atm_vcc *vcc, int arg);
int lec_vcc_attach(struct atm_vcc *vcc, void *arg);
int lec_mcast_attach(struct atm_vcc *vcc, int arg);
int make_lec(struct atm_vcc *vcc);
int send_to_lecd(struct lec_priv *priv,
                 atmlec_msg_type type, unsigned char *mac_addr,
                 unsigned char *atm_addr);
void lec_push(struct atm_vcc *vcc, struct sk_buff *skb);
#endif _LEC_H_

