/* drivers/atm/tneta1570.h - TI TNETA1570 (SAR) declarations */
 
/* Written 1996 by Rolf Fiedler
 * based on the atm drivers by Werner Almesberger, EPFL LRC 
 */

#ifndef DRIVERS_ATM_TNETA1570_H
#define DRIVERS_ATM_TNETA1570_H

#include <linux/atmioc.h>

#ifdef __KERNEL__

#include <linux/atm.h>
#include <linux/atmdev.h>
#include <linux/sonet.h>
#include <linux/skbuff.h>
#include <linux/time.h>

#define NR_VCI     256
#define NR_VCI_LD  8
#define NR_VPI     32
#define NR_VPI_LD  5
#define MAX_VCI    255
#define MAX_VPI    255

/* this is better put in pci.h, but I want to keep my driver independent */
#define MEM_VALID 0xfffffff0
#ifndef PCI_VENDOR_ID_TI
#define PCI_VENDOR_ID_TI 0x104c
#endif
#ifndef PCI_DEVICE_ID_TI_TNETA1570 
#define PCI_DEVICE_ID_TI_TNETA1570 0xa001
#endif
#ifndef PCI_DEVICE_ID_TI_TNETA1575
#define PCI_DEVICE_ID_TI_TNETA1575 0xa102
#endif


#define DEV_LABEL	 "tneta157x"
#define RAM_INCREMENT	1024		/* check in 4 kB increments */

#ifdef CONFIG_ALPHA_CIA
#define TNETA1570_MEM_OFFSET	CIA_DENSE_MEM
#define TNETA_ALIGN		7
#elif CONFIG_ALPHA_LCA
#define TNETA1570_MEM_OFFSET	LCA_DENSE_MEM
#define TNETA_ALIGN		7
#else
#define TNETA1570_MEM_OFFSET	0
#define TNETA_ALIGN		3
#endif


/*---------------------------------------------------------*/

/* transmit DMA state table, in control memory */
struct tneta_tx_dma_state_table{
     unsigned int control;
     unsigned int current_buffer_ptr;
     unsigned int atm_header;
     unsigned int dma_state_flag;          /* only initialized value */
     unsigned int next_buffer_ptr;
     unsigned int start_of_buffer_address;
     unsigned int partial_AAL5_tx_CRC;
     unsigned int AAL5_control_length_field;
};

/* receive DMA state table, in control memory */
#define RX_DMA_CONTROL_AAL5 ((1<<29)|(1<<26))
#define RX_DMA_CONTROL_AAL0 ((1<<29)|(1<<27))
#define RX_TIME_OUT         (0xfff<<12)
#define DMA_ON              (1<<31)          
#define DMA_FILTER          (1<<30)
struct tneta_rx_dma_state_table{
     unsigned int control;             /* needs initialization */
     unsigned int current_buffer_ptr;
     unsigned int sob_ptr;
     unsigned int EOP;
     unsigned int sop_ptr;
     unsigned int AAL0_cells;          /* needs initialization if AAL0 */
     unsigned int dma_on_idx;          /* needs initialization */
     unsigned int rx_timeout;          /* needs initialization */
};

#define MAX_AAL5_PDU	9600	        /* max size for AAL5 PDUs buffers */
#define MAX_AAL5_CELLS  200
#define MAX_AAL0_PDU    48
#define MAX_AAL0_CELLS  1
#define AAL0_BUFS       32
#define AAL5_BUFS       16
#define FBR_AAL0_32     ((2<<16) | (1<<10) | (0))
#define FBR_AAL5_16     ((MAX_AAL5_CELLS<<16) | (0<<10) | (0))
#define RX_HDR          20
#define MAX_FBR_ENTRIES 256
struct tneta_rx_fbrptr {
        unsigned int buf_ptr;
        unsigned int buf_size;
};

#define AAL5_IND    	(1<<26)
#define PAKET_OVERFLOW	(1<<31)
#define CRC_ERROR	(1<<30)
#define BUFFER_STARV	(1<<29)
#define RX_TIMEOUT	(1<<28)
#define RX_ABORT	(1<<27)
#define RX_ERRORCOND	0xf000

struct tneta_rx_compl {
     unsigned int atm_header;
     unsigned int error;
     unsigned int buf_ptr;
     unsigned int trailer;
     unsigned int control;             /* needs initialization */
     unsigned int res1;
     unsigned int res2;
     unsigned int res3;
};

/* device dependent vcc data */
#define TX_SEG_RING_SIZE 256

struct tneta1570_vcc {
	/*-------------------------------- RX part */
        int (*rx)(struct atm_vcc *vcc);
        int rxing;                        /* pending cells */
        struct wait_queue * rx_wait;      /* for close */
        int dma_channel;                  /* for close */
        int fbr_idx;
        /*-------------------------------- TX part */
        struct wait_queue * tx_wait;      /* for close */
        int txing;                        /* cells to be tx'd */
        int scheduler_idx;                /* index in scheduler table */
        unsigned int * seg_ring;          /* size aligned 1K */
        unsigned int * seg_ring_mptr;     /* word aligned */
        int seg_ring_idx;
   
        struct atm_vcc *next;             /* next pending rx */
        struct sk_buff *last;             /* last PDU */
};


/* device dependent data */
#define TXCMPLR_SZ_IRQ 256   /* size aligned */
#define TXCMPLR_SZ_NOI 128   /* size aligned */
#define RXCMPLR_SZ_IRQ 128   /* size aligned */
#define RXCMPLR_SZ_NOI 64    /* size aligned */
#define SAR_REG_WORD(dev, x) (dev->reg[x])
#define SAR_REG_SHORT(dev, x) (((unsigned short *)dev->reg)[x])
#define MAX_SCHED_ENTRIES 1022
#define MAX_TXDMA_ENTRIES 1022
#define TX_CMPL_R_IRQ(dev)  (dev->txcmplringptr_irq[dev->txcmpl_ring_idx_irq])
#define TX_CMPL_R_NOI(dev)  (dev->txcmplringptr_noi[dev->txcmpl_ring_idx_noi])
#define RX_CMPL_R_IRQ(dev)  (dev->rxcmplringptr_irq[dev->rxcmpl_ring_idx_irq])
#define RX_CMPL_R_NOI(dev)  (dev->rxcmplringptr_noi[dev->rxcmpl_ring_idx_noi])

struct tneta1570_dev {
	/*-------------------------------- TX part */
        int txcmpl_ring_idx_noi, txcmpl_ring_idx_irq;
        unsigned int *txcmplringptr_noi, *txcmplringptr_irq;
        unsigned int *txcmpl_ring;

        /*-------------------------------- RX part */
        int rxcmpl_ring_idx_noi, rxcmpl_ring_idx_irq;
        struct tneta_rx_compl *rxcmplringptr_noi, *rxcmplringptr_irq;
        struct tneta_rx_compl *rxcmpl_ring;

        /*-------------------------------- maps */
        int oam_fbr_idx;
   
        /*-------------------------------- stats */
        unsigned int lost;              /* lost rx cells */
        /*-------------------------------- other pointers */

	/*-------------------------------- TNETA links */
	struct atm_dev *more;		/* other TNETA devices */
	/*-------------------------------- general information */
   
        volatile unsigned int *ram;           /* base of phy device */
        volatile unsigned int *scheduler;     /* base of scheduler table */
        volatile unsigned int *reg;           /* base of sar regs device */
        volatile struct tneta_rx_fbrptr *free_buf_ptr;  /* free buffer pointers */
        volatile unsigned int *rx_vpi_vci;    /* rx vpi vci table */
        volatile struct tneta_tx_dma_state_table *tx_dma_state;  /* tx dma state table */
        volatile struct tneta_rx_dma_state_table *rx_dma_state;  /* rx dma state table */
        volatile unsigned int *phy;           /* base of phy device */
   
	int mem;			/* RAM on board (in bytes) */
	void * base;		        /* board base address */
        unsigned long base_diff;        /* virtual - phy base address */
        unsigned int pci_map_size;      /* pci map size of board */     
        unsigned char irq;		/* IRQ */
        unsigned char chiptype;		/* 1570 = 0,  1575 = 1 */
	unsigned char bus;		/* PCI stuff */
	unsigned char dev_fn;
};


#define TNETA1570_DEV(d) ((struct tneta1570_dev *) (d)->dev_data)
#define TNETA1570_VCC(d) ((struct tneta1570_vcc *) (d)->dev_data)

/*---------------------------------------------------------*/

#include <linux/atmioc.h>

#define TNETA_LOOP	_IOW('a',ATMIOC_SARPRV+1,int) 
		/* set/reset tneta loopback mode*/
#define TNETA_INVHEC	_IOW('a',ATMIOC_SARPRV+2,long) 
		/* set/reset inverse HEC generation */
#define TNETA_ENTX	_IOW('a',ATMIOC_SARPRV+3,long) 
		/* enable/disab txing */
#define TNETA_ENRX	_IOW('a',ATMIOC_SARPRV+4,long) 
		/* enable/disabable  rxing */
#define TNETA_BP	_IOW('a',ATMIOC_SARPRV+5,long) 
		/* intr per paket or per buffer ring     */
#define TNETA_RAT	_IOW('a',ATMIOC_SARPRV+6,int) 
		/* enable/disable  rx aging timer*/
#define TNETA_RXUNKN	_IOR('a',ATMIOC_SARPRV+7,unsigned long) 
		/* header of an unknown received cell */
#define TNETA_HECERR	_IOR('a',ATMIOC_SARPRV+8, unsigned long) 
		/* number of HEC errors received */
#define TNETA_AAL5DISC	_IOR('a',ATMIOC_SARPRV+9, unsigned long) 
		/* number of AAL 5 pdus discarded by tneta */
#define TNETA_RXCELL	_IOR('a',ATMIOC_SARPRV+10, unsigned long) 
		/* number of cells received */
#define TNETA_TXCELL	_IOR('a',ATMIOC_SARPRV+11, unsigned long) 
		/* number of cells transmitted */


						

/*
 * Directly Addressable Registers, memory mapped
 */
#define TNETA_REG_BASE_OFFSET   0x3200   /* offset PCI base to registers */
#define TNETA_CONFIG         0   /* configuration register */
#define TNETA_STATUS         1   /* status register */
#define TNETA_INT_MASK       2   /* interrupt mask register */ 
#define TNETA_S_RAT          6   /* 3L  RAT cycle # rx DMA state table */
#define TNETA_S_RGT          7   /* 3H global reass. timer */
#define TNETA_RXUNKNOWN      4   /* rx unknown register */
#define TNETA_S_TXCOMPLSIZEI 10  /* 5L TX compl. ring size W/ interrupt */
#define TNETA_S_TXCOMPLSIZE  11  /* 5H TX compl. ring size W/O interrupt */
#define TNETA_S_RXCOMPLSIZEI 12  /* 6L RX completion ring size W/ interrupt */ 
#define TNETA_S_RXCOMPLSIZE  13  /* 6H RX completion ring size W/O interrupt */ 
#define TNETA_S_FIFO_FREE    14  /* 7L FIFO free-buffer-size */
#define TNETA_S_TXSEGSIZE    15  /* 7H Segmentation ring size */
#define TNETA_S_AAL_DISCARD  16  /* 8L discarded AAL5 rx cell counter */
#define TNETA_S_HEC_ERR      17  /* 8H HEC error counter */
#define TNETA_UNKNOWN_P      9   /* Unknown Protocols RX # */
#define TNETA_CELL_RXC       10  /* ATM Cells RX'd # */
#define TNETA_CELL_TXC       11  /* ATM Cells TX'd # */
#define TNETA_S_TXM_RXM      24  /* 12L TX FIFO & RX FIFO max occupancy */ 
#define TNETA_TXM_RXM	     12  /* TX FIFO & RX FIFO for 1575 */
#define TNETA_S_VCIM         25  /* 12H VCI mask */ 
#define TNETA_S_SCHEDSIZE    26  /* 13L scheduler-table-size register */
#define TNETA_SCHEDSIZE	     13  /* scheduler-table-size register 1575 only */
#define TNETA_RESET          14  /* software reset register */
#define TNETA_TXQUEUE	     15  /* TX queue register  1575 only*/
#define TNETA_RXCOMPLCNT     16  /* RX compl ring count 1575 only */
#define TNETA_SIDEFBR1CNT    17  /* sideband free buffer ring cnt 1575 only */
#define TNETA_SIDEFBR2CNT    18  /* sideband free buffer ring cnt1575 only */
#define TNETA_RXTRES         19  /* rx compl ring threshold 1575 only */
#define TNETA_SIDERXTRES     20  /* sideband free buffer ring thresh 1575 only */
#define TNETA_SIDEFBR1PTR    21  /* sideband free buffer ring ptr 1575 only */
#define TNETA_SIDEFBR2PTR    22  /* sideband free buffer ring ptr 1575 only */
#define TNETA_SIDEFBR1SIZE   23  /* sideband free buffer ring size 1575 only */
#define TNETA_SIDEFBR2SIZE   24  /* sideband free buffer ring size 1575 only */
#define TNETA_CHNLADD1       25  /* channel add/del register 1 1575 only */
#define TNETA_CHNLADD2       26  /* channel add/del register 2 1575 only */
#define TNETA_TXPAUSE        27  /* transmit pause 1575 only */
#define TNETA_DMACOUNT       28  /*  dma channel count 1575 only */

#define TNETA_TXCOMPLNOI     128 /* TX completion ring W/O interrupt pointer */
#define TNETA_TXCOMPLIRQ     129 /* TX completion ring W/  interrupt pointer */
#define TNETA_RXCOMPLNOI     130 /* RX completion ring W/O interrupt pointer */
#define TNETA_RXCOMPLIRQ     131 /* RX completion ring W/  interrupt pointer */

/* configuration register bits */
#define TNETA_R0_TXBLOCK   0x40000 /* block txing 1575 only */
#define TNETA_R0_ENBBUS    0x20000 /* enable bus reports 1575 only */
#define TNETA_R0_NOPOLLTX  0x10000 /* enable performance tx 1575 only */
#define TNETA_R0_NOPOLLRX  0x8000  /* enable performance rx 1575 only */
#define TNETA_R0_COPRO     0x4000  /* coprocessor present 1575 only */
#define TNETA_R0_UNI       0x2000  /* uni / nni */
#define TNETA_R0_MAX_RETRY (0xf << 9)  /* 1111 max retry master */
#define TNETA_R0_LOOP      0x0100      /* set to loop-back (reset!, no enable) */
#define TNETA_R0_TX_HECERR 0x0080      /* force HEC error */
#define TNETA_R0_SMALL_MAP 0x0040
#define TNETA_R0_TX_ENABLE 0x0020
#define TNETA_R0_RX_ENABLE 0x0010
#define TNETA_R0_ENDIAN    (0x0 << 3)  /* 00 little endian */
#define TNETA_R0_PERBUFFER 0x002
#define TNETA_R0_RAT_ENABL 0x001       /* enable reass. aging timer */
#define TNETA_R0_STANDARD_MODE 0x00002000
                             
/* status register bits */

#define TNETA_R1_LOCALBUS  0x8000	/* local bus interrupt ? 1575 only */
#define TNETA_R1_COPRO     0x4000	/* copro interrupt ?  1575 only */
#define TNETA_R1_SIDE1     0x2000	/* rx sideband 1 interrupt? 1575 only */
#define TNETA_R1_SIDE2     0x1000	/* rx sideband 2 interrupt? 1575 only */
#define TNETA_R1_RXCOMP	   0x800	/* rx compl half inter. ? 1575 only */
#define TNETA_R1_RXHALF    0x400	/* Rx fifo half full  1575 only */
#define TNETA_R1_PCI_MODE  0x400        /* 32/64 bit */
#define TNETA_R1_RX_FREEZE 0x200        /* rx ring overflow */
#define TNETA_R1_TX_FREEZE 0x100        /* tx ring overflow */
#define TNETA_R1_CP_RX     0x080        /* packet reassembly completed */
#define TNETA_R1_RX_IRR    0x040        /* rx-unknown written */
#define TNETA_R1_HEC_OVER  0x020        /* HEC error counter overflow */
#define TNETA_R1_UP_OVER   0x010        /* unknown proto counter overflow */
#define TNETA_R1_APD_OVER  0x008        /* AAL5 PDU discard counter overflow */
#define TNETA_R1_ACR_OVER  0x004        /* ATM cell rxd counter overflow */
#define TNETA_R1_ACT_OVER  0x002        /* ATM cell txd counter overflow */
#define TNETA_R1_CP_TX     0x001        /* packet segmentation completed */
#define TNETA_R2_STANDARD_INTS (TNETA_R1_RX_FREEZE | \
                                TNETA_R1_TX_FREEZE | \
				TNETA_R1_RX_IRR | \
                                TNETA_R1_CP_RX | \
                                TNETA_R1_CP_TX)
/* control memory map offsets */
#define TNETA_SCHED_TABLE             0x0
#define TNETA_FREE_BUFFER_POINTERS    0x3800
#define TNETA_RX_VPIVCI_DMA_POINTERS  0x4000
#define TNETA_TX_DMA_STATE_TABLE      0x8000
#define TNETA_RX_DMA_STATE_TABLE      0x10000
#define TNETA_SUNI_OFFSET             (0x20000 << 2)

/* reserved control memory area */
#define RESERVED_LL                   0xc00   /* lower limit */
#define RESERVED_UL                   0xe00   /* upper limit */

#define TNETA_SUNI_RDREQ              0x00100
#define TNETA_SUNI_RD_D_AV            (0x10)

#define OWN (0x1 << 31)        /* own bit, set to 1 if owned by tneta1570 */
#define SEG_PTR(x) (0x3fffff00 & ((unsigned int)virt_to_bus(x) >> 2))
#define BUF_PTR(x) (((unsigned int)x >> 2) & 0x3fffffff)
                   /* pointer to tx data buffer header, aligned to 4 byte */

/* tx data buffer header */
struct tx_buffer_descriptor{
     unsigned int control;
     unsigned int next_buffer;
     unsigned int atm_header;
     unsigned int AAL5_control;
};

/* control word layout */
#define TNETA_TXDBH_CTRL_RDY        (0x1 << 31)
#define TNETA_TXDBH_CTRL_SOP        (0x1 << 30)
#define TNETA_TXDBH_CTRL_EOP        (0x1 << 29)
#define TNETA_TXDBH_CTRL_ABORT      (0x1 << 28)
#define TNETA_TXDBH_CTRL_AAL0_PTI   (0x0 << 26)  
#define TNETA_TXDBH_CTRL_AAL5       (0x1 << 26)  
#define TNETA_TXDBH_CTRL_AAL0_NOPTI (0x2 << 26)  
#define TNETA_TXDBH_CTRL_INT_NOINT  (0x1 << 25)
#define TNETA_TXDBH_CTRL_BUF_OFFSET (0xff << 16)
#define TNETA_TXDBH_CTRL_BYTE_COUNT (0xffff << 0)

#define AAL5_PDU_INT (TNETA_TXDBH_CTRL_RDY | \
                      TNETA_TXDBH_CTRL_SOP | \
                      TNETA_TXDBH_CTRL_EOP | \
                      TNETA_TXDBH_CTRL_AAL5 | \
                      TNETA_TXDBH_CTRL_INT_NOINT)

#define AAL0_PDU_INT (TNETA_TXDBH_CTRL_RDY | \
                      TNETA_TXDBH_CTRL_SOP | \
                      TNETA_TXDBH_CTRL_EOP | \
                      TNETA_TXDBH_CTRL_AAL0_PTI)

/* ATM header */
#define TNETA_TXDBH_ATMH_GFC        (0xff << 24)
#define TNETA_TXDBH_ATMH_VPI        (0xff << 16)
#define TNETA_TXDBH_ATMH_VCI        (0xffff << 4)
#define TNETA_TXDBH_ATMH_PTI        (0x7 << 1)
#define TNETA_TXDBH_ATMH_CLP        (0x1 << 0)					
	
/* AAL5 control */
#define TNETA_TXDBH_AAL5_CPCS_UU    (0xff << 24)
#define TNETA_TXDBH_AAL5_CPI        (0xffff << 16)
#define TNETA_TXDBH_AAL5_USER       (0xffff << 0)

/* TX completion rings */
#define TNETA_TXCR_OWN         (0x1 << 31)
#define TNETA_TXCR_ABORT       (0x1 << 30)
#define TNETA_TXCR_BUFADDR     (0x3fffffff << 0)

#endif __KERNEL__
#endif
