/* drivers/atm/tneta1570.c - ti tneta1570 atm driver 
 * 
 *   Copyright (c) 1996 University of Technology Chemnitz
 *
 *   This program is free software; you can redistribute it and/or modify 
 *   it under the terms of the GNU General Public License as published by 
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version. 
 *  
 *   This program is distributed in the hope that it will be useful, 
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *   GNU General Public License for more details. 
 * 
 *   You should have received a copy of the GNU General Public License 
 *   along with this program; if not, write to the Free Software 
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 * 
 *   Written 1996 by Rolf Fiedler (atm@infotech.tu-chemnitz.de) 
 * 
 * 08/08/1996	adapted to axp, phy access, 64 bit bus access  
 * 	        some minor bugs in dma allocation,segring overflow, 
 *              timeout,... removed 
 * 	        (paetz@infotech.tu-chemnitz.de) 
 * 10/01/1996   bug in dma channel allocattion fixed (multiple channels)
 * 
*/

#include <linux/config.h> /* for extended debugging options */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/bios32.h>
#include <linux/pci.h>
#include <linux/errno.h>
#include <linux/atm.h>
#include <linux/atmdev.h>
#include <linux/sonet.h>
#include <linux/skbuff.h>
#include <linux/time.h>
#include <linux/sched.h> /* for xtime */
#include <linux/delay.h>
#include <linux/uio.h>
#include <asm/io.h>
#include <asm/system.h>
#include <asm/string.h>
#include <asm/byteorder.h>

#include "tneta1570.h"
#include "suni.h"

#define SLOW_DOWN_FACTOR 8
#define TNETA1570_TIMEOUT	5*HZ
/*
 * KNOWN BUGS:
 * OAM support ??? , various vci/vpis
 * doesn't work a bit
 */

inline void write_sched_tab(struct tneta1570_dev * dev, int i, unsigned int v);
inline int read_sched_tab(struct tneta1570_dev * dev, int i);
static void dequeue_OAM(struct atm_dev *dev);   /* dequeue OAM cells */
static void dequeue_AAL0(struct atm_dev *dev);  /* dequeue AAL0 cells */
static void dequeue_AAL5(struct atm_dev *dev);  /* dequeue AAL5 cells */
static int alloc_dma(struct atm_vcc *vcc);
static unsigned char tneta1570_phy_get(struct atm_dev *,unsigned long);
static void tneta1570_phy_put(struct atm_dev *,unsigned char, unsigned long);


#if 0
#define DPRINTK(format,args...) printk(format,args...)
#else
#define DPRINTK(format,args...) 
#endif

#ifndef CONFIG_ATM_TNETA1570_DEBUG


#define NULLCHECK(x)

#define EVENT(s,a,b)


static inline void event_dump(void)
{
}


#else


/* 
 * NULL pointer checking
 */

#define NULLCHECK(x) \
  if ((unsigned long) (x) < 0x30) printk(#x "==0x%x\n", (int) (x))

/*
 * Very extensive activity logging. Greatly improves bug detection speed but
 * costs a few Mbps if enabled.
 */

#define EV 64

static const char *ev[EV];
static unsigned long ev_a[EV],ev_b[EV];
static int ec = 0;

static void EVENT(const char *s,unsigned long a,unsigned long b)
{
	ev[ec] = s; 
	ev_a[ec] = a;
	ev_b[ec] = b;
	ec = (ec+1) % EV;
}


static void event_dump(void)
{
	int n,i;

	for (n = 0; n < EV; n++) {
		i = (ec+n) % EV;
		printk(ev[i] ? ev[i] : "(null)",ev_a[i],ev_b[i]);
	}
}


#endif /* CONFIG_ATM_TNETA1570_DEBUG */


static int tx_complete = 0,dma_complete = 0,queued = 0,requeued = 0,
  backlogged = 0,rx_enqueued = 0,rx_dequeued = 0,pushed = 0,submitted = 0,
  putting = 0;

static struct atm_dev *tneta1570_boards = NULL;


/*-------------------------------- utilities --------------------------------*/


static void dump_mem(struct tneta1570_dev *tneta1570_dev)
{
}


static void dump(struct atm_dev *dev)
{
    struct tneta1570_dev *tneta1570_dev;

    tneta1570_dev = TNETA1570_DEV(dev);
    printk("\nFree memory\n");
    dump_mem(tneta1570_dev);
    printk("TX buffers\n");

    printk("RX buffers\n");

    printk("----\n");
}

/* --------------------------memory allocation-------------------------------*/


/*
 * allocate a skbuff with enough room for the SDU header
 */
static struct sk_buff *tneta1570_alloc_tx(struct atm_vcc *vcc, unsigned int size)
{
        struct sk_buff *ptr;
   
        ptr = alloc_skb(size + 48, GFP_KERNEL);
        if(ptr == NULL) return NULL;
        skb_reserve(ptr, 48);        /* room for push */
        return ptr;
}

/*------------------------------ utopia phy dummies ------------------------ */
static int utopia_start(struct atm_dev *dev)
{return 0;}
static void utopia_int(struct atm_dev *dev)
{}
static int utopia_ioctl(struct atm_dev *dev, unsigned cmd, unsigned long arg)
{return 0;}
static const struct atmphy_ops utopia_ops = {
	utopia_start,
	utopia_ioctl,
	utopia_int
};
int utopia_init(struct atm_dev *dev)
{
	dev->phy = &utopia_ops;
	return 0;
}

/*----------------------------------- RX ------------------------------------*/

/*
 * alloc_dma
 * find entry in dma state table for this vpi/vci
 * if vpi already open increase valid vci range
 * else open vpi, set vci range to vci
 * return index to dma channel, return 0 on error
 */
static int alloc_dma(struct atm_vcc *vcc)
{
        unsigned char sorted[0x78];
        unsigned int x;
        int i;
        struct tneta1570_dev *tneta1570_dev;

        EVENT(">alloc_rx_dma\n",0,0);
   
        tneta1570_dev = TNETA1570_DEV(vcc->dev);
         
        x = tneta1570_dev->rx_vpi_vci[vcc->vpi];
        if(x != 0) {
	        x = 0x7fff & (x >> 16);
	        return  x + vcc->vci - 0x800;  /* idx = value - offset */
	} else {
		for(i=0;i<0x78;i++)
  			sorted[i]=0x00;  
                /* create a new vpi entry */
	        /* first, find dma channel range */
                for(i=0; i<=MAX_VPI; i++) {
	                if(tneta1570_dev->rx_vpi_vci[i] & OWN) {
				x = tneta1570_dev->rx_vpi_vci[i];
				sorted[(0x7f & (x >> 24))-8]=1;
		        } 
		}
                for(i=0; i< 0x78; i++) {
                        if(sorted[i] == 0)
		                break;
		}
                if(i == 0x78) return 0;
                tneta1570_dev->rx_vpi_vci[vcc->vpi]=OWN|((i+8)<<24)|(MAX_VCI);
	        return 256*i + vcc->vci;
	}
}



/*
 * interrupt driven dequeue for rx buffer
 * -> for each and every post in the completion ring do
 *    increment completion ring ptr (modulo RXCMPLR_SIZE_IRQ)
 *    get skb address from buffer address
 *    get free-buffer ring address from tneta1570_vcc
 *    prepare skb with header information
 *    vcc->push skb
 *    alloc (ATOMIC) new skb for this vci, put in free-buffer ring
 *    increment free-buffer ring index (modulo FBR_SIZE)
 */
static void dequeue_rx(struct atm_dev * dev)
{
        struct tneta1570_dev * tneta1570_dev;

        EVENT(">dequeue_rx\n",0,0);
        NULLCHECK(dev);
	tneta1570_dev = TNETA1570_DEV(dev);
	NULLCHECK(tneta1570_dev);

        /* find here completion ring entries */
        DPRINTK(">completion ring post: %08x, %d\n",
 	        (unsigned int)&RX_CMPL_R_IRQ(tneta1570_dev).atm_header,
	        tneta1570_dev->rxcmpl_ring_idx_irq); 
        while(!(RX_CMPL_R_IRQ(tneta1570_dev).control & OWN)) { 
        	if((RX_CMPL_R_IRQ(tneta1570_dev).error & BUFFER_STARV)) {
 			printk(DEV_LABEL " rx starvation error \n");
                        cli();
                        RX_CMPL_R_IRQ(tneta1570_dev).control = OWN;
                	tneta1570_dev->rxcmpl_ring_idx_irq++;
                	tneta1570_dev->rxcmpl_ring_idx_irq %= RXCMPLR_SZ_IRQ;
			sti();
			continue;
		}
		cli();
                if((RX_CMPL_R_IRQ(tneta1570_dev).control & 0xff) == 0) {
	                dequeue_OAM(dev);
                } else { 
		        if((RX_CMPL_R_IRQ(tneta1570_dev).error & AAL5_IND)) {
	                       if((RX_CMPL_R_IRQ(tneta1570_dev).buf_ptr & (1<<31)))
	                        	dequeue_AAL5(dev);
                        } else {
	                        dequeue_AAL0(dev);

			}
		}
                RX_CMPL_R_IRQ(tneta1570_dev).control = OWN;
                tneta1570_dev->rxcmpl_ring_idx_irq++;
                tneta1570_dev->rxcmpl_ring_idx_irq %= RXCMPLR_SZ_IRQ;
		sti();
	}
        return ;
}



static void dequeue_OAM(struct atm_dev *dev)  /* dequeue AAL0 cells */
{
        struct tneta1570_dev * tneta1570_dev;
        struct sk_buff *skb, *new_skb;
        unsigned int *p, *fbr;

        EVENT(">dequeue_rx_OAM\n",0,0);
        NULLCHECK(dev);
	tneta1570_dev = TNETA1570_DEV(dev);
	NULLCHECK(tneta1570_dev);
      


        p = (unsigned int *)bus_to_virt(RX_CMPL_R_IRQ(tneta1570_dev).buf_ptr << 2);
        skb = (struct sk_buff *)phys_to_virt(*(p-1));  /* get skb */
 	      
        /* get new buffer before dequeueing old
         */
        new_skb = alloc_skb(MAX_AAL0_PDU+RX_HDR, GFP_ATOMIC);
        if(new_skb == NULL) {
        /* add just received buffer to free list, drop pdu :-( */
                fbr = bus_to_virt((unsigned long)tneta1570_dev->free_buf_ptr[0].buf_ptr);
                fbr[tneta1570_dev->oam_fbr_idx] = OWN | (virt_to_bus(p) >> 2);
                tneta1570_dev->oam_fbr_idx++;
                tneta1570_dev->oam_fbr_idx %= AAL0_BUFS;
                return;
	}
	   
        /* add newly allocated buffer to free list */
        fbr = bus_to_virt((unsigned long)tneta1570_dev->free_buf_ptr[0].buf_ptr);
        fbr[tneta1570_dev->oam_fbr_idx] = OWN | (virt_to_bus(new_skb->data+4) >> 2);
	*(struct sk_buff **)(new_skb->data) = new_skb;
        tneta1570_dev->oam_fbr_idx++;
        tneta1570_dev->oam_fbr_idx %= AAL0_BUFS;
	   
        /* push received buffer to protocol */
		   
	skb->data[16] = skb->data[12];
	skb->data[17] = skb->data[13];
	skb->data[18] = skb->data[14];
	skb->data[19] = skb->data[15];
	skb->data += 4*4;  /* skip header */
	skb->len  = 52;
        /* should check for multiple buffers @@@@ */
	skb->tail = skb->data + skb->len;

	        if (/*vcc->push*/1) {
	        dev_kfree_skb(skb, GFP_ATOMIC);
/*              printk(DEV_LABEL " Don't know how to call push !\n"); */
/*	        vcc->push(vcc,skb); */
	} else {
	        dev_kfree_skb(skb, GFP_ATOMIC);
	        printk(DEV_LABEL "(itf %d) No push in protocol.\n", 
		       dev->number); 
	}
}


static void dequeue_AAL0(struct atm_dev *dev)  /* dequeue AAL0 cells */
{
        struct tneta1570_dev * tneta1570_dev;
        struct tneta1570_vcc * tneta1570_vcc;
   	struct atm_vcc *vcc;
	struct sk_buff *skb, *new_skb;
        unsigned int *p, v, *fbr;
   
        EVENT(">dequeue_rx_AAL0\n",0,0);
        NULLCHECK(dev);
	tneta1570_dev = TNETA1570_DEV(dev);
	NULLCHECK(tneta1570_dev);
   
        p = (unsigned int *)bus_to_virt(RX_CMPL_R_IRQ(tneta1570_dev).buf_ptr << 2);
        skb = (struct sk_buff *)phys_to_virt(*(p-1));  /* get skb */
	      
        vcc = skb->atm.vcc;
        NULLCHECK(vcc);
        tneta1570_vcc = TNETA1570_VCC(vcc);
        NULLCHECK(tneta1570_vcc);   
        /* get new buffer before dequeueing old
         * if error in peek drop pdu
         */
        new_skb = vcc->peek(vcc, MAX_AAL0_PDU+RX_HDR, NULL);
        if(new_skb == NULL) {
        /* add just received buffer to free list, drop pdu :-( */
                v = RX_CMPL_R_IRQ(tneta1570_dev).control;
                fbr = bus_to_virt((unsigned long)tneta1570_dev->free_buf_ptr[0xff & v].buf_ptr);
                fbr[tneta1570_vcc->fbr_idx] = OWN | (virt_to_bus(p) >> 2);
                tneta1570_vcc->fbr_idx++;
                tneta1570_vcc->fbr_idx %= AAL0_BUFS;
                return;
	}
	   
        /* add newly allocated buffer to free list */
                       
	v = RX_CMPL_R_IRQ(tneta1570_dev).control;
        fbr = bus_to_virt((unsigned long)tneta1570_dev->free_buf_ptr[0xff & v].buf_ptr);
        fbr[tneta1570_vcc->fbr_idx] = OWN | (virt_to_bus(new_skb->data+4) >> 2);
	*(struct sk_buff **)(new_skb->data) = new_skb;
        new_skb->atm.vcc = vcc;      /* link vcc info */
        tneta1570_vcc->fbr_idx++;
        tneta1570_vcc->fbr_idx %= AAL0_BUFS;
	   
        /* push received buffer to protocol */
		   
	skb->data[16] = skb->data[12];
	skb->data[17] = skb->data[13];
	skb->data[18] = skb->data[14];
	skb->data[19] = skb->data[15];
	skb->data += 4*4;  /* skip header */
	skb->len  = 52;
        /* should check for multiple buffers @@@@ */
	skb->tail = skb->data + skb->len;
        skb->free = 1;   /* set free flag, is there need for it? */
		   
        if (vcc->push) {
                vcc->push(vcc,skb);
	} else {
	        printk(DEV_LABEL "(itf %d) No push in protocol.\n", 
		       dev->number); 
	}
        /* update statistics */
        vcc->stats->rx++;
rx_dequeued++;
}
  
   
   
static void dequeue_AAL5(struct atm_dev *dev)   /* dequeue AAL5 PDU */
{
        struct tneta1570_dev *tneta1570_dev;
        struct tneta1570_vcc *tneta1570_vcc;
        struct atm_vcc *vcc;
	struct sk_buff *skb, *new_skb;
        unsigned int *p, v, *fbr;
   
        EVENT(">dequeue_rx_AAL5\n",0,0);
        NULLCHECK(dev);
	tneta1570_dev = TNETA1570_DEV(dev);
	NULLCHECK(tneta1570_dev);
        p = (unsigned int *)bus_to_virt(RX_CMPL_R_IRQ(tneta1570_dev).buf_ptr << 2);
        skb = (struct sk_buff *)phys_to_virt(*(p-1));  /* get skb */

        DPRINTK(" p %08x, skb %08x, head %08x, err %08x, sop %08x, trailer %08x, idx %08x",
	         p, skb, 
	         RX_CMPL_R_IRQ(tneta1570_dev).atm_header,
	         RX_CMPL_R_IRQ(tneta1570_dev).error, 
	         RX_CMPL_R_IRQ(tneta1570_dev).buf_ptr,
	         RX_CMPL_R_IRQ(tneta1570_dev).trailer,
	         RX_CMPL_R_IRQ(tneta1570_dev).control);
   
        vcc = skb->atm.vcc;
        DPRINTK(" vcc %08x", vcc);
   
        NULLCHECK(vcc);
        tneta1570_vcc = TNETA1570_VCC(vcc);
        NULLCHECK(tneta1570_vcc);   
        /* get new buffer before dequeueing old
         * if error in peek drop pdu
         */
        new_skb=NULL; 
        
        if((RX_CMPL_R_IRQ(tneta1570_dev).error & CRC_ERROR)) {
       		DPRINTK(DEV_LABEL " rx crc error.\n\n");
	} 
	if((RX_CMPL_R_IRQ(tneta1570_dev).error & RX_TIMEOUT)) {
       		DPRINTK(DEV_LABEL " rx time out.\n");
       	} 
       	if((RX_CMPL_R_IRQ(tneta1570_dev).error & PAKET_OVERFLOW)) {
       		DPRINTK(DEV_LABEL " rx packet overflow.\n");
       	} 
       	if(!(RX_CMPL_R_IRQ(tneta1570_dev).error & (CRC_ERROR | RX_TIMEOUT | PAKET_OVERFLOW)))
                new_skb = vcc->peek(vcc, MAX_AAL5_PDU+RX_HDR, NULL);
	if(new_skb == NULL) { 
        /* add just received buffer to free list :-( */
                v = RX_CMPL_R_IRQ(tneta1570_dev).control;
	        fbr = bus_to_virt((unsigned long)tneta1570_dev->free_buf_ptr[(0xffff & v)].buf_ptr);
	        fbr[tneta1570_vcc->fbr_idx] = OWN | (virt_to_bus(p) >> 2);
	        tneta1570_vcc->fbr_idx++;
	        tneta1570_vcc->fbr_idx %= AAL5_BUFS;
	        return;
	}
        /* add newly allocated buffer to free list */
                
	v = RX_CMPL_R_IRQ(tneta1570_dev).control;
        fbr = bus_to_virt((unsigned long)tneta1570_dev->free_buf_ptr[(v & 0xffff)].buf_ptr);
        v=virt_to_bus(new_skb->data+4);
        fbr[tneta1570_vcc->fbr_idx] = OWN | (v >> 2);
	*(struct sk_buff **)(new_skb->data) = new_skb;
        new_skb->atm.vcc = vcc;   /* link context info */
        tneta1570_vcc->fbr_idx++;
        tneta1570_vcc->fbr_idx %= AAL5_BUFS;
	   
        /* push received buffer to protocol */
	skb->data += 5*4;  /* skip header */
	skb->len  = RX_CMPL_R_IRQ(tneta1570_dev).trailer & 0xffff;
        /* should check for multiple buffers @@@@ */
	skb->tail = skb->data + skb->len;
        skb->free = 1;   /* set free flag, is there need for it? */
   
        if (vcc->push) {
                vcc->push(vcc,skb);
	} else {
	        printk(DEV_LABEL "(itf %d) No push in protocol.\n", 
	               dev->number); 
	}
        /* update statistics */
        vcc->stats->rx++;
rx_dequeued++;
}

   
/*
 * --- bring rx up ---
 * init vpi/vci table - well, already done by tneta1570_init
 * kmalloc completion rings 
 * write ptrs to completion rings into sar regs
 * init completion rings to all OWN
 * write completion ring related data to device structure, set index to 0
 * init dma channels 0, 1, 2 for reception of OAM cells
 */
static int start_rx(struct atm_dev *dev)
{
        int i;
        unsigned long x;
        unsigned int *ptr;
        struct sk_buff *buf;
        struct tneta1570_dev *tneta1570_dev;

        EVENT(">start_rx\n",0,0);
        tneta1570_dev = TNETA1570_DEV(dev);
   	tneta1570_dev->lost = 0;
   
        /* init rx completion rings */
        tneta1570_dev->rxcmpl_ring =
                       kmalloc(2*RXCMPLR_SZ_IRQ*sizeof(struct tneta_rx_compl),
			       GFP_KERNEL);
        DPRINTK("RX_CMPL_R->%08x", tneta1570_dev->rxcmpl_ring);
        if(!tneta1570_dev->rxcmpl_ring) {
	        printk(DEV_LABEL "(itf %d) malloc on rx start failed.\n", dev->number);
	        return -ENOMEM;
	}
        /* align completion-ring with irq to its size */
        x = (unsigned long)(&tneta1570_dev->rxcmpl_ring[RXCMPLR_SZ_IRQ]);
        tneta1570_dev->rxcmplringptr_irq = (struct tneta_rx_compl *)
                     (x & (~(RXCMPLR_SZ_IRQ*sizeof(struct tneta_rx_compl) - 1)));
        /* the rest is for the completion ring w/o irq */   
        if((tneta1570_dev->rxcmplringptr_irq - RXCMPLR_SZ_NOI) 
	   > tneta1570_dev->rxcmpl_ring) {
                tneta1570_dev->rxcmplringptr_noi = 
	                tneta1570_dev->rxcmplringptr_irq - RXCMPLR_SZ_NOI;
        } else {
                tneta1570_dev->rxcmplringptr_noi = 
	                tneta1570_dev->rxcmplringptr_irq + RXCMPLR_SZ_IRQ;
        }
        EVENT(">init rx completion ring irq\n",0,0);
        for(i=0; i<RXCMPLR_SZ_IRQ; i++)
                tneta1570_dev->rxcmplringptr_irq[i].control = OWN;
        EVENT(">init tx completion ring noi\n",0,0);
        for(i=0; i<RXCMPLR_SZ_NOI; i++)
                tneta1570_dev->rxcmplringptr_noi[i].control = OWN;
   
        tneta1570_dev->rxcmpl_ring_idx_noi = 0;
        tneta1570_dev->rxcmpl_ring_idx_irq = 0;

        SAR_REG_SHORT(tneta1570_dev, TNETA_S_RXCOMPLSIZEI) = RXCMPLR_SZ_IRQ-1; 
        SAR_REG_SHORT(tneta1570_dev, TNETA_S_RXCOMPLSIZE)  = RXCMPLR_SZ_NOI-1; 
        SAR_REG_WORD(tneta1570_dev, TNETA_RXCOMPLNOI) = (unsigned int)virt_to_bus(tneta1570_dev->rxcmplringptr_noi);
        SAR_REG_WORD(tneta1570_dev, TNETA_RXCOMPLIRQ) = (unsigned int)virt_to_bus(tneta1570_dev->rxcmplringptr_irq);

        /* init dma 0,1,2 for oam */
   
        /* alloc memory for oam fbr & buffers */
        ptr = kmalloc(AAL0_BUFS*4, GFP_KERNEL);
        if(ptr == NULL) printk(DEV_LABEL "PANIC on start rx \n");
        for(i=0; i<AAL0_BUFS; i++) {
	        buf = alloc_skb(MAX_AAL0_PDU+RX_HDR, GFP_KERNEL);
                if(buf == NULL) printk(DEV_LABEL "PANIC on start rx \n");
                *(struct sk_buff **)(buf->data) = buf;  /* link */
                ptr[i] = OWN | ((unsigned int)virt_to_bus(buf->data + 4) >> 2);
	}

        tneta1570_dev->oam_fbr_idx = 0;
        /* oam cells use fbr 0 */
        tneta1570_dev->free_buf_ptr[0].buf_ptr = (unsigned int)virt_to_bus(ptr);
        tneta1570_dev->free_buf_ptr[0].buf_size = FBR_AAL0_32;
   
        tneta1570_dev->rx_dma_state[0].control = RX_DMA_CONTROL_AAL0;
        tneta1570_dev->rx_dma_state[0].AAL0_cells = MAX_AAL0_CELLS;
        tneta1570_dev->rx_dma_state[0].rx_timeout = RX_TIME_OUT;
        tneta1570_dev->rx_dma_state[0].dma_on_idx = DMA_ON + 0; 
        tneta1570_dev->rx_dma_state[1].control = RX_DMA_CONTROL_AAL0;
        tneta1570_dev->rx_dma_state[1].AAL0_cells = MAX_AAL0_CELLS;
        tneta1570_dev->rx_dma_state[1].rx_timeout = RX_TIME_OUT;
        tneta1570_dev->rx_dma_state[1].dma_on_idx = DMA_ON + 0; 
        tneta1570_dev->rx_dma_state[2].control = RX_DMA_CONTROL_AAL0;
        tneta1570_dev->rx_dma_state[2].AAL0_cells = MAX_AAL0_CELLS;
        tneta1570_dev->rx_dma_state[2].rx_timeout = RX_TIME_OUT;
        tneta1570_dev->rx_dma_state[2].dma_on_idx = DMA_ON + 0; 

        SAR_REG_WORD(tneta1570_dev, TNETA_CONFIG) |= TNETA_R0_RX_ENABLE;

        return 0;
}

/* 
 * open vpi/vci for rx
 * alloc 16 words for free-buffer ring
 * alloc 16 skbs, size: if AAL5
 *                         IP-MTU + a little something (64K SDUs need multiple bufs)
 *                      else
 *                         48 + a little something
 * enter pointers to skbs in free-buffer ring
 * find unused free-buffer ring-pointer table entry
 * put pointer to free-buffer ring in ring-pointer table & tneta1570_vcc
 * check if vpi has already alloc'd a dma table range
 *       if not, alloc range in dma table for vpi (size is 256*8 words)
 * if AAL5 -> set dma state table to AAL5
 * else use counter terminated AAL0 (cell count 1)
 * prepare dma state table entry for this vpi/vci (may turn rx for this vci on)
 * if range < vci -> set range to vci (max) (this definitely turns it on)
 */
static int open_rx(struct atm_vcc *vcc)
{
        int i, dma_idx, fbr_idx;
        struct sk_buff *skb;
        unsigned int *ptr;
        struct tneta1570_dev *tneta1570_dev;
        struct tneta1570_vcc *tneta1570_vcc;

        EVENT(">open_rx\n",0,0);
   
        tneta1570_dev = TNETA1570_DEV(vcc->dev);
        tneta1570_vcc = TNETA1570_VCC(vcc);

   	tneta1570_vcc->rx_wait = NULL;

	tneta1570_vcc->dma_channel = alloc_dma(vcc);
        if(!tneta1570_vcc->dma_channel) return -1;

        dma_idx = tneta1570_vcc->dma_channel;
   
        for(fbr_idx = 0; fbr_idx<MAX_FBR_ENTRIES; fbr_idx++)
	        if(tneta1570_dev->free_buf_ptr[fbr_idx].buf_ptr == 0)
	                break;
        if(fbr_idx == MAX_FBR_ENTRIES) return -1;
	   
        if(vcc->aal == ATM_AAL0) {
                /* alloc memory for fbr & buffers */
                ptr = kmalloc(AAL0_BUFS*4,
		              GFP_KERNEL);
                if(ptr == NULL) printk(DEV_LABEL "PANIC on open rx \n");
                for(i=0; i<AAL0_BUFS; i++) {
	                skb = vcc->peek(vcc, MAX_AAL0_PDU+RX_HDR, NULL);
                        if(skb == NULL) printk(DEV_LABEL "PANIC on open rx \n");
		        *(struct sk_buff **)(skb->data) = skb; /* link */
		        skb->atm.vcc = vcc; /* link vcc info */
                        ptr[i] = OWN | ((unsigned int)virt_to_bus(skb->data+4) >> 2);
	        }

   	        tneta1570_vcc->fbr_idx = 0;
	        tneta1570_dev->free_buf_ptr[fbr_idx].buf_ptr = virt_to_bus(ptr);
                tneta1570_dev->free_buf_ptr[fbr_idx].buf_size = FBR_AAL0_32;

                tneta1570_dev->rx_dma_state[dma_idx].control = RX_DMA_CONTROL_AAL0;
                tneta1570_dev->rx_dma_state[dma_idx].AAL0_cells = MAX_AAL0_CELLS;
                tneta1570_dev->rx_dma_state[dma_idx].rx_timeout = RX_TIME_OUT;
                tneta1570_dev->rx_dma_state[dma_idx].dma_on_idx = DMA_ON + fbr_idx;
	   
	} else if(vcc->aal == ATM_AAL5) {
                /* alloc memory for fbr & buffers */
                ptr = kmalloc(AAL5_BUFS*4,
		              GFP_KERNEL);
                if(ptr == NULL) printk(DEV_LABEL "PANIC on open rx \n");
                for(i=0; i<AAL5_BUFS; i++) {
	                skb = vcc->peek(vcc, MAX_AAL5_PDU+RX_HDR, NULL);
                        if(skb == NULL) printk(DEV_LABEL "PANIC on open rx \n");
		        *(struct sk_buff **)(skb->data) = skb; /* link */
		        skb->atm.vcc = vcc;  /* link vcc info */
                        ptr[i] = OWN | ((unsigned int)virt_to_bus(skb->data+4) >> 2);
	        }

	        tneta1570_vcc->fbr_idx = 0;
	        tneta1570_dev->free_buf_ptr[fbr_idx].buf_ptr = virt_to_bus(ptr);
                tneta1570_dev->free_buf_ptr[fbr_idx].buf_size = FBR_AAL5_16;
   
                tneta1570_dev->rx_dma_state[dma_idx].control = RX_DMA_CONTROL_AAL5;
                tneta1570_dev->rx_dma_state[dma_idx].rx_timeout = RX_TIME_OUT;
                tneta1570_dev->rx_dma_state[dma_idx].dma_on_idx = DMA_ON + fbr_idx;
	   
	} else return -1;
        return 0;
}

/* 
 * close vpi/vci for rx
 * 
 * get free-buffer ring table address
 * clear dma state table entry (rx for vci off)
 * get free-buffer ring address
 * wait for rx to drain
 * free all remaining skbs (the ones with own enabled, e.g. all)
 * free free-buffer ring
 * set free-buffer-ring ptr to 0 to indicate 'freeness'
 * check if vpi has another vci enabled
 * -> yes:    if other_vci > this_vci -> do nothing
 *            else                    -> set max vci to highest other vci
 * -> no: set vci in vpi/vci table to 0
 */
static void close_rx(struct atm_vcc *vcc)
{
        int i, fbr_idx, dma_idx, x;
        struct sk_buff *skb;
        unsigned int *buf, *ptr;
        struct tneta1570_dev *tneta1570_dev;
        struct tneta1570_vcc *tneta1570_vcc;

        EVENT(">close_rx\n",0,0);
   
        tneta1570_dev = TNETA1570_DEV(vcc->dev);
        tneta1570_vcc = TNETA1570_VCC(vcc);

        dma_idx = tneta1570_vcc->dma_channel;
        fbr_idx = tneta1570_dev->rx_dma_state[dma_idx].dma_on_idx & 0xff;
        /* wait for EOP */
        while(tneta1570_dev->rx_dma_state[dma_idx].control & OWN);
        tneta1570_dev->rx_dma_state[dma_idx].dma_on_idx = 0; /* off */
        /* mark as empty */
        ptr = bus_to_virt((unsigned long)tneta1570_dev->free_buf_ptr[fbr_idx].buf_ptr);
        tneta1570_dev->free_buf_ptr[fbr_idx].buf_ptr = 0;  /* mark fbr free */
        x = tneta1570_dev->free_buf_ptr[fbr_idx].buf_size;
        x = x >> 10;  /* ring size */
        x &= 0x3f; 
        x++;
        x = 16 * x;
        for(i=0; i<x; i++) {
	        if(ptr[i] && OWN) {
	                buf = (unsigned int *)bus_to_virt(ptr[i] << 2);
	                skb = (struct sk_buff *)phys_to_virt(*(buf-1));
	                skb->free = 1; /* ???? */
                        kfree_skb(skb, FREE_READ);
		}
        }
        kfree(ptr);

        /* if last vci on this vpi is closed, close vpi */

}


/*----------------------------------- TX ------------------------------------*/

/* 
 * -- bring tx up --
 * init scheduler table to all 0s
 * kmalloc tx completion rings
 * write ptrs to completion rings into sar regs
 * init completion rings to all 0x80000000s
 * write completion ring related data to device structure, set index to 0
 * enable tx in sar
 */
static int start_tx(struct atm_dev *dev)
{
        int i;
        unsigned int  *seg_ring, *seg_ring_mptr;
        unsigned long x;
        struct tneta1570_dev *tneta1570_dev;

        EVENT(">start_tx\n",0,0);
   
        tneta1570_dev = TNETA1570_DEV(dev);
     
        /* init tx completion rings */
        tneta1570_dev->txcmpl_ring=kmalloc(2*TXCMPLR_SZ_IRQ*4, GFP_KERNEL);
        if(!tneta1570_dev->txcmpl_ring) {
	        printk(DEV_LABEL "(itf %d) malloc on tx start failed.\n", dev->number);
	        return -ENOMEM;
	}
        DPRINTK("TX_CMPL_R->%08x", tneta1570_dev->txcmpl_ring);
        /* align completion-ring with irq to its size */
        x = (unsigned long)(&tneta1570_dev->txcmpl_ring[TXCMPLR_SZ_IRQ]);
        tneta1570_dev->txcmplringptr_irq = (unsigned int *)(x & (~(TXCMPLR_SZ_IRQ*4 - 1)));
        /* the rest is for the completion ring w/o irq */   
        if((tneta1570_dev->txcmplringptr_irq - TXCMPLR_SZ_NOI) > tneta1570_dev->txcmpl_ring) {
                tneta1570_dev->txcmplringptr_noi = tneta1570_dev->txcmplringptr_irq - TXCMPLR_SZ_NOI;
        } else {
                tneta1570_dev->txcmplringptr_noi = tneta1570_dev->txcmplringptr_irq + TXCMPLR_SZ_IRQ;
        }
        EVENT(">init tx completion ring irq\n",0,0);
        for(i=0; i<TXCMPLR_SZ_IRQ; i++)
                tneta1570_dev->txcmplringptr_irq[i] = OWN;
        EVENT(">init tx completion ring noi\n",0,0);
        for(i=0; i<TXCMPLR_SZ_NOI; i++)
                tneta1570_dev->txcmplringptr_noi[i] = OWN;
   
        tneta1570_dev->txcmpl_ring_idx_noi = 0;
        tneta1570_dev->txcmpl_ring_idx_irq = 0;

        /* dma state reread bug fix - allocate dma 1 */
        seg_ring_mptr=kmalloc(2*TX_SEG_RING_SIZE*4, GFP_KERNEL);
        if(!seg_ring_mptr) {
	        printk(DEV_LABEL "(itf %d) malloc on tx open failed.\n",
		       dev->number);
	        return -ENOMEM;
	}
        x = (unsigned long)(seg_ring_mptr + TX_SEG_RING_SIZE);
        seg_ring = (unsigned int *)(x & ~(TX_SEG_RING_SIZE * 4 - 1));

        for(i=0; i<TX_SEG_RING_SIZE; i++) 
                seg_ring[i]=0;            /* turn off - high cost :-( */

        write_sched_tab(tneta1570_dev, 0, 1);
        tneta1570_dev->tx_dma_state[1].dma_state_flag = (OWN | SEG_PTR(seg_ring));

        /* init registers */ 
        SAR_REG_SHORT(tneta1570_dev, TNETA_S_TXCOMPLSIZEI) = TXCMPLR_SZ_IRQ - 1; 
        SAR_REG_SHORT(tneta1570_dev, TNETA_S_TXCOMPLSIZE)  = TXCMPLR_SZ_NOI - 1; 
        SAR_REG_SHORT(tneta1570_dev, TNETA_S_TXSEGSIZE)  = TX_SEG_RING_SIZE - 1; 
        SAR_REG_SHORT(tneta1570_dev, TNETA_S_SCHEDSIZE) = 2 + SLOW_DOWN_FACTOR;
   
        SAR_REG_WORD(tneta1570_dev, TNETA_TXCOMPLNOI) = (unsigned int)virt_to_bus((void*)tneta1570_dev->txcmplringptr_noi);
        SAR_REG_WORD(tneta1570_dev, TNETA_TXCOMPLIRQ) = (unsigned int)virt_to_bus((void*)tneta1570_dev->txcmplringptr_irq);

        SAR_REG_WORD(tneta1570_dev, TNETA_CONFIG) |= TNETA_R0_TX_ENABLE;        

        return 0;
}
 
/*
 * -- open vpi/vci --
 * find free entry in scheduler table
 * init tx_dma_state table for this vpi/vci
 * kmalloc tx_seg_ring, init with all 0s
 * add entry to scheduler table
 */
static int open_tx(struct atm_vcc *vcc)
{
        int scheduler_idx, i;
        unsigned long x;
        struct tneta1570_dev *tneta1570_dev;
        struct tneta1570_vcc *tneta1570_vcc;

        EVENT(">open_tx\n",0,0);
   
        tneta1570_dev = TNETA1570_DEV(vcc->dev);
        tneta1570_vcc = TNETA1570_VCC(vcc);

   	tneta1570_vcc->tx_wait = NULL;

        if (vcc->qos.txtp.traffic_class == ATM_NONE) return 0;

        tneta1570_vcc->seg_ring_mptr=kmalloc(2*TX_SEG_RING_SIZE*4, GFP_KERNEL);
        if(!tneta1570_vcc->seg_ring_mptr) {
	        printk(DEV_LABEL "(itf %d) malloc on tx open failed.\n",
		       vcc->dev->number);
	        return -ENOMEM;
	}
        /* align seg-ring to its size */
        x = (unsigned long)(tneta1570_vcc->seg_ring_mptr + TX_SEG_RING_SIZE);
        tneta1570_vcc->seg_ring = (unsigned int *)(x & ~(TX_SEG_RING_SIZE * 4 - 1));

        EVENT("> seg ring is at phy %x\n", (unsigned int)tneta1570_vcc->seg_ring,0);
        for(i=0; i<TX_SEG_RING_SIZE; i++) 
                tneta1570_vcc->seg_ring[i]=0;
        tneta1570_vcc->seg_ring_idx = 0;

        /* find free scheduler table entry */
        scheduler_idx=0;
        while(read_sched_tab(tneta1570_dev, scheduler_idx) != 0) {
                scheduler_idx++;
                if(scheduler_idx >= MAX_SCHED_ENTRIES) {
		        printk(DEV_LABEL "(itf %d) tx scheduler full on open.\n", 
			       vcc->dev->number);
                        return -ENOMEM;
	        }
	}

        if((scheduler_idx < SLOW_DOWN_FACTOR) && 
	   (scheduler_idx + SLOW_DOWN_FACTOR < MAX_SCHED_ENTRIES))
                SAR_REG_SHORT(tneta1570_dev, TNETA_S_SCHEDSIZE) =
                                             scheduler_idx + SLOW_DOWN_FACTOR;
        else SAR_REG_SHORT(tneta1570_dev, TNETA_S_SCHEDSIZE) =scheduler_idx;

        tneta1570_vcc->scheduler_idx = scheduler_idx;
        write_sched_tab(tneta1570_dev, scheduler_idx, scheduler_idx+1); /* 1 in entry 0 */
   
        tneta1570_dev->tx_dma_state[scheduler_idx+1].dma_state_flag = 
             (OWN | SEG_PTR(tneta1570_vcc->seg_ring));

        tneta1570_vcc->txing = 0;  /* init txing flag */

        return 0;
}

/* 
 * -- close vpi/vci --
 * remove scheduler table entry
 * free tx_seg_ring
 */
static void close_tx(struct atm_vcc *vcc)
{
        struct tneta1570_dev *tneta1570_dev;
        struct tneta1570_vcc *tneta1570_vcc;

        EVENT(">close_tx\n",0,0);
   
        tneta1570_dev = TNETA1570_DEV(vcc->dev);
        tneta1570_vcc = TNETA1570_VCC(vcc);

        current->timeout=jiffies+TNETA1570_TIMEOUT;
        while(tneta1570_vcc->txing) {
	     interruptible_sleep_on(&tneta1570_vcc->tx_wait);  /* wait for tx to drain */
	if(!current->timeout){
		printk(DEV_LABEL " timed out while closing (%d)\n",
			tneta1570_vcc->txing);
		break;
		}
	}
  
        /* remove ptr to segmentation ring */
        
        tneta1570_dev->tx_dma_state[tneta1570_vcc->scheduler_idx+1].dma_state_flag = 0;
        write_sched_tab(tneta1570_dev, tneta1570_vcc->scheduler_idx, 0); 
   
        kfree(tneta1570_vcc->seg_ring_mptr);    
}

/* 
 * -- send skb --
 * prepare buffer with header data 
 * set entry in segmentation ring for vpi/vci and increment index
 * smile ;-)
 */
static int do_tx(struct sk_buff * skb)
{
        struct atm_vcc *vcc=0;
        struct tneta1570_dev *tneta1570_dev=0;
        struct tneta1570_vcc *tneta1570_vcc=0;
        unsigned int seg_ring_entry, i;
        unsigned int *buffer;
   
        NULLCHECK(skb);
        EVENT(">do_tx: skb=0x%lx, %d bytes\n",(unsigned long) skb,skb->len);
        vcc = skb->atm.vcc;
        NULLCHECK(vcc);
        tneta1570_dev = TNETA1570_DEV(vcc->dev);
        NULLCHECK(tneta1570_dev);
        tneta1570_vcc = TNETA1570_VCC(vcc);
        if(tneta1570_vcc->txing>=TX_SEG_RING_SIZE-1)
        	interruptible_sleep_on(&tneta1570_vcc->tx_wait); 
        if ((unsigned long) skb->data & TNETA_ALIGN) {
                printk(DEV_LABEL "(itf %d): VCI %d has mis-aligned TX data\n",
                    vcc->dev->number,vcc->vci);
		dev_kfree_skb(skb,FREE_WRITE);
	        return -EINVAL;
	}
        /* ping seems to be a problem here, its not using my alloc function */
        /* prepare buffer descriptor */
        if((skb->data - skb->head) < 24) {
                DPRINTK(DEV_LABEL "(itf %d): skbuff push impossible (%d)\n",
                    vcc->dev->number, skb->data - skb->head);
                /* copy the data if push not possible */
                if(!(buffer=kmalloc(skb->len + 24, GFP_ATOMIC))) {
                        printk(DEV_LABEL "(itf %d): malloc on send failed.\n",
                               vcc->dev->number);
			dev_kfree_skb(skb,FREE_WRITE);
                        return -ENOMEM;
	        }
                EVENT(">tx mem alloc'd at %08x\n", (unsigned int)buffer,0);
                NULLCHECK(skb->data);
                if(vcc->aal == ATM_AAL0) {
                        buffer[3] = 0;
                        buffer[4] = ((unsigned int *)skb->data)[0];  /* copy atm header */
                        buffer[5] = 0;
                        for(i=0; i<skb->len/4+1; i++)
		                buffer[6+i] = ((unsigned int *)skb->data)[1+i];
	                buffer[2] = AAL0_PDU_INT | (0xffff & skb->len); /* offset 0 */
	        } else {
                        buffer[3] = 0;
                        buffer[4] = (vcc->vpi << 20) | (vcc->vci << 4);  /* prepare atm header */
                        buffer[5] = 0;
                        for(i=0; i<skb->len/4+1; i++)
		                buffer[6+i] = ((unsigned int *)skb->data)[i];
	                buffer[2] = AAL5_PDU_INT | (0xffff & skb->len);
                }
                buffer[1] = (unsigned int)virt_to_bus(skb);
                buffer[0] = 0; /* push size */
                EVENT(">data copied\n",0,0);
	} else { /* push skb and put header in front of sdu */
                if(vcc->aal == ATM_AAL0) { /* sdu contains header */
                        buffer = (unsigned int *)skb_push(skb, 20);  /* make room for 1+4 words header */
                        buffer[3] = 0;
                        buffer[4] = buffer[5];  /* copy atm header */
                        buffer[5] = 0;
	                buffer[2] = AAL0_PDU_INT | (0xffff & (skb->len-20)); /* offset 0 */
                        buffer[0] = 20; /* push size */
	        } else {
                        buffer = (unsigned int *)skb_push(skb, 24);  /* make room for 1+4 words header */
                        buffer[3] = 0;
                        buffer[4] = (vcc->vpi << 20) | (vcc->vci << 4);  /* prepare atm header */
                        buffer[5] = 0;
	                buffer[2] = AAL5_PDU_INT | (0xffff & (skb->len-24));
                        buffer[0] = 24; /* push size */
                }
                buffer[1] = (unsigned int)virt_to_bus(skb);  /* store skb ptr for dequeue */
	}

        seg_ring_entry = ((unsigned int)virt_to_bus(&buffer[2]) >> 2) | OWN;
        tneta1570_vcc->seg_ring[tneta1570_vcc->seg_ring_idx] = seg_ring_entry;
        DPRINTK(">zippered up");
        DPRINTK(">sched %d,"
 		">dma %08x,"
 		">sridx %d >seg_r %08x (%08x),"
 		">buffer %08x, (%08x),"
		"atm header %08x\n",
 	        read_sched_tab(tneta1570_dev, tneta1570_vcc->scheduler_idx),
 		tneta1570_dev->tx_dma_state[tneta1570_vcc->scheduler_idx+1].dma_state_flag,
 	        tneta1570_vcc->seg_ring_idx,
 	        tneta1570_vcc->seg_ring, 
 		tneta1570_vcc->seg_ring[tneta1570_vcc->scheduler_idx],
 	        &buffer[2], buffer[2], buffer[4]);
	 
        /* index to seg.ring entry for next SDU */
        cli(); /*@@@ restore flags ??*/
        tneta1570_vcc->seg_ring_idx++;
        tneta1570_vcc->seg_ring_idx %= TX_SEG_RING_SIZE;
        tneta1570_vcc->txing++;
        sti(); 
        return 0;
}

/*
 * -- dequeue tx buffer on tx complete interrupt --
 * check completion ring
 * while valid entry 
 *     dequeue skb (needs a bit of backtracking)
 *     increment completion ring index
 */
static void dequeue_tx(struct atm_dev * dev)
{
	struct tneta1570_dev *tneta1570_dev;
	struct atm_vcc *vcc;
	struct sk_buff *skb;
        unsigned int * p, v;

        EVENT(">dequeue_tx\n",0,0);
   
        NULLCHECK(dev);
	tneta1570_dev = TNETA1570_DEV(dev);
	NULLCHECK(tneta1570_dev);
        cli();
        /* find here completion ring entries */
        EVENT(">completion ring post: %08x, %d\n", TX_CMPL_R_IRQ(tneta1570_dev),
		tneta1570_dev->txcmpl_ring_idx_irq); 
        while(!(TX_CMPL_R_IRQ(tneta1570_dev) & OWN)) { 
	        p = (unsigned int *)bus_to_virt(TX_CMPL_R_IRQ(tneta1570_dev) << 2);
	        TX_CMPL_R_IRQ(tneta1570_dev) = OWN;
	   	tneta1570_dev->txcmpl_ring_idx_irq++;
	        tneta1570_dev->txcmpl_ring_idx_irq %= TXCMPLR_SZ_IRQ;
	        skb = (struct sk_buff *)bus_to_virt(*(p-1));  /* get skb */
                v = *(p-2);    /* get skb push size */

                if(v==0) {     /* free copy area */
		        kfree(p-2);
                } else {       /* correct skb */
                        skb_pull(skb, v);
		}
	      
	        vcc = skb->atm.vcc;
                NULLCHECK(vcc);
	        if (vcc->pop) vcc->pop(vcc,skb);
		else dev_kfree_skb(skb,FREE_WRITE);

	        vcc->stats->tx++;
	        TNETA1570_VCC(vcc)->txing--;
	        wake_up_interruptible(&(TNETA1570_VCC(vcc)->tx_wait));
dma_complete++;
	};
sti();
}



/*--------------------------------- common ----------------------------------*/


static void foo(void)
{
printk("tx_complete=%d,dma_complete=%d,queued=%d,requeued=%d,sub=%d,\n"
  "backlogged=%d,rx_enqueued=%d,rx_dequeued=%d,putting=%d,pushed=%d\n",
  tx_complete,dma_complete,queued,requeued,submitted,backlogged,
  rx_enqueued,rx_dequeued,putting,pushed);
printk("loss: %d\n",TNETA1570_DEV(tneta1570_boards)->lost);
}


static void tneta1570_int(int irq,void *dev_id,struct pt_regs *regs)
{
	struct atm_dev *dev;
	struct tneta1570_dev *tneta1570_dev;
	unsigned int reason;

	EVENT(">tneta_int\n",0,0);
	dev = dev_id;
	tneta1570_dev = TNETA1570_DEV(dev);
	while ( (reason = 0x3ff & SAR_REG_WORD(tneta1570_dev, TNETA_STATUS)) ) {
		DPRINTK(DEV_LABEL ": int 0x%08x\n",reason);
		if (reason & TNETA_R1_RX_IRR) {
			DPRINTK(DEV_LABEL " Unknown PDU %d.%d\n",
			(SAR_REG_WORD(tneta1570_dev,TNETA_RXUNKNOWN) & 0x3f00) >> 16,
			SAR_REG_WORD(tneta1570_dev,TNETA_RXUNKNOWN) & 0x0ff);
		}	/* unknown cell received */
		if (reason & TNETA_R1_RX_FREEZE) {
			EVENT("INT: RX Freeze - RX ring overflow\n",0,0);
		}	/* ? */
		if (reason & TNETA_R1_TX_FREEZE) {
			EVENT("INT: TX Freeze - TX ring overflow\n",0,0);
		        dequeue_tx(dev);
		}	/* ? */
		if (reason & TNETA_R1_CP_RX) {
			EVENT("INT: RX Complete - RX buffer completed\n",0,0);
                        dequeue_rx(dev);
		}	/* ? */
	        if (reason & TNETA_R1_CP_TX) {
			EVENT("INT: TX Complete - TX Buffer completed\n",0,0);
		        dequeue_tx(dev);
			/* pop buffers which have been completed */
		}
	}
}

/* 
 * perform SAR software reset 
 */
static int reset_sar(struct tneta1570_dev * tneta1570_dev)
{
   int i;
   unsigned int pci_config[64];
   
   for(i=0; i<64; i++)
      if(pcibios_read_config_dword(tneta1570_dev->bus,
				   tneta1570_dev->dev_fn,
				   i*4,
				   &pci_config[i])
	    !=PCIBIOS_SUCCESSFUL) return -1;

   SAR_REG_WORD(tneta1570_dev, TNETA_RESET) = 0;  /* reset sar */

   for(i=0; i<64; i++)
      if(pcibios_write_config_dword(tneta1570_dev->bus,
				    tneta1570_dev->dev_fn,
				    i*4,
				    pci_config[i])
	    !=PCIBIOS_SUCCESSFUL) return -1;
   return 0;
   
}


inline void write_sched_tab(struct tneta1570_dev * dev, int i, unsigned int v)
{ 
        if(1 & i)
                dev->scheduler[i>>1] = (dev->scheduler[i>>1] & 0xffff) | (v << 16);
        else
                dev->scheduler[i>>1] = (dev->scheduler[i>>1] & 0xffff0000) | (v & 0xffff);
}

inline int read_sched_tab(struct tneta1570_dev * dev, int i)
{ 
        if(1 & i)
                return (dev->scheduler[i>>1] >> 16);
        else
                return (0xffff & dev->scheduler[i>>1]);
}

/*--------------------------------- entries ---------------------------------*/


static int tneta1570_init(struct atm_dev *dev)
{
	struct tneta1570_dev *tneta1570_dev;
	unsigned long base;
	unsigned int real_base;
	unsigned short command;
	unsigned char revision;
	int error,i,last;

	EVENT(">tneta1570_init\n",0,0);

        dev->ci_range.vpi_bits = NR_VPI_LD;
	dev->ci_range.vci_bits = NR_VCI_LD;
	tneta1570_dev = TNETA1570_DEV(dev);

        if ((error = pcibios_read_config_word(tneta1570_dev->bus,
			    tneta1570_dev->dev_fn, PCI_COMMAND,&command)) 
	    || (error = pcibios_read_config_dword(tneta1570_dev->bus,
	                    tneta1570_dev->dev_fn,PCI_BASE_ADDRESS_0,&real_base)) 
	    || (error = pcibios_read_config_byte(tneta1570_dev->bus,
			    tneta1570_dev->dev_fn, PCI_INTERRUPT_LINE,&tneta1570_dev->irq)) 
	    || (error = pcibios_read_config_byte(tneta1570_dev->bus,
			    tneta1570_dev->dev_fn, PCI_REVISION_ID,&revision))) {
 		printk(DEV_LABEL "(itf %d): init error %s\n",dev->number,
		    pcibios_strerror(error));
		return -EINVAL;
	}

        /* find mapping size of board */
        if(pcibios_write_config_dword(tneta1570_dev->bus,
				      tneta1570_dev->dev_fn,
				      PCI_BASE_ADDRESS_0,
				      0xffffffff)!=PCIBIOS_SUCCESSFUL)
        {
 		printk(DEV_LABEL "(itf %d): init error %s\n",dev->number,
		    pcibios_strerror(error));
		return -EINVAL;
        }

        if(pcibios_read_config_dword(tneta1570_dev->bus,
			             tneta1570_dev->dev_fn,
				     PCI_BASE_ADDRESS_0,
				     &(tneta1570_dev->pci_map_size))!=PCIBIOS_SUCCESSFUL)
        { 
 		printk(DEV_LABEL "(itf %d): init error %s\n",dev->number,
		    pcibios_strerror(error));
		return -EINVAL;
        }
        tneta1570_dev->pci_map_size=~tneta1570_dev->pci_map_size+1;
   
        if(pcibios_write_config_dword(tneta1570_dev->bus,
				      tneta1570_dev->dev_fn,
				      PCI_BASE_ADDRESS_0,
				      real_base)!=PCIBIOS_SUCCESSFUL)
        { 
 		printk(DEV_LABEL "(itf %d): init error %s\n",dev->number,
		    pcibios_strerror(error));
		return -EINVAL;
	}
   
   
        real_base &= MEM_VALID; /* strip flags */
	if ((error = pcibios_write_config_word(tneta1570_dev->bus,
			                       tneta1570_dev->dev_fn,
					       PCI_COMMAND,
					       PCI_COMMAND_MEMORY)))
        {
		printk(DEV_LABEL "(itf %d): can't enable memory (%s)\n",
		    dev->number,pcibios_strerror(error));
		return error;
	}
	printk(DEV_LABEL "(itf %d): rev.%d,irq=%d,",dev->number,
	    revision,tneta1570_dev->irq);
	if (!(base = (unsigned long) vremap((unsigned long)real_base+TNETA1570_MEM_OFFSET,tneta1570_dev->pci_map_size))) {
		printk(DEV_LABEL "(itf %d): can't set up page mapping\n",
		    dev->number);
		return error;
	}
	tneta1570_dev->base_diff = real_base-base;
 	tneta1570_dev->reg = (volatile unsigned int *) (base+TNETA_REG_BASE_OFFSET);
	tneta1570_dev->scheduler = (volatile unsigned int *) (base+TNETA_SCHED_TABLE);
	tneta1570_dev->ram = (volatile unsigned int *) (base+TNETA_SCHED_TABLE);
        tneta1570_dev->free_buf_ptr = (volatile struct tneta_rx_fbrptr *) (base+TNETA_FREE_BUFFER_POINTERS);
	tneta1570_dev->rx_vpi_vci = (volatile unsigned int *) (base+TNETA_RX_VPIVCI_DMA_POINTERS);
	tneta1570_dev->tx_dma_state = (volatile struct tneta_tx_dma_state_table *) (base+TNETA_TX_DMA_STATE_TABLE);
	tneta1570_dev->rx_dma_state = (volatile struct tneta_rx_dma_state_table *) (base+TNETA_RX_DMA_STATE_TABLE);
	tneta1570_dev->phy = (volatile unsigned int *) (base+TNETA_SUNI_OFFSET);

        tneta1570_dev->txcmpl_ring_idx_noi = 0;
        tneta1570_dev->txcmpl_ring_idx_irq = 0;
        tneta1570_dev->rxcmpl_ring_idx_noi = 0;
        tneta1570_dev->rxcmpl_ring_idx_irq = 0;

        /* test board memory, find amount of memory installed */ 
        last = (TNETA_SUNI_OFFSET)/4;  /* max up to phy ctrl */
	for (i = last-RAM_INCREMENT; i >= 0; i -= RAM_INCREMENT) {
	        if(i>RESERVED_UL || i<RESERVED_LL) {
	   	        tneta1570_dev->ram[i] = 0x55555555;
		        if (tneta1570_dev->ram[i] != 0x55555555) last = i;
		        else {
			        tneta1570_dev->ram[i] = 0xAAAAAAAA;
			        if (tneta1570_dev->ram[i] != 0xAAAAAAAA) last = i;
			        else tneta1570_dev->ram[i] = i;
		        }
	        }
	}
	for (i = 0; i < last; i += RAM_INCREMENT)
	        if(i>RESERVED_UL || i<RESERVED_LL)
   	                if (tneta1570_dev->ram[i] != i) break;

        tneta1570_dev->mem = i << 2; /* byte count */
        /* init control memory with all 0s (including regs) */
        last = i;
        /* bring sar up */
        for(i=0; i<last; i++)
	        if(i>RESERVED_UL || i<RESERVED_LL)
                        tneta1570_dev->ram[i] = 0; 

        (tneta1570_dev->reg[TNETA_STATUS] & 0x00000400)?
       		printk("mem=%dkB,mode=64 bit,",tneta1570_dev->mem >> 10):	  
        	printk("mem=%dkB,mode=32 bit,",tneta1570_dev->mem >> 10);

        /* reset SAR */
        reset_sar(tneta1570_dev);

        for(i=0; i<last; i++)
	        if(i>RESERVED_UL || i<RESERVED_LL)
                        tneta1570_dev->ram[i] = 0; 
       
	return 0;
}


static int tneta1570_start(struct atm_dev *dev)
{
	struct tneta1570_dev *tneta1570_dev;
	int error;
	unsigned char phy;

	EVENT(">tneta1570_start\n",0,0);
	tneta1570_dev = TNETA1570_DEV(dev);
	if (request_irq(tneta1570_dev->irq,&tneta1570_int,0,DEV_LABEL,dev)) {
		printk(DEV_LABEL "(itf %d): IRQ%d is already in use\n",
		    dev->number,tneta1570_dev->irq);
		return -EAGAIN;
	}
	/* @@@ should release IRQ on error */

        if ((error = pcibios_write_config_word(tneta1570_dev->bus,
					       tneta1570_dev->dev_fn,
	                                       PCI_COMMAND,
					       PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER)))
        {
		printk(DEV_LABEL "(itf %d): can't enable memory+master (%s)\n",
		    dev->number,pcibios_strerror(error));
		return error;
	}

        if((phy=tneta1570_phy_get(dev,0))==0x30) {
                 printk("pm5346,rev.%d \n",phy&0x0f); 	 	
                 suni_init(dev);	
        } else {          	
                 printk("utopia,rev.%d \n",phy&0x0f);
                 utopia_init(dev);  	
        }   

        SAR_REG_WORD(tneta1570_dev, TNETA_CONFIG) = TNETA_R0_STANDARD_MODE;
        SAR_REG_WORD(tneta1570_dev, TNETA_INT_MASK) = TNETA_R2_STANDARD_INTS;
        
	error = dev->phy->start(dev);
	if (error) return error;
        error = start_tx(dev);
	if (error) return error;
	error = start_rx(dev);
	if (error) return error;

        return 0;
}



static void tneta1570_close(struct atm_vcc *vcc)
{
	EVENT(">tneta1570_close\n",0,0);
	if (!TNETA1570_VCC(vcc)) return;
	vcc->flags &= ~ATM_VF_READY;
	close_rx(vcc);
	close_tx(vcc);
	EVENT("tneta1570_close: done waiting\n",0,0);
	/* deallocate memory */
	kfree(TNETA1570_VCC(vcc));
	TNETA1570_VCC(vcc) = NULL;
	vcc->flags &= ~ATM_VF_ADDR;
	/*foo();*/
}


static int tneta1570_open(struct atm_vcc *vcc,short vpi,int vci)
{
	struct tneta1570_dev *tneta1570_dev;
	struct tneta1570_vcc *tneta1570_vcc;
	int error;

	EVENT(">tneta1570_open\n",0,0);
	TNETA1570_VCC(vcc) = NULL;
        vcc->alloc_tx = tneta1570_alloc_tx; /* set my skb_alloc function */
   
        tneta1570_dev = TNETA1570_DEV(vcc->dev);
	error = atm_find_ci(vcc,&vpi,&vci);  /* get connection id */
	if (error) return error;
	vcc->vpi = vpi;
	vcc->vci = vci;
	if (vcc->aal != ATM_AAL0 && vcc->aal != ATM_AAL5) return -EINVAL;
	tneta1570_vcc = kmalloc(sizeof(struct tneta1570_vcc),GFP_KERNEL);
	if (!tneta1570_vcc) return -ENOMEM;
	TNETA1570_VCC(vcc) = tneta1570_vcc;
	DPRINTK(DEV_LABEL "(itf %d): open %d.%d\n",vcc->dev->number,vcc->vpi,
	    vcc->vci);
	vcc->flags |= ATM_VF_ADDR; /* set flag */
	if ((error = open_rx(vcc))) {    /* open rx */
	        tneta1570_close(vcc);    /* close on error */
		return error;
	}
	if ((error = open_tx(vcc))) {    /* open tx */
		tneta1570_close(vcc);
		return error;
	}
	vcc->flags |= ATM_VF_READY;  /* set IF ready */
	/* should power down SUNI while !ref_count @@@ */
	return 0;
}


static int tneta1570_ioctl(struct atm_dev *dev,unsigned int cmd,unsigned long arg)
{
	struct tneta1570_dev *tneta1570_dev;

        tneta1570_dev = TNETA1570_DEV(dev);
	printk(DEV_LABEL "ioctl \n");
	switch (cmd) {
		case TNETA_LOOP:
			if(arg)
			SAR_REG_WORD(tneta1570_dev, TNETA_CONFIG) |=
				TNETA_R0_LOOP;
			else
			SAR_REG_WORD(tneta1570_dev, TNETA_CONFIG) &=
			 	~TNETA_R0_LOOP;
			return 0;
		case TNETA_INVHEC:
			if(arg)
			SAR_REG_WORD(tneta1570_dev, TNETA_CONFIG) |=
				TNETA_R0_TX_HECERR;
			else
			SAR_REG_WORD(tneta1570_dev, TNETA_CONFIG) &=
			 	~TNETA_R0_TX_HECERR;
			return 0;
		case TNETA_ENTX:
			if(arg)
			SAR_REG_WORD(tneta1570_dev, TNETA_CONFIG) |=
				TNETA_R0_TX_ENABLE;
			else
			SAR_REG_WORD(tneta1570_dev, TNETA_CONFIG) &=
			 	~TNETA_R0_TX_ENABLE;
			return 0;
		case TNETA_ENRX:
			if(arg)
			SAR_REG_WORD(tneta1570_dev, TNETA_CONFIG) |=
				TNETA_R0_RX_ENABLE;
			else
			SAR_REG_WORD(tneta1570_dev, TNETA_CONFIG) &=
			 	~TNETA_R0_RX_ENABLE;
			return 0;
		case TNETA_BP:
			if(arg)
			SAR_REG_WORD(tneta1570_dev, TNETA_CONFIG) |=
				TNETA_R0_PERBUFFER;
			else
			SAR_REG_WORD(tneta1570_dev, TNETA_CONFIG) &=
			 	~TNETA_R0_PERBUFFER;
			return 0;
		case TNETA_RAT:
			if(arg)
			SAR_REG_WORD(tneta1570_dev, TNETA_CONFIG) |=
				TNETA_R0_RAT_ENABL;
			else
			SAR_REG_WORD(tneta1570_dev, TNETA_CONFIG) &=
			 	~TNETA_R0_RAT_ENABL;
			return 0;
		case TNETA_RXUNKN:
			return(SAR_REG_SHORT(tneta1570_dev, TNETA_UNKNOWN_P));
		case TNETA_HECERR:
			return(SAR_REG_SHORT(tneta1570_dev, TNETA_S_HEC_ERR));
		case TNETA_AAL5DISC:
		return(SAR_REG_SHORT(tneta1570_dev, TNETA_S_AAL_DISCARD));
		case TNETA_RXCELL:
			return(SAR_REG_WORD(tneta1570_dev, TNETA_CELL_RXC));
 		case TNETA_TXCELL:
			return(SAR_REG_WORD(tneta1570_dev, TNETA_CELL_TXC));
		default:
			if (!dev->phy->ioctl) return -EINVAL;
			return dev->phy->ioctl(dev,cmd,arg);
		}
}


static int tneta1570_getsockopt(struct atm_vcc *vcc,int level,int optname,
    char *optval,int *optlen)
{
	return -EINVAL;
}


static int tneta1570_setsockopt(struct atm_vcc *vcc,int level,int optname,
    char *optval,int optlen)
{
	return -EINVAL;
}

/*
 * -- send a PDU (AAL5 or AAL0)
 */
static int tneta1570_send(struct atm_vcc *vcc,struct sk_buff *skb)
{
	EVENT(">tneta1570_send\n",0,0);
	if (!skb) {
		printk("!skb in tneta1570_send ?\n");
		dev_kfree_skb(skb,FREE_WRITE);
		return -EINVAL;
	}
	if (vcc->aal == ATM_AAL0) {
		if (skb->len != ATM_CELL_SIZE-1) {
			dev_kfree_skb(skb,FREE_WRITE);
			return -EINVAL;
		}
     	        *(unsigned long *) skb->data = htonl(*(unsigned long *)
		                                       skb->data);
	                /* correct byte order of an AAL0 header */
	}
   
submitted++;
	skb->atm.vcc = vcc;

        return do_tx(skb);
}


static void tneta1570_phy_put(struct atm_dev *dev,unsigned char value,
    unsigned long addr)
{
	int i=0;
	TNETA1570_DEV(dev)->phy[addr+TNETA_SUNI_RDREQ]=0;
	while(i++<20 && !(TNETA1570_DEV(dev)->phy[addr+TNETA_SUNI_RDREQ] & 0x10));
	TNETA1570_DEV(dev)->phy[addr] = value;
}



static unsigned char tneta1570_phy_get(struct atm_dev *dev,unsigned long addr)
{
	volatile unsigned tmp; /* force 32 bit access */
	int i=0;
        /* set address */
	TNETA1570_DEV(dev)->phy[addr+TNETA_SUNI_RDREQ]=0;
        while(i++<20 && !(TNETA1570_DEV(dev)->phy[addr+TNETA_SUNI_RDREQ] & 0x10));
	TNETA1570_DEV(dev)->phy[addr+TNETA_SUNI_RDREQ]=i=0;
	while(i++<20 && !(TNETA1570_DEV(dev)->phy[addr+TNETA_SUNI_RDREQ] & 0x08));
        tmp=TNETA1570_DEV(dev)->phy[addr];

	return (unsigned char)(0xff & tmp);
}


static struct atmdev_ops ops = {
	tneta1570_open,
	tneta1570_close,
	tneta1570_ioctl,
	tneta1570_getsockopt,
	tneta1570_setsockopt,
	tneta1570_send,
	NULL,                   /* no tneta1570_sg_send */
	NULL,			/* no poll */
	NULL,			/* no send_oam ???? */
	tneta1570_phy_put,
	tneta1570_phy_get,
	NULL,			/* no feedback */
	NULL,			/* no change_qos */
	NULL			/* no free_rx_skb */
};


int tneta1570_detect(void)
{
	struct atm_dev *dev;
	struct tneta1570_dev *tneta1570_dev;
	int index;

	if (!pcibios_present()) {
		printk(DEV_LABEL " driver but no PCI BIOS ?\n");
		return 0;
	}
	tneta1570_dev = (struct tneta1570_dev *) kmalloc(sizeof(struct tneta1570_dev),
	    GFP_KERNEL);
	if (!tneta1570_dev) return -ENOMEM;
	index = 0;
	while (!pcibios_find_device(PCI_VENDOR_ID_TI,
	                            PCI_DEVICE_ID_TI_TNETA1570,
				    index,
	                            &tneta1570_dev->bus,
				    &tneta1570_dev->dev_fn)
	     || !pcibios_find_device(PCI_VENDOR_ID_TI,
	                            PCI_DEVICE_ID_TI_TNETA1575,
				    index,
	                            &tneta1570_dev->bus,
				    &tneta1570_dev->dev_fn))
        {
		dev = atm_dev_register(DEV_LABEL,&ops,0);
		if (!dev) break;
		TNETA1570_DEV(dev) = tneta1570_dev;
	   
	        if (tneta1570_init(dev) || tneta1570_start(dev)) {
			atm_dev_deregister(dev);
			break;
		}
		tneta1570_dev->more = tneta1570_boards;
		tneta1570_boards = dev;
		index++;
		tneta1570_dev = (struct tneta1570_dev *) kmalloc(sizeof(struct tneta1570_dev),
		    GFP_KERNEL);
		if (!tneta1570_dev) break;
                
	}
	return index;
}

#ifdef MODULE

int init_module(void)
{
	if (!tneta1570_detect()) {
		printk(DEV_LABEL ": no adapter found\n");
		return -ENXIO;
	}
	MOD_INC_USE_COUNT;
	return 0;
}
void cleanup_module(void){

/* not yet */
}
#endif
