/* nicstar.c, M. Welsh (matt.welsh@cl.cam.ac.uk)
 *
 * Linux driver for the IDT77201 NICStAR PCI ATM controller.
 * PHY component is expected to be 155 Mbps S/UNI-Lite or IDT 77155;
 * see init_nicstar() for PHY initialization to change this. This driver
 * expects the Linux ATM stack to support scatter-gather lists 
 * (skb->atm.iovcnt != 0) for Rx skb's passed to vcc->push.
 *
 * Implementing minimal-copy of received data:
 *   IDT always receives data into a small buffer, then large buffers
 *     as needed. This means that data must always be copied to create
 *     the linear buffer needed by most non-ATM protocol stacks (e.g. IP)
 *     Fix is simple: make large buffers large enough to hold entire
 *     SDU, and leave <small_buffer_data> bytes empty at the start. Then
 *     copy small buffer contents to head of large buffer.
 *   Trick is to avoid fragmenting Linux, due to need for a lot of large
 *     buffers. This is done by 2 things:
 *       1) skb->destructor / skb->atm.recycle_buffer
 *            combined, allow nicstar_free_rx_skb to be called to
 *            recycle large data buffers
 *       2) skb_clone of received buffers
 *   See nicstar_free_rx_skb and linearize_buffer for implementation
 *     details.
 *
 * Copyright (c) 1996 University of Cambridge Computer Laboratory
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * M. Welsh, 6 July 1996
 *
 */
/*
static char _modversion[] = "@(#) nicstar.c, $Id: nicstar.c,v 1.1 1997/09/01 14:09:47 heiko Exp $";
*/

#include <linux/fs.h>
#include <asm/system.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <asm/irq.h>
#include <linux/errno.h>
#include <linux/major.h>
#include <asm/segment.h>
#include <linux/kernel.h>
#include <linux/signal.h>
#include <linux/module.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/bios32.h>
#include <asm/pgtable.h>
#include <linux/pci.h>
#include <linux/ioport.h>
#include <asm/param.h>
#include <linux/sched.h>
#include <linux/timex.h>
#include <linux/atm.h>
#include <linux/atmdev.h>
#include <linux/sonet.h>

#include <linux/version.h>
char kernel_version[] = UTS_RELEASE;

#include "nicstar.h"
#include "../../net/atm/protocols.h" /* hack! */

#undef NICSTAR_DEBUG              /* Define for verbose debugging output */
#define NICSTAR_RC_FLAG 1

#ifdef NICSTAR_DEBUG
#define PRINTK(args...) printk(args)
#else
#define PRINTK(args...) 
#endif

#define SLEEP_ON(x) interruptible_sleep_on(x)
#define WAKE_UP(x) wake_up_interruptible(x)

#define TRUE 1
#define FALSE 0
#define MIN(a,b) ((a)<(b)?(a):(b))
#define MAX(a,b) ((a)>(b)?(a):(b))

#define CMD_BUSY(node) (readl((node)->membase + STAT) & 0x0200)

/* For timing */
unsigned long get_usec(void) {
  struct timeval t;
  do_gettimeofday(&t);
  return (unsigned long)((1e6 * t.tv_sec) + t.tv_usec);
}

/* Global definitions ******************************************************/
static struct nicstar_dev *nicstar_devs[NICSTAR_MAX_DEVS];
static int num_found = 0;
#if 0
static int nicstar_rx_count = 0;
static u32 nicstar_timer = 0;
#endif

/* Global/static function declarations *************************************/
static int init_nicstar(struct nicstar_dev *node);
static void nicstar_interrupt(int irq, void *dev_id, struct pt_regs *regs);
static void process_tx_statq(nicstar_devp node);
static void process_rx_statq(nicstar_devp node);
#ifdef NICSTAR_RCQ
static void process_rx_rawcellq(nicstar_devp node);
static void dequeue_rawcell(u32 *cell);
#endif
static void push_rxbufs(nicstar_devp node, int lg, 
			u32 handle1, u32 addr1, u32 handle2, u32 addr2);
static void open_rx_connection(nicstar_devp node,int index,
			       u32 status);
static void close_rx_connection(nicstar_devp node, int index);
static void push_scq_entry(nicstar_devp node, struct nicstar_scq *scq,
			   nicstar_tbd *entry, int xmit_now);
int init_module(void);
void cleanup_module(void);
static int get_ci(nicstar_devp node, struct atm_vcc *vcc,
		  short *vpi, int *vci);
static void dequeue_rx(nicstar_devp node);
static void nicstar_free_rx_skb(struct atm_vcc *vcc, struct sk_buff *skb);
static void drain_scq(nicstar_devp node, struct nicstar_scq *scq, int index);
static void free_rx_buf(nicstar_devp node, int lg, u32 buf_addr);
#if NICSTAR_CBR
static void alloc_tx(nicstar_devp node, struct atm_vcc *vcc);
static void close_cbr (nicstar_devp node, struct atm_vcc *vcc);
static void close_cbr_final (nicstar_devp node, struct atm_vcc *vcc);
#endif
static int create_scq (struct nicstar_scq **scq, int size);
static u32 linearize_buffer (struct nicstar_dev *node, struct sk_buff *skb,
			     struct atm_vcc *vcc);

/* Module entry points *****************************************************/

static int nicstar_open(struct atm_vcc *vcc, short vpi, int vci) {
  nicstar_devp node;
  int error;

  node = vcc->dev->dev_data;

  /* XXX AAL5 and AAL0 only supported */
  if ((vcc->aal != ATM_AAL5) && (vcc->aal != ATM_AAL0))
    return -EINVAL; 

  error = get_ci(node, vcc, &vpi, &vci);
  if (error) return error;
  vcc->vpi = vpi;
  vcc->vci = vci;
  vcc->dev_data = &(node->vci_map[(vpi << NUM_VCI_BITS) | vci]);
  vcc->flags |= ATM_VF_ADDR;

#if NICSTAR_CBR
#ifndef ATM_013
  if (vcc->qos.txtp.traffic_class == ATM_CBR) {
#else
  if (vcc->txtp.class == ATM_CBR) {
#endif
    PRINTK("Opening CBR connection on vpci %d.%d\n",vpi,vci);
    alloc_tx(node,vcc);
  }
#endif

  if (!(vcc->flags & ATM_VF_PARTIAL)) {

#ifndef ATM_013
    if (vcc->qos.rxtp.traffic_class != ATM_NONE) {
#else
    if (vcc->rxtp.class != ATM_NONE) {
#endif
      u32 status = 0x8000;
      switch (vcc->aal) {
	case ATM_AAL0 : break;
	case ATM_AAL34 : status |= 0x10000; break;
        case ATM_AAL5 : status |= 0x20000; break;
        default : 
	  printk ("nicstar%d: invalid AAL %d on open",node->index,vcc->aal);
	  return -EINVAL;
      }
      open_rx_connection(node, (vpi << NUM_VCI_BITS) | vci, status);
    }

    vcc->flags |= ATM_VF_PARTIAL;

  }

  vcc->flags |= ATM_VF_READY;

  return 0;
  
}

static void nicstar_close(struct atm_vcc *vcc) {
  nicstar_devp node;
  nicstar_vcimap *mapval;

  node = vcc->dev->dev_data;
  mapval = vcc->dev_data;

  vcc->flags &= ~ATM_VF_READY;

#ifndef ATM_013
  if (vcc->qos.rxtp.traffic_class != ATM_NONE) {
#else
  if (vcc->rxtp.class != ATM_NONE) {
#endif
    close_rx_connection(node, (vcc->vpi << NUM_VCI_BITS) | vcc->vci);
    vcc->flags &= ~ATM_VF_PARTIAL;
    mapval->rx = 0;
  }
  
#ifndef ATM_013
  if (vcc->qos.txtp.traffic_class != ATM_NONE) {
#else
  if (vcc->txtp.class != ATM_NONE) {
#endif
    mapval->tx = 0;
#ifndef ATM_013
    if (vcc->qos.txtp.traffic_class == ATM_CBR) {
#else
    if (vcc->txtp.class == ATM_CBR) {
#endif
#if NICSTAR_CBR
      close_cbr(node,vcc);
#endif
    }
  }

  vcc->flags &= ~ATM_VF_ADDR;
}

static int nicstar_ioctl(struct atm_dev *dev, unsigned int cmd,
			 unsigned long arg) {
  if (!dev->phy->ioctl) return -EINVAL;
  return dev->phy->ioctl(dev,cmd,arg);
}

static int nicstar_getsockopt(struct atm_vcc *vcc,
			      int level,
			      int optname,
			      char *optval,
			      int *optlen) {
  return -EINVAL;
}

static int nicstar_setsockopt(struct atm_vcc *vcc, int level,
			      int optname, char *optval, int optlen) {
  struct atm_cirange ci;
  int error;

  if (level != SOL_ATM) return -EINVAL;
  switch (optname) {
  case SO_CIRANGE:
    if (!suser()) return -EPERM;
    if (optlen != sizeof(struct atm_cirange))
      return -EINVAL;
    error = verify_area(VERIFY_READ,optval,sizeof(struct atm_cirange));
    if (error) return error;
    memcpy_fromfs(&ci,optval,sizeof(struct atm_cirange));
    if ((ci.vpi_bits == 0 || ci.vpi_bits == ATM_CI_MAX) &&
	(ci.vci_bits == NUM_VCI_BITS || ci.vci_bits == ATM_CI_MAX)) return 0;
    return -EINVAL;
    break;
    
  default:
    return -EINVAL;
  }
}

/* XXX It is possible here that when pushing multiple TBD's, the 
 * next TSI will be placed between them, causing an incomplete PDU xmit.
 * Probably fix by not allowing a TSI to be pushed onto the SCQ when
 * force_xmit == 0.
 */
static int nicstar_send(struct atm_vcc *vcc, struct sk_buff *skb) {
  nicstar_devp node;
  int pad, sindex, vci;
  nicstar_tbd tbd1;
  nicstar_scq *scq;
  u32 rate;
#ifdef NICSTAR_FASTXMIT
  int len;
  int len_buf = 0;
#else
  nicstar_tbd tbd2, tbd3;
#endif

  node = vcc->dev->dev_data;

  if (!((nicstar_vcimap *)vcc->dev_data)->tx) {
    printk("nicstar%d: Tx on a non-Tx VC?\n",
	   ((nicstar_devp)vcc->dev->dev_data)->index);
    vcc->stats->tx_err++;
    dev_kfree_skb(skb,FREE_WRITE);
    return -EINVAL;
  }
#ifndef NICSTAR_CBR
#ifndef ATM_013
  if (vcc->qos.txtp.traffic_class == ATM_CBR) {
#else
  if (vcc->txtp.class == ATM_CBR) {
#endif /* ATM_013*/
    printk("nicstar: Sorry - CBR support in progress.\n");
    dev_kfree_skb(skb,FREE_WRITE);
    return -EINVAL;
  }
#endif /* NICSTAR_CBR */

  /* Verify AAL is supported */
  if ((vcc->aal != ATM_AAL5) && (vcc->aal != ATM_AAL0)) {
    printk("nicstar%d: Only AAL5 and AAL0 supported, sorry.\n",
	   ((nicstar_devp)vcc->dev->dev_data)->index);
    vcc->stats->tx_err++;
    dev_kfree_skb(skb,FREE_WRITE);
    return -EINVAL;
  }

  /* Verify that we have a linear buffer to transmit */
  if (skb->atm.iovcnt) {
    printk("nicstar%d: S/G not supported, sorry.\n",
	   ((nicstar_devp)vcc->dev->dev_data)->index);
    vcc->stats->tx_err++;
    dev_kfree_skb(skb,FREE_WRITE);
    return -EINVAL;
  }

  skb->atm.vcc = vcc;

  vci = (vcc->vpi << NUM_VCI_BITS) | vcc->vci;
#ifndef ATM_013
  if (vcc->qos.txtp.traffic_class == ATM_CBR) {
#else
  if (vcc->txtp.class == ATM_CBR) {
#endif
#ifdef NICSTAR_CBR
    scq = (node->vci_map[vci]).scq;
    rate = 0;
#endif
  }
  else {
    scq = node->scq;
    rate = TBD_UBR_M | TBD_UBR_N;
  }
  sindex = scq->next - scq->base;


  /* AAL-dependent transmits */
  if (vcc->aal == ATM_AAL0) {
    /* size is always 48 bytes */
    len = ATM_CELL_PAYLOAD & ~3;
    /* create tbd */
    tbd1.status = TBD_AAL0 | TBD_ENDPDU | rate | len;
    tbd1.buf_addr = (u32)(skb->data + 4);
    tbd1.ctl_len = 0x0;
    tbd1.cell_hdr = (u32)(*((u32 *)skb->data));
    push_scq_entry(node,scq,&tbd1,1);
  }
  else if (vcc->aal == ATM_AAL5) {   /* Transmit AAL 5 */
#ifdef NICSTAR_FASTXMIT
    /* Speed up and simplify transmission by "cheating".
       Basic problem is that rev C3 only supports mod 48
       buffer sizes; Matt's original uses shadow cells
       to provide this. We cheat, instead, and simply
       lie about the length of the buffer. This may 
       have security implications and problems from
       going over the end of a buffer.

       For rev D or better, there is a better way; these
       chips support mod 4 buffer sizes. Thus, we simply
       pad the entire PDU length to mod 48 by adding a
       pointer to the zero'd tmp_cell buffer. (SWD) */
    len = (((skb->len + 8 + 47) / 48) * 48);
    pad = len - skb->len;
    PRINTK ("new len %d skb len %ld skb end %x skb tail %x\n",len,skb->len,(u32)skb->end,(u32)skb->tail);

    if (!(node->rev_C3)) { /* SAR supports mod 4 buffer sizes */
      len_buf = (((skb->len + 3) / 4) * 4);
      if (pad == 0) {
	tbd1.status = TBD_AAL5 | TBD_ENDPDU | rate | skb->len;
      }
      else {
	tbd1.status = TBD_AAL5 | rate | len_buf;
      }
    }
    else {
      tbd1.status = TBD_AAL5 | TBD_ENDPDU | rate | len;
      if (skb->end - pad < skb->tail) {
	PRINTK ("nicstar%d: Whoops. overpadding a buffer!\n",node->index);
      }
    }

    tbd1.buf_addr = (u32)skb->data;
    tbd1.ctl_len = skb->len;
    tbd1.cell_hdr = (vcc->vpi << 20) | (vcc->vci << 4);
    scq->scq_shadow[sindex].skb = skb;

    if (!(node->rev_C3) && (pad != 0)) {
      push_scq_entry(node, scq, &tbd1, 0);
      tbd1.status = TBD_AAL5 | TBD_ENDPDU | rate | (len - len_buf);
      tbd1.buf_addr = (u32)node->tmp_cell;
      tbd1.ctl_len = skb->len;
      tbd1.cell_hdr = (vcc->vpi << 20) | (vcc->vci << 4);
      push_scq_entry(node, scq, &tbd1, 1);
    }
    else {
      push_scq_entry(node, scq, &tbd1, 1);
    }

#else
    if (skb->len > 48) {
    /* YYY If we transmit a buffer where pad == 48, an extra cell
     is transmitted under this scheme if skb->len > 48 (SWD) */

      pad = 48 - ((skb->len + 8) % 48);
      if (pad <= 40) {
        memcpy(node->scq->scq_shadow[sindex].cell,
	       skb->data + (skb->len) - (40 - pad),
	       40 - pad);
        
        /* Need to do this unless we want to send garbage at
         * end of last cell ... only bad for security */
        memset(node->scq->scq_shadow[sindex].cell + (40 - pad), 0, pad);

        tbd1.status = 0x08810000 | (((skb->len - (40 - pad))/sizeof(u32)) << 2); 
        tbd1.buf_addr = (u32)skb->data;
        tbd1.ctl_len = skb->len;
        tbd1.cell_hdr = (vcc->vpi << 20) | (vcc->vci << 4);
      
        node->scq->scq_shadow[sindex].skb = skb;
        push_scq_entry(node, node->scq, &tbd1, 0);

        tbd2.status = 0x48810030; 
        tbd2.buf_addr = (u32)(node->scq->scq_shadow[sindex].cell);
        tbd2.ctl_len = skb->len;
        tbd2.cell_hdr = (vcc->vpi << 20) | (vcc->vci << 4);

        push_scq_entry(node, node->scq, &tbd2, 1);

     } else {
        /* Pad > 40 */
        pad = pad - 40; /* Pad of second-to-last cell */
        memcpy(node->scq->scq_shadow[sindex].cell,
 	       skb->data + (skb->len) - (48 - pad), 48 - pad);
        memset(node->scq->scq_shadow[sindex].cell + (48 - pad), 0, pad);

        tbd1.status = 0x08810000 | (((skb->len - (48 - pad))/sizeof(u32)) << 2);
        tbd1.buf_addr = (u32)skb->data;
        tbd1.ctl_len = skb->len;
        tbd1.cell_hdr = (vcc->vpi << 20) | (vcc->vci << 4);

        node->scq->scq_shadow[sindex].skb = skb;
        push_scq_entry(node, node->scq, &tbd1, 0);

        tbd2.status = 0x08810030;
        tbd2.buf_addr = (u32)(node->scq->scq_shadow[sindex].cell);
        tbd2.ctl_len = skb->len;
        tbd2.cell_hdr = (vcc->vpi << 20) | (vcc->vci << 4);

        push_scq_entry(node, node->scq, &tbd2, 0);

        tbd3.status = 0x48810030;
        tbd3.buf_addr = (u32)(node->tmp_cell);
        tbd3.ctl_len = skb->len;
        tbd3.cell_hdr = (vcc->vpi << 20) | (vcc->vci << 4);

        push_scq_entry(node, node->scq, &tbd3, 1);
      }
      
    } else if (skb->len <= 40) {

      pad = 48 - (skb->len + 8);
      memcpy(node->scq->scq_shadow[sindex].cell,
	     skb->data, skb->len);
      memset(node->scq->scq_shadow[sindex].cell + skb->len, 0, pad);
      tbd1.status = 0x48810030; 
      tbd1.buf_addr = (u32)(node->scq->scq_shadow[sindex].cell);
      tbd1.ctl_len = skb->len;
      tbd1.cell_hdr = (vcc->vpi << 20) | (vcc->vci << 4);

      node->scq->scq_shadow[sindex].skb = skb;
      push_scq_entry(node, node->scq, &tbd1, 1);

    } else {

      /* 41 .. 48 inclusive */
      pad = 48 - (skb->len);
      memcpy(node->scq->scq_shadow[sindex].cell,
             skb->data, skb->len);
      memset(node->scq->scq_shadow[sindex].cell + skb->len, 0, pad);

      tbd1.status = 0x08810030;
      tbd1.buf_addr = (u32)(node->scq->scq_shadow[sindex].cell);
      tbd1.ctl_len = skb->len;
      tbd1.cell_hdr = (vcc->vpi << 20) | (vcc->vci << 4);

      node->scq->scq_shadow[sindex].skb = skb;
      push_scq_entry(node, node->scq, &tbd1, 0);

      tbd2.status = 0x48810030;
      tbd2.buf_addr = (u32)(node->tmp_cell);
      tbd2.ctl_len = skb->len;
      tbd2.cell_hdr = (vcc->vpi << 20) | (vcc->vci << 4);

      push_scq_entry(node, node->scq, &tbd2, 1);

    }
#endif /* NICSTAR_FASTXMIT */
  }

  vcc->stats->tx++;
  return 0;

}

static struct atmdev_ops atm_ops = {
  nicstar_open,        /* open */
  nicstar_close,       /* close */
  nicstar_ioctl,       /* ioctl */
  nicstar_getsockopt,  /* getsockopt */
  nicstar_setsockopt,  /* setsockopt */
  nicstar_send,        /* send */
  NULL,  /* sg_send */
  NULL,  /* poll */
  NULL,  /* send_oam */
  NULL,  /* phy_put */
  NULL,  /* phy_get */
  NULL,  /* feedback */
#ifndef ATM_013
  NULL,  /* change_qos */
#endif
  nicstar_free_rx_skb, /* free_rx_skb */
};

static int do_detect(void) {

  int error;
  static unsigned char pci_bus, pci_devfn;
  int pci_index;
  struct nicstar_dev *node;
  unsigned short pci_command;
  unsigned char pci_latency;
  unsigned char irq;
  int i;
  u32 gp;

  PRINTK("nicstar: init_module called\n");

  if (!pcibios_present()) {
    printk("nicstar: Fatal error: No PCI BIOS present\n");
    return -EIO;
  }

  for (i = 0; i < NICSTAR_MAX_DEVS; i++)
    nicstar_devs[i] = NULL;

  PRINTK("nicstar: Searching for IDT77201 NICStAR: vendor 0x%x, device 0x%x\n",
	 PCI_VENDOR_ID_IDT,
	 PCI_DEVICE_ID_IDT_IDT77201);

  for (pci_index = 0; pci_index < 8; pci_index++) {

    if ((error = pcibios_find_device(PCI_VENDOR_ID_IDT,
				     PCI_DEVICE_ID_IDT_IDT77201,
				     pci_index, &pci_bus, &pci_devfn))
	!= PCIBIOS_SUCCESSFUL)
      break;

    /* Allocate a device structure */
    node = kmalloc(sizeof(struct nicstar_dev),GFP_KERNEL);
    if (node == NULL) {
      printk("nicstar: Can't allocate device struct for device %d!\n",pci_index);
      return -EIO;
    }
    nicstar_devs[num_found] = node;
    node->index = num_found;
    node->pci_bus = pci_bus;
    node->pci_devfn = pci_devfn;

    error = pcibios_read_config_byte(pci_bus,pci_devfn,PCI_REVISION_ID,
				     &(node->revid));
    if (error) {
      printk("nicstar: Can't read REVID from device %d\n",pci_index);
      return -EIO;
    }
  
    error = pcibios_read_config_dword(pci_bus,pci_devfn,PCI_BASE_ADDRESS_0,
				      (unsigned int *)&(node->iobase));
    if (error) {
      printk("nicstar: Can't read iobase from device %d.\n",pci_index);
      return -EIO;
    }
  
    error = pcibios_read_config_dword(pci_bus,pci_devfn,PCI_BASE_ADDRESS_1,
				      (unsigned int *)&(node->membase));
    if (error) {
      printk("nicstar: Can't read membase from device %d.\n",pci_index);
      return -EIO;
    }
  
    if (pcibios_read_config_byte(pci_bus,pci_devfn,PCI_INTERRUPT_LINE,
				 (unsigned char *)&irq)) {
      printk("nicstar: Can't read INTLN from device %d.\n",pci_index);
      return -EIO;
    }
    node->irq = irq;
    
    node->iobase &= PCI_BASE_ADDRESS_IO_MASK;
    node->membase &= PCI_BASE_ADDRESS_MEM_MASK;

    node->membase = (unsigned long)vremap((unsigned int)node->membase,NICSTAR_IO_SIZE);
    if ((unsigned long *)node->membase == NULL) {
      printk("nicstar: Can't remap membase for board %d.\n",node->index);
      return -EIO;
    }

    printk("nicstar: Found device index %d, bus %d, function %d.\n",
	   pci_index,pci_bus,pci_devfn);
    printk("nicstar: Revision 0x%x, using IRQ %d.\n",node->revid,node->irq);
    printk("nicstar: iobase: 0x%lx, membase: 0x%lx\n",node->iobase,node->membase);

    /* Enable memory space and busmastering */
    if (pcibios_read_config_word(pci_bus,pci_devfn,PCI_COMMAND,&pci_command)) {
      printk("nicstar: Can't read PCI_COMMAND from device %d.\n",pci_index);
      return -EIO;
    }

    pci_command |= (PCI_COMMAND_MEMORY | PCI_COMMAND_SPECIAL | PCI_COMMAND_MASTER);
    error = pcibios_write_config_word(pci_bus,pci_devfn,PCI_COMMAND,
				      pci_command);

    if (error) {
      printk("nicstar: Can't enable board memory for device %d.\n",pci_index);
      return -EIO;
    }
    
    /* Check latency timer */
    if (pcibios_read_config_byte(pci_bus,pci_devfn,PCI_LATENCY_TIMER,&pci_latency)) {
      printk("nicstar: Can't read latency timer from device %d.\n",pci_index);
      return -EIO;
    }
    if (pci_latency < 32) {
      printk("nicstar: Setting latency timer from %d to 32 clocks.\n",pci_latency);
      pcibios_write_config_byte(pci_bus,pci_devfn,PCI_LATENCY_TIMER,32);
    }

    num_found++;

    /* Check for SAR, revision C3; on this revision, bit 15/GP is
     not writable, only readable. */
    gp = readl(node->membase + GP);
    writel(gp | 0x8000,node->membase + GP);
    node->rev_C3 = (readl(node->membase + GP) & 0x8000) ? 0 : 1;
    if (node->rev_C3) {
      printk("nicstar%d: Revision C3 77201 detected\n",node->index);
    }
    else {
      printk("nicstar%d: Revision D or later 77201 detected\n",node->index);
    }
    writel(gp,node->membase + GP); /* restore original contents */

    /* Initialize random things */
    node->inuse = 0;

    /* Get IRQ */
    if (request_irq(node->irq, &nicstar_interrupt, SA_INTERRUPT, "nicstar", node)) {
      printk("nicstar: Can't allocate IRQ %d.\n",node->irq);
      return -EIO;
    }
    
    if (init_nicstar(node)) {
      printk("nicstar: Error initializing device %d.\n",pci_index);
      free_irq(node->irq, node);
      return -EIO;
    }

    /* Register device */
    node->dev = atm_dev_register("nicstar", &atm_ops, 0);
    if (node->dev == NULL) {
      printk("nicstar: Can't register device %d.\n",node->index);
      free_irq(node->irq, node);
      return -EIO;
    }
    node->dev->esi[0] = '\0';
    node->dev->esi[1] = '\34';
    node->dev->esi[2] = '\100';
    node->dev->esi[3] = '\0';
    node->dev->esi[4] = '\17';
    node->dev->esi[5] = '\110';
    node->dev->dev_data = node;

    node->dev->ci_range.vpi_bits = NUM_VPI_BITS;
    node->dev->ci_range.vci_bits = NUM_VCI_BITS;

  }
    
  if (num_found == 0) return -EIO;

  printk("nicstar: %d devices found.\n",num_found);
  return num_found;
  
}


#ifndef MODULE


int nicstar_detect(void)
{
    int devs;

    devs = do_detect();
    return devs < 0 ? 0 : devs;
}


#else


int init_module(void) {
	int devs;

	printk("nicstar: Installing %s of %s %s.\n",__FILE__,__DATE__,__TIME__);
	devs = do_detect(void);
	if (!devs) {
		printk(KERN_ERR "nicstar: no adapter found\n");
		return -ENXIO;
	}
	return 0;
}


void cleanup_module(void) {
  int i, error;
  nicstar_devp node;
  unsigned short pci_command;
  
  PRINTK("nicstar: cleanup_module called\n");

  if (MOD_IN_USE) 
    PRINTK("nicstar: Device busy, remove delayed.\n");

  /* Free up the device resources */
  for (i = 0; i < NICSTAR_MAX_DEVS; i++) {
    if (nicstar_devs[i] != NULL) {
      node = nicstar_devs[i];
      /* Turn everything off */
      writel(0x00000000, (node->membase)+CFG);

      /* Disable PCI busmastering */
      if (pcibios_read_config_word(node->pci_bus,node->pci_devfn,PCI_COMMAND,&pci_command)) {
        printk("nicstar: Can't read PCI_COMMAND from device %d.\n",node->index);
	continue;
      }
      pci_command &= ~PCI_COMMAND_MASTER;
      error = pcibios_write_config_word(node->pci_bus,node->pci_devfn,
        PCI_COMMAND, pci_command);
      if (error) {
        printk("nicstar: Can't disable busmastering for device %d.\n",node->index);
        continue;
      }
      
      drain_scq(node,node->scq,node->scq->next - node->scq->base);
      kfree(node->rx_statq_orig);
      kfree(node->tx_statq_orig);
      kfree(node->sm_bufs);
      kfree(node->lg_bufs);
      kfree(node->scq->orig);
      kfree(node->scq);
      free_irq(node->irq, node);

      {
	struct sk_buff *skb;
	while ((skb = skb_dequeue(&node->rx_skb_queue)))
	  dev_kfree_skb(skb, FREE_READ);
      }

      atm_dev_deregister(node->dev);

      kfree(nicstar_devs[i]);
    }
  }

  printk("nicstar: Module cleanup succeeded.\n");

}


#endif


/* NICStAR functions *******************************************************/

static inline u32 read_sram(nicstar_devp node, u32 addr) {
  u32 data;
  unsigned long flags;
  
  save_flags(flags); cli();
  while (CMD_BUSY(node)) ;
  writel(((0x50000000) | ((addr & 0x1ffff) << 2)), node->membase + CMD);
  while (CMD_BUSY(node)) ;
  data = readl(node->membase + DR0);
  restore_flags(flags);
  return data;
}

static inline void write_sram(nicstar_devp node, int count, u32 addr,
			     u32 d0, u32 d1, u32 d2, u32 d3) {
  unsigned long flags;

  save_flags(flags); cli();
  while (CMD_BUSY(node));
  switch (count) {
  case 4:
    writel(d3, node->membase + DR3);
  case 3:
    writel(d2, node->membase + DR2);
  case 2:
    writel(d1, node->membase + DR1);
  case 1:
    writel(d0, node->membase + DR0);
    break;
  default:
    restore_flags(flags);
    return;
  }

  writel((0x40000000) | ((addr & 0x1ffff) << 2) | (count-1), node->membase+CMD);
  restore_flags(flags);
  return;
}
    
static int init_nicstar(struct nicstar_dev *node) {
  u32 i, d;
  unsigned short pci_command;
  int error; 
  u32 tmp;

  d = readl(node->membase + STAT);
  /* Clear timer if overflow */
  if (d & 0x00000800) {
    writel(0x00000800, node->membase+STAT);
  }

  /* S/W reset */
  writel(0x80000000, node->membase+CFG);
  SLOW_DOWN_IO; SLOW_DOWN_IO; SLOW_DOWN_IO;
  writel(0x00000000, node->membase+CFG);
  
  /* Turn everything off, but use 4k recv status queue */
  writel(0x00400000, node->membase+CFG);
  d = readl(node->membase + CFG);

  /* Determine PHY type/speed by reading PHY reg 0 (from IDT) */
  /* YYY This tested only for 77105 (SWD) */
  writel(0x80000200,node->membase + CMD); /* read PHY reg 0 command */
  while (CMD_BUSY(node)) ;
  tmp = readl(node->membase + DR0);
  if (tmp == 0x9) {
    node->max_pcr = IDT_25_PCR;
    printk ("nicstar%d: PHY device appears to be 25Mbps\n",node->index);
  }
  else if (tmp == 0x30) {
    node->max_pcr = ATM_OC3_PCR;
    printk ("nicstar%d: PHY device appears to be 155Mbps\n",node->index);
  }

#ifdef TEST_LOOPBACK 
  /* PHY reset */
  d = readl(node->membase+GP);
  writel(d & 0xfffffff7, node->membase+GP); /* Out of reset */
  { int j; for (j = 0; j < 100000; j++) ; }
  writel(d | 0x00000008, node->membase+GP); /* Reset */
  { int j; for (j = 0; j < 100000; j++) ; }
  writel(d & 0xfffffff7, node->membase+GP); /* Out of reset */

  /* PHY loopback */
  while(CMD_BUSY(node)) ;
  printk ("nicstar%d: Setting PHY Loopback\n",node->index);
  writel(0x00000022, node->membase + DR0);
  if (node->max_pcr == IDT_25_PCR) { /* loopback for 77105 */
    writel(0x90000202, node->membase + CMD);
  }
  else { /* assume 77155 */
    writel(0x90000205, node->membase + CMD);
  }

#else

  /* PHY reset */
  d = readl(node->membase+GP);
  writel(d & 0xfffffff7, node->membase+GP); /* Out of reset */
  { int j; for (j = 0; j < 100000; j++) ; }
  writel(d | 0x00000008, node->membase+GP); /* Reset */
  { int j; for (j = 0; j < 100000; j++) ; }
  writel(d & 0xfffffff7, node->membase+GP); /* Out of reset */

  while (CMD_BUSY(node)) ;
  writel(0x80000100, node->membase+CMD);    /* Sync UTOPIA with SAR clock */
  { int j; for (j = 0; j < 100000; j++) ; }

  /* Normal mode */
  while(CMD_BUSY(node)) ;
  writel(0x00000020, node->membase + DR0);
  if (node->max_pcr == IDT_25_PCR) { /* 77105 */
    writel(0x90000202, node->membase + CMD);
  }
  else { /* assume 77155 */
    writel(0x90000205, node->membase + CMD);
  }

#endif /* PHY_LOOPBACK */

  /* Re-enable memory space and busmastering --- in case the PCI reset
   * turned us off ...
   */
  if (pcibios_read_config_word(node->pci_bus,node->pci_devfn,PCI_COMMAND,&pci_command)) {
    printk("nicstar: Can't read PCI_COMMAND from device %d.\n",node->index);
    return -EIO;
  }
  pci_command |= (PCI_COMMAND_MEMORY | PCI_COMMAND_SPECIAL | PCI_COMMAND_MASTER);
  error = pcibios_write_config_word(node->pci_bus,node->pci_devfn,PCI_COMMAND,
				    pci_command);
  if (error) {
    printk("nicstar: Can't enable board memory for device %d.\n",node->index);
    return -EIO;
  }

  /* Set up the recv connection table */
  for (i = 0; i < RX_CTABLE_SIZE; i+=4) {
#if NICSTAR_RCQ
    write_sram(node, 4, i, 0x00008000, 0x00000000, 0x00000000, 0xffffffff);
#else
    write_sram(node, 4, i, 0x00000000, 0x00000000, 0x00000000, 0xffffffff);
#endif
  }

  writel(0x0, node->membase + VPM);

  /* Clear VCI map and SCQ shadow */
  memset(node->vci_map, 0, sizeof(node->vci_map));
  memset(node->tmp_cell, 0, sizeof(node->tmp_cell));

  /* Allocate Rx status queue */
  /* XXX What about 64-bit pointers? */
  node->rx_statq_orig = kmalloc(2*8192, GFP_KERNEL);
  if (!node->rx_statq_orig) {
    printk("nicstar%d: Can't allocate Rx status queue.\n",node->index);
    return -ENOMEM;
  }
  node->rx_statq = (nicstar_rsqe *)(((u32)node->rx_statq_orig + (8192 - 1)) & ~(8192 - 1));

  printk("nicstar%d: Rx status queue at 0x%x (0x%x).\n",node->index, (u32)node->rx_statq, (u32)node->rx_statq_orig);

  for (i = 0; i < RX_STATQ_ENTRIES; i++) {
    node->rx_statq[i].vpi_vci = 0x0;
    node->rx_statq[i].buf_handle = 0x0;
    node->rx_statq[i].crc = 0x0;
    node->rx_statq[i].status = 0x0;
  }

  node->rx_statq_next = node->rx_statq;
  node->rx_statq_last = node->rx_statq + (RX_STATQ_ENTRIES - 1);
  writel((u32)node->rx_statq, node->membase + RSQB);
  writel((u32)0x0, node->membase + RSQH);

  /* Allocate Tx status queue */
  /* XXX What about 64-bit pointers? */
  node->tx_statq_orig = (caddr_t)kmalloc(2*8192, GFP_KERNEL);
  if (!node->tx_statq_orig) {
    printk("nicstar%d: Can't allocate Tx status queue.\n",node->index);
    return -ENOMEM;
  }
  node->tx_statq = (nicstar_tsi *)(((u32)node->tx_statq_orig + (8192 - 1)) & ~(8192 - 1));

  printk("nicstar%d: Tx status queue at 0x%x (0x%x).\n",node->index, (u32)node->tx_statq, (u32)node->tx_statq_orig);

  for (i = 0; i < TX_STATQ_ENTRIES; i++) {
    node->tx_statq[i].timestamp = 0x80000000;
  }
  node->tx_statq_next = node->tx_statq;
  node->tx_statq_last = node->tx_statq + (TX_STATQ_ENTRIES - 1);
  
  writel((u32)node->tx_statq, node->membase + TSQB);
  writel((u32)0x0, node->membase + TSQH);

  /* Allocate buffer queues */

  node->sm_bufs = (caddr_t)kmalloc(SM_BUFSZ * NUM_SM_BUFS, GFP_KERNEL);
  if (!node->sm_bufs) {
    printk("nicstar%d: Can't allocate %d %d-byte small buffers!\n",
	   node->index, NUM_SM_BUFS, SM_BUFSZ);
    return -ENOMEM;
  }

#if 0
  /* Old code for handling large buffers */
  node->lg_bufs = (caddr_t)kmalloc((SM_BUFSZ + LG_BUFSZ) * NUM_LG_BUFS, GFP_KERNEL);
  if (!node->lg_bufs) {
    printk("nicstar%d: Can't allocate %d %d-byte large buffers!\n",
	   node->index, NUM_LG_BUFS, LG_BUFSZ);
    return -ENOMEM;
  }
  p = node->lg_bufs + SM_BUFSZ;
#endif
  for (i = 0; i < NUM_LG_BUFS; i++) {
    node->rx_lg_skb[i] = alloc_skb(SM_BUFSZ + LG_BUFSZ, GFP_KERNEL);
    skb_reserve(node->rx_lg_skb[i],SM_BUFSZ);
  }

#if NICSTAR_RCQ
  node->rcq.rcq_head = (u32) rx_lg_skb[0]->data;
  node->rcq.rcq_base = (u32) rx_lg_skb[0]->data;
  /* Manual says we should check head vs tail here; but tail reg
     isn't updated until after we turn on RX path. Moved check to
     end of init_module... */
#endif

#if NICSTAR_LBUFCNT
  node->lbuf_cnt = NUM_LG_BUFS - 1;
#endif
  for (i = 0; i < NUM_SM_BUFS; i+=2) {
    push_rxbufs(node, 0, 
		(u32)(node->sm_bufs + (i*SM_BUFSZ) + 4),
		(u32)(node->sm_bufs + (i*SM_BUFSZ) + 4),
		(u32)(node->sm_bufs + ((i+1)*SM_BUFSZ) + 4),
		(u32)(node->sm_bufs + ((i+1)*SM_BUFSZ) + 4));
  }

  for (i = 0; i < NUM_LG_BUFS; i+=2) {
    push_rxbufs(node, 1, 
		(u32)(node->rx_lg_skb[i]->data),
		(u32)(node->rx_lg_skb[i]->data),
		(u32)(node->rx_lg_skb[i+1]->data),
		(u32)(node->rx_lg_skb[i+1]->data));
  }

  skb_queue_head_init(&node->rx_skb_queue);
  for (i = 0; i < NUM_RX_SKB; i++) {
    struct sk_buff *skb;
    skb = alloc_skb(
		    SM_BUFSZ > MAX_SDU_BUFS * sizeof(struct iovec) ?
		    SM_BUFSZ :  MAX_SDU_BUFS * sizeof(struct iovec),
		    GFP_KERNEL);
    if (!skb) {
      printk("nicstar%d: Can't allocate Rx skb!\n",node->index);
      return -ENOMEM;
    }
    skb_queue_tail(&node->rx_skb_queue,skb);
  }
      

  /* Allocate seg chan queue */
  /* XXX What about 64-bit pointers? */
  i = create_scq(&(node->scq),SCQ_SIZE);
  node->scq->id = SCQFULL_MAGIC;
  if (i != 0) { return i; }

  printk("nicstar%d: SCQ at 0x%x.\n",node->index,(u32)node->scq->base);

  write_sram(node, 4, NICSTAR_VBR_SCD0,
	     (u32)node->scq->base,      /* SCQ base */
	     (u32)0x0,            /* tail */
	     (u32)0xffffffff,     /* AAL5 CRC */
	     (u32)0x0);           /* rsvd */
  node->scq->scd = NICSTAR_VBR_SCD0;

  /* Install TST */

  /* Create host transmit schedule table */
  if ((((u32) node->tx_statq - (u32) node->tx_statq_orig) - (4 * TST_SIZE)) < 1) {
    printk ("Need to move host TST to later %d\n",4*TST_SIZE);
    node->host_tst = (u32 *) ((u32) node->tx_statq_orig - 8 - (4 * TST_SIZE));
    printk ("New host_tst %x\n",(u32)node->host_tst);
  }
  else {
    node->host_tst = (u32 *) node->tx_statq_orig;
  }
  
  node->available_slots = TST_SIZE;

  memset((caddr_t) node->host_tst, 0, 4 * TST_SIZE);

#ifdef NICSTAR_CBR
  for (i = NICSTAR_TST_REGION; i < NICSTAR_TST_REGION + TST_SIZE; i += 4) {
    write_sram(node, 4, i,
	       (u32) TSTE_VBR, /* Tx VBR SCD */
	       (u32) TSTE_VBR, /* Tx VBR SCD */
	       (u32) TSTE_VBR, /* Tx VBR SCD */
	       (u32) TSTE_VBR ); /* Tx VBR SCD */
  }

  write_sram(node, 1, NICSTAR_TST_REGION + TST_SIZE,
	     (u32)0x60000000 | NICSTAR_TST_REGION, /* Jump to start */
	     0x0, 0x0, 0x0);

#else
  write_sram(node, 2, NICSTAR_TST_REGION + TST_SIZE,
	     (u32)0x40000000, /* Tx VBR SCD */
	     /* (u32)0x20000000 | NICSTAR_SCD_REGION, *//* Tx SCD */
	     (u32)0x60000000 | NICSTAR_TST_REGION, /* Jump to start */
	     0x0, 0x0);
#endif

  writel((u32)(NICSTAR_TST_REGION << 2), node->membase + TSTB);

#if NICSTAR_RCQ
  /* Turn on everything */
  writel(0x21801c38 | LG_BUFMSK | CFG_VPIVCI | 0x200 | 0x800, node->membase+CFG);
  /* | 0x800 turns on raw cell interrupts */
  /* | 0x200 for raw cell receive */
  /* | 0x8000 to receive cells that don't map via RCT (not there/closed) */
#else
  writel(0x21801c38 | LG_BUFMSK | CFG_VPIVCI, node->membase+CFG); /* Turn on everything */
#endif

  while(CMD_BUSY(node));
#if NICSTAR_RCQ
  printk("nicstar%d: Checking raw cell queue pointer...\n",node->index);
  if (node->rcq.rcq_head != readl(node->membase + RAWCT)) {
    printk("nicstar%d: Raw cell head %x tail %x don't match\n",
	   node->index,(u32)node->rcq.rcq_head, readl(node->membase + RAWCT));
  }
#endif
  return 0;
}

static void nicstar_interrupt(int irq, void *dev_id, struct pt_regs *regs) {
  nicstar_devp node = (nicstar_devp)dev_id;
  u32 stat, wr_stat;

  PRINTK("nicstar: interrupt on %d.\n",irq);

  wr_stat = 0x0;
  stat = readl(node->membase + STAT);
  PRINTK("nicstar%d: Interrupt: STAT 0x%x.\n",node->index, stat);

  if (stat & 0x8000) {
    wr_stat |= 0x8000;
    PRINTK("nicstar%d: TSI written to Tx status queue.\n",node->index);
    process_tx_statq(node);
  }
  if (stat & 0x4000) {
    wr_stat |= 0x4000;
    printk("nicstar%d: Incomplete CS-PDU transmitted.\n",node->index);
  }

  if (stat & 0x800) {
    wr_stat |= 0x800;
    PRINTK("nicstar%d: Timer overflow.\n",node->index);
  }
  if (stat & 0x100) {
    printk("nicstar%d: Small buffer queue full (this is OK).\n",node->index);
  }
  if (stat & 0x80) {
    printk("nicstar%d: Large buffer queue full (this is OK).\n",node->index);
  }
  if (stat & 0x40) {
    printk("nicstar%d: Rx status queue full.\n",node->index);
    wr_stat |= 0x40;
    process_rx_statq(node); 
  }
  if (stat & 0x20) {
    /* wr_stat |= 0x20; */
    writel(0x20, node->membase + STAT); /* XXX mdw ack immediately */
    PRINTK("nicstar%d: End of PDU received.\n", node->index);
    process_rx_statq(node);
  }
  if (stat & 0x10) {
    wr_stat |= 0x10;
    printk("nicstar%d: Raw cell received.\n",node->index);
#ifdef NICSTAR_RCQ
    process_rx_rawcellq(node);
#endif
  }
  if (stat & 0x8) {
    printk("nicstar%d: Small buffer queue empty, disabling interrupt.\n",node->index);
    writel((readl(node->membase + CFG) & ~0x01000000), node->membase + CFG); 
    wr_stat |= 0x8;
  }
  if (stat & 0x4) {
    printk("nicstar%d: Large buffer queue empty, disabling interrupt.\n",node->index);
    writel((readl(node->membase + CFG) & ~0x01000000), node->membase + CFG); 
    wr_stat |= 0x4;
  }
  if (stat & 0x2) {
    printk("nicstar%d: Rx status queue almost full.\n",node->index);
    wr_stat |= 0x2;
    /* XXX Turn off interrupt */
    writel((readl(node->membase + CFG) & ~0x00000400), node->membase + CFG); 
    process_rx_statq(node); 
  }

  if (wr_stat != 0x0) {
    writel(wr_stat, node->membase + STAT);
  }

  return;
}


static void process_tx_statq(nicstar_devp node) {

  volatile nicstar_tsi *next;
  nicstar_scq *scq;
  u32 id;

  /* Spin until something arrives. If we get a timer-rollover and a
   * TSI simultaneously this could be dangerous; disabling timer 
   * interrupt will prevent that.
   */
  next = node->tx_statq_next;
  while (next->timestamp & 0x80000000) ;

  while (!(node->tx_statq_next->timestamp & 0x80000000)) {
    PRINTK("nicstar%d: Tx statq entry 0x%x 0x%x\n",
	   node->index,
	   node->tx_statq_next->status,
	   node->tx_statq_next->timestamp);

    PRINTK ("Processing entry at %x with val %x\n",(u32) node->tx_statq_next, node->tx_statq_next->status);

#if 0 /* original method */
    if ((node->tx_statq_next->status & 0xffff0000) == SCQFULL_MAGIC) {
      drain_scq(node, node->scq, node->tx_statq_next->status & 0xffff);
      /* SCQ drained, wake 'er up */
      node->scq->full = 0; 
      wake_up_interruptible(&node->scq->scqfull_waitq); 
    }
#else
    id = (node->tx_statq_next->status & 0xc0000000) >> 16;
    switch (id) {
      case (SCQFULL_MAGIC >> 16):
	scq = node->scq;
	break;
#if NICSTAR_CBR
      case (CBRSCQFULL_MAGIC_CLOSE >> 16):
	scq = node->vci_map[(node->tx_statq_next->status & 0x3fff0000)>>16].scq;
	scq->closing = 1;
        break;
      case (CBRSCQFULL_MAGIC >> 16):
	scq = node->vci_map[(node->tx_statq_next->status & 0x3fff0000)>>16].scq;
        break;
#endif
      default:
	scq = node->scq;
    }
    PRINTK ("Draining SCQ %d at %x\n",id,(u32)scq);
    drain_scq(node, scq, node->tx_statq_next->status & 0xffff);
    /* SCQ drained, wake 'er up */
    scq->full = 0; 
    wake_up_interruptible(&scq->scqfull_waitq); 
#endif

    node->tx_statq_next->timestamp |= 0x80000000;
    
    if (node->tx_statq_next == node->tx_statq_last)
      node->tx_statq_next = node->tx_statq;
    else
      node->tx_statq_next++;
  }

  writel((u32)node->tx_statq_next - (u32)node->tx_statq,
	 node->membase + TSQH);

  return;
}

#if NICSTAR_RCQ
static void process_rx_rawcellq (nicstar_devp node) {
  u32 tail;
  u32 timer;
  u32 last;
  int count;

  tail = readl(node->membase + RAWCT);
  timer = readl(node->membase + TMR);
  count = 0;
  while (node->rcq.rcq_head != tail - RAWCELLSZ) {
    if (count > 3) {
      printk ("nicstar OOOPS. Count > 3\n");
      return;
    }
    /* service head */
    PRINTK ("nicstar: dequeueing 0x%x to 0x%x\n",(u32)node->rcq.rcq_head,
	    readl(node->membase + RAWCT));
    dequeue_rawcell((u32 *) node->rcq.rcq_head);
    node->rcq.rcq_head += RAWCELLSZ;
    last = node->rcq.rcq_base + LG_BUFSZ - RAWCELLSZ;
    PRINTK ("Last %x tail %x rawct %x\n",(u32)last,(u32)tail,readl(node->membase + RAWCT));
    if (node->rcq.rcq_head == (node->rcq.rcq_base + LG_BUFSZ - RAWCELLSZ)) {
      node->rcq.rcq_head = *((u32 *) node->rcq.rcq_head);
      free_rx_buf(node, 1, node->rcq.rcq_base);
      node->rcq.rcq_base = node->rcq.rcq_head;
      if (node->rcq.rcq_base == tail) return; /* watch out for jumps */
    }
    count++;
  }
  /* YYY Might be able to avoid this by polling the cell header;
     requires initializing the header to a known value. Also
     assumes that we can find such a known value */
  while (((readl(node->membase+TMR)) - timer) < 30);
  /* service one cell */
  dequeue_rawcell((u32 *) node->rcq.rcq_head);
  node->rcq.rcq_head += RAWCELLSZ;
  if (node->rcq.rcq_head == (node->rcq.rcq_base + LG_BUFSZ - RAWCELLSZ)) {
    node->rcq.rcq_head = *((u32 *) node->rcq.rcq_head);
    free_rx_buf(node, 1, node->rcq.rcq_base);
    node->rcq.rcq_base = node->rcq.rcq_head;
  }
}

static void dequeue_rawcell (u32 *cell) {
  u32 header;
  u32 *payload;
  int i;

  header = *cell;
  payload = cell + 4;
  printk("nicstar: *Raw cell %x arrival hdr=%x pay=%x*\n",
	 (u32)cell,header,(u32)payload);
  printk("nicstar: Header contents VCI=%d Last=%d\n",
	 header >> 4, (header & 0x2) >> 1);
  if (header & 0x2) {
    printk("nicstar: PDU length %d\n",
	   ((payload[10] & 0xff000000) >> 24) +
	   (256 * ((payload[10] & 0x00ff0000) >> 16)));
  }
  for (i = 0; i < 12; i++) {
    printk ("    %d    0x%x\n",i,(u32)payload[i]);
  }
}
#endif /* NICSTAR_RCQ */

static void process_rx_statq(nicstar_devp node) {
  u32 prev;

  /* XXX Problem here: The 77201 can interrupt us (and set EPDU
   * bits in STAT) before the Rx statq entry arrives in memory.
   * The interrupt could be raised for multiple incoming PDUs,
   * and we could miss seeing those entries that are slow to 
   * arrive in memory at the end of this loop. We'll see the PDU
   * on the next Rx EPDU interrupt, of course.
   */
  while (node->rx_statq_next->status & 0x80000000) {
    PRINTK("nicstar%d: Rx statq entry 0x%x 0x%x 0x%x 0x%x\n",
	   node->index,
	   node->rx_statq_next->vpi_vci,
	   node->rx_statq_next->buf_handle,
	   node->rx_statq_next->crc,
	   node->rx_statq_next->status);
    
    dequeue_rx(node);
    node->rx_statq_next->status = 0x0;

    if (node->rx_statq_next == node->rx_statq_last)
      node->rx_statq_next = node->rx_statq;
    else
      node->rx_statq_next++;

  }

  /* Update the RSQH to point to the last entry actually processed. */
  if (node->rx_statq_next == node->rx_statq) {
    prev = (u32)node->rx_statq_last;
  } else {
    prev = (u32)(node->rx_statq_next - 1);
  }
  writel((u32)prev - (u32)node->rx_statq,
	 node->membase + RSQH);

  PRINTK("RSQ: 0x%x, next 0x%x, diff 0x%x, tail 0x%x\n",
	 (u32)node->rx_statq,
	 (u32)node->rx_statq_next,
	 (u32)node->rx_statq_next - (u32)node->rx_statq,
	 readl(node->membase + RSQT));

  PRINTK("nicstar%d: Status reads 0x%x.\n", node->index, 
    readl(node->membase + STAT));

  return;
}

static void push_rxbufs(nicstar_devp node, int lg, 
			u32 handle1, u32 addr1, u32 handle2, u32 addr2) {

  unsigned long flags;
  int bc;


  if (lg) {
    bc = (readl(node->membase + STAT) & 0x00ff0000) >> 16;
  } else {
    bc = (readl(node->membase + STAT) & 0xff000000) >> 24;
  }
  if (bc >= 254) return; 
    
  save_flags(flags); cli();
  while (CMD_BUSY(node)) ;
  
  writel(handle1, node->membase + DR0);
  writel(addr1, node->membase + DR1);
  writel(handle2, node->membase + DR2);
  writel(addr2, node->membase + DR3);

  writel(0x60000000 | (lg ? 0x01 : 0x00), node->membase + CMD);
  restore_flags(flags);
  return;
}

static void open_rx_connection(nicstar_devp node,
			       int index,
			       u32 status) {
#if 0
  unsigned long flags;
#endif

  u32 sram_addr = index * (sizeof(nicstar_rcte)/sizeof(u32));

  write_sram(node, 1, sram_addr, status | 0x80000,
	     0x0, 0x0, 0x0);

#if 0
  /* This is actually unnecessary, I believe, if we add the
     open connection bit to the SRAM write. Alternately, if
     we initialize the AAL during init_module, we can just 
     use the open connection command... (SWD) */
  save_flags(flags); cli();
  while (CMD_BUSY(node)) ;
  writel(0x20080000 | (sram_addr << 2), node->membase + CMD);
  restore_flags(flags);
#endif
}

static void close_rx_connection(nicstar_devp node,
				int index) {
  unsigned long flags;
  u32 sram_addr = index * (sizeof(nicstar_rcte)/sizeof(u32));

  save_flags(flags); cli();
  while (CMD_BUSY(node)) ;
  writel(0x20000000 | (sram_addr << 2), node->membase + CMD);
  restore_flags(flags);
}

static void push_scq_entry(nicstar_devp node, struct nicstar_scq *scq,
			   nicstar_tbd *entry,
			   int xmit_now) {

  unsigned long flags;
  nicstar_tbd magic_bullet = {0xa0000000, scq->id, 0x0, 0x0};
  static int delayed_bullet = 0;

  PRINTK("tail 0x%x next 0x%x\n",
   (u32) scq->tail, (u32) scq->next);
  PRINTK("CFG: 0x%x\n",readl(node->membase + CFG));

  if (scq->tail == scq->next) {
    PRINTK ("Going to sleep\n");
    /* Sleep until the tail moves */
    save_flags(flags); cli();
    scq->full = 1;
      
    current->timeout = jiffies + SCQFULL_TIMEOUT;
    interruptible_sleep_on(&scq->scqfull_waitq);
    restore_flags(flags);

    if (scq->full) {
      printk("nicstar%d: SCQ drain timed out.\n",node->index);
      scq->full = 0;
      /* XXX Just proceed here, although the SAR's probably hosed ... */
    }
  }

  *scq->next = *entry;
  
  if (scq->next == scq->last)
    scq->next = scq->base;
  else
    scq->next++;

  if (((scq->next - scq->base) % NUM_SCQ_TRAIL) == NUM_SCQ_TRAIL-1) {
    delayed_bullet = 1;
  }

  if (xmit_now) {
    if (delayed_bullet) {
      delayed_bullet = 0;
      magic_bullet.buf_addr |= (u32)(scq->next - scq->base);
      if (scq->tail == scq->next) {
	PRINTK ("Going to sleep!!\n");
	/* Sleep until the tail moves */
	save_flags(flags); cli();
	scq->full = 1;
      
	current->timeout = jiffies + SCQFULL_TIMEOUT;
	interruptible_sleep_on(&scq->scqfull_waitq);
	restore_flags(flags);

	if (scq->full) {
	  printk("nicstar%d: SCQ drain timed out.\n",node->index);
	  scq->full = 0;
	  /* XXX Just proceed here, although the SAR's probably hosed ... */
	}
      }
      *scq->next = magic_bullet;
      if (scq->next == scq->last)
        scq->next = scq->base;
      else
        scq->next++;
    }
    write_sram(node, 1, scq->scd,
	       (u32)(scq->next),
	       0x0, 0x0, 0x0);
  }

  return;
}

static int get_ci(nicstar_devp node,
		  struct atm_vcc *vcc,
		  short *vpi, int *vci) {
  nicstar_vcimap *mapval;

  PRINTK ("Reserving vci %d\n",(u32) *vci);
  /* Default, accept VPI 0 and VCI 0-4095. */
  if (*vpi == ATM_VPI_UNSPEC || *vci == ATM_VCI_UNSPEC) return -EINVAL;
  
  if (*vpi == ATM_VPI_ANY) *vpi = 0;
  if (*vpi >= NUM_VPI) return -EINVAL;
  
  if (*vci == ATM_VCI_ANY) {
    /* Find lowest VCI */
#ifndef ATM_013
    for (*vci = (*vpi) * NUM_VCI + ATM_NOT_RSV_VCI; *vci < (*vpi + 1) * NUM_VCI; (*vci)++) {
      mapval = &(node->vci_map[*vci]);
      if (vcc->qos.rxtp.traffic_class != ATM_NONE && mapval->rx) continue;
      if (vcc->qos.txtp.traffic_class != ATM_NONE && mapval->tx) continue;
      if (vcc->qos.rxtp.traffic_class != ATM_NONE) {
	mapval->rx = 1; mapval->rx_vcc = vcc;
      }
      if (vcc->qos.txtp.traffic_class != ATM_NONE) {
	mapval->tx = 1; mapval->tx_vcc = vcc;
      }
#else
      if (vcc->qos.rxtp.class != ATM_NONE && mapval->rx) continue;
      if (vcc->qos.txtp.class != ATM_NONE && mapval->tx) continue;
      if (vcc->qos.rxtp.class != ATM_NONE) {
	mapval->rx = 1; mapval->rx_vcc = vcc;
      }
      if (vcc->txtp.class != ATM_NONE) {
	mapval->tx = 1; mapval->tx_vcc = vcc;
      }
      break;
#endif /* ATM_013 */
    }
    if (*vci == NUM_VCI) return -EADDRINUSE;
    
  } else {
    /* Use this particular VCI */
    if (*vci >= NUM_VCI) return -EINVAL;
    mapval = &(node->vci_map[(*vpi << NUM_VCI_BITS) | *vci]);
#ifndef ATM_013
    if (vcc->qos.rxtp.traffic_class != ATM_NONE && mapval->rx) return -EADDRINUSE;
    if (vcc->qos.txtp.traffic_class != ATM_NONE && mapval->tx) return -EADDRINUSE;
    if (vcc->qos.rxtp.traffic_class != ATM_NONE) {
      mapval->rx = 1; mapval->rx_vcc = vcc;
    } 
    if (vcc->qos.txtp.traffic_class != ATM_NONE) {
      mapval->tx = 1; mapval->tx_vcc = vcc;
    }
#else
    if (vcc->rxtp.class != ATM_NONE && mapval->rx) return -EADDRINUSE;
    if (vcc->txtp.class != ATM_NONE && mapval->tx) return -EADDRINUSE;
    if (vcc->rxtp.class != ATM_NONE) {
      mapval->rx = 1; mapval->rx_vcc = vcc;
    } 
    if (vcc->txtp.class != ATM_NONE) {
      mapval->tx = 1; mapval->tx_vcc = vcc;
    }
#endif
  }

  return 0;
}

static void free_rx_buf(nicstar_devp node, int lg, u32 buf_addr) {
  static u32 sm_buf = 0x0, lg_buf = 0x0;

  if (lg) {
    /* Large buffer */
    if (lg_buf == 0x0) {
      lg_buf = buf_addr;
    } else {
      u32 buf;

#ifdef NICSTAR_LBUFCNT
      node->lbuf_cnt += 2;
#endif
      buf = buf_addr;
      push_rxbufs(node, 1, lg_buf, lg_buf, buf, buf);
      lg_buf = 0x0;
    }
  } else {
    /* Small buffer */
    if (sm_buf == 0x0) {
      sm_buf = buf_addr;
    } else {
      u32 buf;
      buf = buf_addr;
      push_rxbufs(node, 0, sm_buf, sm_buf, buf, buf);
      sm_buf = 0x0;
    }
  }
  return;
}

static void dequeue_rx(nicstar_devp node) {
  
  struct atm_vcc *vcc;
  struct sk_buff *skb;
  struct iovec *iov;
  nicstar_vcimap *mapval;
  volatile nicstar_rsqe *entry = (volatile nicstar_rsqe *)node->rx_statq_next;
  unsigned short aal5_len;
  unsigned short aal0_len;
  u32 *aal0_ptr;
  u32 vpi, vci;
  
  vpi = (entry->vpi_vci & 0x00ff0000) >> 16;
  vci = entry->vpi_vci & 0xffff;

  if ((vpi >= NUM_VPI) || (vci >= NUM_VCI)) {
    printk("nicstar%d: SDU received for out of range VC %d.%d.\n",
	   node->index, vpi, vci);
    return;
  }

  mapval = &(node->vci_map[(vpi << NUM_VCI_BITS) | vci]);
  if (!mapval->rx) {
    printk("nicstar%d: SDU received on inactive VC %d.%d.\n",
	   node->index, vpi, vci);
  }
  vcc = mapval->rx_vcc;

  if (!(skb = mapval->rx_skb)) {
    /* Starting new SKB */
    skb = mapval->rx_skb = skb_dequeue(&node->rx_skb_queue);
    if (!skb) {
      printk("nicstar%d: Out of Rx skb's!\n",node->index);
      goto drop_message;
    }
  }

  if (vcc->aal == ATM_AAL0) {
    iov = &(((struct iovec *)skb->data)[skb->atm.iovcnt]);
    aal0_ptr = (u32 *)entry->buf_handle;
    aal0_ptr--;
    *(aal0_ptr) =  /* "reconstruct" cell header */
      (entry->vpi_vci << 4) | ((entry->status & 0x4000) >> 14)|
      ((entry->status & 0x400) >> 10);
    PRINTK("Reconstructed cell header %x at %x, bufhandle %x\n",
	   *aal0_ptr,(u32)aal0_ptr,entry->buf_handle);
    iov->iov_base = (void *)(aal0_ptr);
    aal0_len = ATM_CELL_SIZE - 1;
    iov->iov_len = aal0_len;
    skb->len = aal0_len;
    skb->atm.iovcnt++;
    vcc->stats->rx++;
    skb->atm.timestamp = xtime;
    if ((u32) vcc->push != (u32) atm_push_raw) {
      skb = (struct sk_buff *) linearize_buffer(node,skb,vcc);
    }
    if (skb) {
      vcc->push(vcc, skb);
    }
    mapval->rx_skb = NULL;
    return;
  }

  iov = &(((struct iovec *)skb->data)[skb->atm.iovcnt]);
  iov->iov_base = (void *)entry->buf_handle;

  if (entry->status & 0x2000) {
    /* Last buffer, get AAL5 len */
    aal5_len = *(unsigned short *)((unsigned char *)entry->buf_handle + ((entry->status & 0x1ff)*48) - 6);
    aal5_len = ((aal5_len & 0xff) << 8) | ((aal5_len & 0xff00) >> 8);
    PRINTK ("Last buffer; AAL5 len is %d\n",aal5_len);

    iov->iov_len = aal5_len - skb->len;
    skb->len = aal5_len;
    skb->atm.iovcnt++;
#ifdef NICSTAR_LBUFCNT
    node->lbuf_cnt -= (skb->atm.iovcnt - 1);
#endif
    PRINTK("Received %d-buffer message %x\n",skb->atm.iovcnt,skb);
    vcc->stats->rx++;


    if ((u32) vcc->push != (u32) atm_push_raw) {
      skb = (struct sk_buff *) linearize_buffer(node,skb,vcc);
      PRINTK("rcv %d-buf len %d\n",skb->atm.iovcnt,skb->len);
    }
    PRINTK("after is rcv %d-buf len %d\n",skb->atm.iovcnt,skb->len);

    if (skb) {
      vcc->push(vcc, skb);
    }
    mapval->rx_skb = NULL;
  } else {
    iov->iov_len = (entry->status & 0x1ff) * 48;
    skb->len += iov->iov_len;
    skb->atm.iovcnt++;
  }
  return;

drop_message:
  printk("nicstar%d: drop_message returned in dequeue_rx.\n",node->index);
  vcc->stats->rx_drop++;

  if (entry->status & 0x1000) {
    /* Large buffer */
    free_rx_buf(node, 1, entry->buf_handle);
  } else {
    /* Small buffer */
    free_rx_buf(node, 0, entry->buf_handle);
  }

  return;
}

static void nicstar_rx_skb_destructor (struct sk_buff *skb) {
  if (skb->count > 1) return;
#if NICSTAR_RC_FLAG
  nicstar_free_rx_skb((struct atm_vcc *) skb->atm.recycle_buffer,skb);
#else
  nicstar_free_rx_skb((struct atm_vcc *) skb->atm.vcc,skb);
#endif
}

static void nicstar_free_rx_skb(struct atm_vcc *vcc, struct sk_buff *skb) {
  nicstar_devp node = vcc->dev->dev_data;
  struct iovec *iov;
  int i;
  u32 ptr_chk;

  if (!skb->atm.iovcnt) {
    for (i = 0; i < NUM_LG_BUFS; i++) {
      if (node->rx_lg_skb[i]->head == skb->head) break;
    }
    if (i < NUM_LG_BUFS) {
      /* this had better be a large buffer on the rebound (SWD) */
      /* restore data pointers */
      (node->rx_lg_skb[i])->data = (node->rx_lg_skb[i])->head + SM_BUF_DATALEN;
      (node->rx_lg_skb[i])->len = 0;
      (node->rx_lg_skb[i])->tail = (node->rx_lg_skb[i])->data;
      free_rx_buf(node, 1, (u32)((node->rx_lg_skb[i])->data));
      return;
    }
    else {
      /* i > NUM_LG_BUFS ==> small buffer, which has already */
      /*   been recycled; just requeue this skb after resetting */
      /*   tail */
      return;
    }
  }
  else { /* handle s-g buffer */
    iov = (struct iovec *)skb->data;
    for (i = 0; i < skb->atm.iovcnt; i++) {
      if (((unsigned char *)iov[i].iov_base >= (unsigned char *)node->sm_bufs) &&
	  ((unsigned char *)iov[i].iov_base < (unsigned char *)node->sm_bufs + SM_BUFSZ * NUM_SM_BUFS)) {
	ptr_chk = (u32)iov[i].iov_base == (u32)node->sm_bufs ? 48 :
	  ((u32)iov[i].iov_base - ((u32)node->sm_bufs + 4)) % SM_BUFSZ;
	if (ptr_chk) {
	  iov[0].iov_base += 4;
#ifdef NICSTAR_PARANOID
	  if (ptr_chk != 48) {
	    printk ("nicstar%d: Misaligned buffer pointer (offset %d)!\n",
		    node->index,ptr_chk);
	  }
	  ptr_chk = (u32)iov[i].iov_base == (u32)node->sm_bufs ? 48 :
	    ((u32)iov[i].iov_base - ((u32)node->sm_bufs + 4)) % SM_BUFSZ;
	  if (ptr_chk) {
	    printk("nicstar%d: Misaligned pointer STILL NOT FIXED!!! %x %d\n",
		   node->index,(u32)iov[i].iov_base,ptr_chk);
	  }
#endif
	}
	free_rx_buf(node, 0, (u32)(iov[i].iov_base));
      } else {
	/* Large buffer */
	free_rx_buf(node, 1, (u32)(iov[i].iov_base));
      }
    }
  }

  skb->atm.iovcnt = 0;
  skb->len = 0;
  skb_queue_tail(&node->rx_skb_queue, skb);

  return;
}
    
static void drain_scq(nicstar_devp node, struct nicstar_scq *scq, int index) {
  nicstar_tbd *last = scq->base + index;
  int sindex = scq->tail - scq->base;
  struct sk_buff *skb;
  struct atm_vcc *vcc;

  while (scq->tail != last) {
    if ((skb = scq->scq_shadow[sindex].skb)) {
      vcc = skb->atm.vcc;
      if (skb->free == 2) printk("drain_scq: Passed skb 0x%x with free == 2.\n", (u32) skb);
      if (vcc->pop) {
        vcc->pop(vcc, skb);
      } else {
        dev_kfree_skb(skb, FREE_WRITE);
      }
      scq->scq_shadow[sindex].skb = 0;
    }
    
    if (scq->tail == scq->last) {
      scq->tail = scq->base; sindex = 0;
    } else {
      scq->tail++; sindex++;
    }
  }


#ifdef NICSTAR_CBR
  if (scq->closing) {
    close_cbr_final(node,node->vci_map[(scq->id >> 16) & 0xfff].tx_vcc);
  }
#endif
}

#ifdef NICSTAR_CBR
static void alloc_tx(nicstar_devp node, struct atm_vcc *vcc) {
  int i, j;
  int slots;
  int spacing;
  int vci;

  vci = (vcc->vpi << NUM_VCI_BITS) | vcc->vci;
  create_scq(&((node->vci_map[vci]).scq),CBRSCQ_SIZE);
  (node->vci_map[vci].scq)->id = CBRSCQFULL_MAGIC | (vci << 16);
  PRINTK ("nicstar%d: Allocating CBR SCQ at %x\n",node->index,(u32) ((node->vci_map[vci]).scq));
  PRINTK ("nicstar%d: Allocating transmission bandwidth\n",node->index);
#ifndef ATM_013
  printk ("in pcr: %d/%d/%d\n",vcc->qos.txtp.min_pcr,
	    vcc->qos.txtp.max_pcr,node->max_pcr);
#else
  printk ("in pcr: %d/%d/%d\n",vcc->txtp.min_pcr,
	    vcc->txtp.max_pcr,node->max_pcr);
#endif
  (node->vci_map[vci].scq)->scd = 0;  /* initialize SCD pointer */
#ifndef ATM_013
  slots = (u32) (((float) vcc->qos.txtp.max_pcr / (float) node->max_pcr) * (float) TST_SIZE) + 1;
#else
  slots = (u32) (((float) vcc->txtp.max_pcr / (float) node->max_pcr) * (float) TST_SIZE) + 1;
#endif
  printk ("out: %d slots\n",slots);
  if (slots > node->available_slots) {
    printk("Reducing requested BW from %d to %d slots -- full up\n",
	   slots,node->available_slots);
    slots = node->available_slots;
  }
  node->available_slots -= slots;
  spacing = TST_SIZE / slots;
  /* sort of randomly pick starting slot */
  i = readl(node->membase + TMR) % TST_SIZE;
  while (slots > 0) {
    j = i;
    while (node->host_tst[j]) {
      j++;
      if (j >= TST_SIZE) {
        j -= TST_SIZE;
      }
    }
    if (!((node->vci_map[vci].scq)->scd)) {
      (node->vci_map[vci].scq)->scd = NICSTAR_SCD_REGION + 12 * j;
      write_sram(node, 4, NICSTAR_SCD_REGION + 12 * j,
		 (u32) ((nicstar_vcimap *)vcc->dev_data)->scq->base, /* base + head */
		 0x0, /* tail */
		 0xffffffff, /* aal5 crc */
		 0x0);
      /* should we also init next 8 words to 0? */
      PRINTK ("CBR SCQ at j %d sram %x ptr %x\n",j,
	      NICSTAR_SCD_REGION + 12 * j,
	      read_sram(node,NICSTAR_SCD_REGION + 12 * j));
    }
    node->host_tst[j] = (u32) vcc;
    write_sram(node, 1, NICSTAR_TST_REGION + j,
	       TSTE_CBR | (node->vci_map[vci].scq)->scd,
	       0x0, 0x0, 0x0);
    i += spacing;
    if (i >= TST_SIZE) {
      i -= TST_SIZE;
    }
    slots--;
  }
}

/* Closes down a CBR connection. */
static void close_cbr (nicstar_devp node, struct atm_vcc *vcc) {
  unsigned long flags;
  struct nicstar_scq *scq = ((nicstar_vcimap *)vcc->dev_data)->scq;

  PRINTK("Initial close down of CBR connection %x...",(u32) vcc);
  /* Push a TSR */
  if (scq->tail == scq->next) {
    /* Sleep until the tail moves */
    save_flags(flags); cli();
    scq->full = 1;
      
    current->timeout = jiffies + SCQFULL_TIMEOUT;
    interruptible_sleep_on(&scq->scqfull_waitq);
    restore_flags(flags);

    if (scq->full) {
      printk("nicstar%d: SCQ drain timed out.\n",node->index);
      scq->full = 0;
      /* XXX Just proceed here, although the SAR's probably hosed ... */
    }
  }

  PRINTK("Writing TSR\n");

  scq->next->status = 0xa0000000;
  scq->next->buf_addr = CBRSCQFULL_MAGIC_CLOSE | scq->id | (u32)(scq->next - scq->base);
  scq->next->ctl_len = 0;
  scq->next->cell_hdr = 0;
  
  if (scq->next == scq->last)
    scq->next = scq->base;
  else
    scq->next++;
  write_sram(node, 1, scq->scd,
	     (u32)(scq->next),
	     0x0, 0x0, 0x0);
}

static void close_cbr_final (nicstar_devp node, struct atm_vcc *vcc) {
  int i, count;

  for (i = 0; i < TST_SIZE; i++) {
    if (node->host_tst[i] == (u32) vcc) {
      node->host_tst[i] = 0;
      node->available_slots += 1;
      write_sram(node,1, NICSTAR_TST_REGION + i,
		 TSTE_VBR, 0x0, 0x0, 0x0);
    }
  }
  count = 0;
  for (i = 0; i < TST_SIZE; i++) {
    if (read_sram(node,NICSTAR_TST_REGION + i) != TSTE_VBR) {
      count ++;
    }
  }
  PRINTK ("non-VBR Count %d avail slots %d\n",count, node->available_slots);
  /* free up the CBR SCQ */
  kfree(((nicstar_vcimap *)vcc->dev_data)->scq->orig);
  kfree(((nicstar_vcimap *)vcc->dev_data)->scq);
}
#endif /* NICSTAR_CBR */

/* General function for creating transmission SCQs */
static int create_scq (struct nicstar_scq **scq, int size) {
  (*scq) = kmalloc(sizeof(struct nicstar_scq), GFP_KERNEL);
  (*scq)->orig = kmalloc(2*size, GFP_KERNEL);
  if (!(*scq)->orig) {
    return -ENOMEM;
  }
  (*scq)->base = (nicstar_tbd *)(((u32)(*scq)->orig + (size - 1)) & ~(size - 1));
  (*scq)->next = (*scq)->base;
  (*scq)->last = (*scq)->base + ((size / TBD_SIZE) - 1);
  (*scq)->tail = (*scq)->last; /* XXX mdw scq_next */
  (*scq)->full = 0;
  (*scq)->scqfull_waitq = NULL;
  (*scq)->closing = 0;
  memset (((*scq)->scq_shadow), 0,
	  (SCQ_ENTRIES) * (sizeof(struct nicstar_scq_shadow)));
  return 0;
}

static u32 linearize_buffer (struct nicstar_dev *node, struct sk_buff *skb, struct atm_vcc *vcc) {
  struct iovec *iov = (struct iovec *)skb->data;
  struct sk_buff *skb_new;

  if (skb->atm.iovcnt > 2) {     /* huge buffer */
    int el, cnt;
    unsigned char *p;

    /* copy data to new skb (brute force), only for huge SDUs */
    skb_new = alloc_skb(skb->len,GFP_ATOMIC);
    if (!skb_new) {
      printk("nicstar%d: Can't get skbuff for s-g ip receive of huge buffer (%d bytes)\n",node->index,(u32)skb->len);
      nicstar_free_rx_skb(vcc, skb);
      return 0;
    }
    p = skb_new->data;
    el = skb->len;
    skb_put(skb_new,skb->len);
    skb_new->free = 1;
    for (cnt = 0; (cnt < skb->atm.iovcnt) && el; cnt++) {
      memcpy(p, iov->iov_base,
	     (iov->iov_len > el) ? el : iov->iov_len);
      p += iov->iov_len;
      el -= (iov->iov_len > el)?el:iov->iov_len;
      iov++;
    }
    nicstar_free_rx_skb(vcc, skb);
    return (u32) skb_new;
  }
  else if (skb->atm.iovcnt == 2) {
    /* simply copy small buffer to free space at start of large */
    int i;

    /* which large buffer are we? */
    iov++;
    for (i = 0; i < NUM_LG_BUFS ; i++) {
      if ((node->rx_lg_skb[i])->data == iov->iov_base) {
	break;
      }
    }
    if (i >= NUM_LG_BUFS) {
      /* should probably copy instead */
      printk("nicstar%d: Corresponding large buffer not found! dropping packet. i %d vcc %p skb %p data %p\n",
	     node->index, i,vcc,skb,skb->data);
      nicstar_free_rx_skb(vcc,skb);
      return 0;
    }
    skb_put(node->rx_lg_skb[i],iov->iov_len);
    iov--;
    skb_push(node->rx_lg_skb[i],SM_BUF_DATALEN);
    memcpy((node->rx_lg_skb[i])->data,iov->iov_base,SM_BUF_DATALEN);
    skb->atm.iovcnt--;
#ifdef NICSTAR_LBUFCNT
    PRINTK("2-buffer receive (IP?) len %d old %x new %x node %x cnt %d vcc %x\n",skb->len,skb,node->rx_lg_skb[i],node,node->lbuf_cnt,vcc);
#endif
    nicstar_free_rx_skb(vcc,skb);
    skb_new = skb_clone(node->rx_lg_skb[i], GFP_ATOMIC);
#ifdef NICSTAR_RC_FLAG
    skb_new->atm.recycle_buffer = (void *) vcc;
#endif
    skb_new->destructor = nicstar_rx_skb_destructor;
    return (u32) skb_new;
  }
  else {
    caddr_t p = iov->iov_base; /* make a copy of pointer to data */

    /* put small buffer contents in skb->data */
    memcpy(skb->data,p,skb->len);
    free_rx_buf(node,0,(u32)p);
    skb->atm.iovcnt = 0;
    skb->tail = skb->data + skb->len;
    skb_new = skb_clone(skb,GFP_ATOMIC);
    skb_new->destructor = nicstar_rx_skb_destructor;
#ifdef NICSTAR_RC_FLAG
    skb_new->atm.recycle_buffer = (void *) vcc;
#endif
    return (u32) skb_new;
  }
}
