/*
 *  linux/arch/i386/kernel/signal.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 */

#include <linux/config.h>

#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/kernel.h>
#include <linux/signal.h>
#include <linux/errno.h>
#include <linux/wait.h>
#include <linux/ptrace.h>
#include <linux/unistd.h>

#include <asm/segment.h>
#include <asm/vm86.h>

#include "../include/exclpage.h"
#include "../include/debug.h"

/* we implement this  interface: */
#include "../include/dispatch.h"

#define _S(nr) (1<<((nr)-1))

#define _BLOCKABLE (~(_S(SIGKILL) | _S(SIGSTOP)))

asmlinkage int sys_waitpid(pid_t pid,unsigned long * stat_addr, int options);


static struct _fpstate * save_i387(struct _fpstate * buf)
{
  return NULL;			/* XXX */
}

static void restore_i387(struct _fpstate *buf)
{
  /* XXX */
}
	

/* sigsuspend(2) and sigreturn(2) system calls (from
   arch/i386/kernel/signal.c) */

/*
 * atomically swap in the new signal mask, and wait for a signal.
 */
asmlinkage int sys_sigsuspend(int restart, unsigned long oldmask, unsigned long set)
{
	unsigned long mask;
	struct pt_regs * regs = &current->tss.ex_page->regs;

	mask = current->blocked;
	current->blocked = set & _BLOCKABLE;
	regs->eax = -EINTR;
	while (1) {
		current->state = TASK_INTERRUPTIBLE;
		schedule();
		if (do_signal(mask,regs))
			return -EINTR;
	}
}

/*
 * This sets regs->esp even though we don't actually use sigstacks yet..
 */
asmlinkage int sys_sigreturn(unsigned long __unused)
{
#define COPY(x) regs->x = context.x
	struct sigcontext_struct context;
	struct pt_regs * regs;

	regs = &current->tss.ex_page->regs;
	if (verify_area(VERIFY_READ, (void *) regs->esp, sizeof(context)))
		goto badframe;
	memcpy_fromfs(&context,(void *) regs->esp, sizeof(context));
	current->blocked = context.oldmask & _BLOCKABLE;
#if 0
	if (context.cs != 0x1b)
	  {
	    printk("sys_sigreturn: cs != 0x1b in %s (%x)\n", current->comm, 
		   current->tss.user_thread_id.lh.low);
	    enter_kdebug("sigret: cs != 0x1b");
	  }
#endif
	COPY(cs);
	COPY(eip);
	COPY(ecx); COPY(edx);
	COPY(ebx);
	COPY(esp); COPY(ebp);
	COPY(edi); COPY(esi);
	regs->eflags &= ~0x40DD5;
	regs->eflags |= context.eflags & 0x40DD5;
	regs->orig_eax = -1;		/* disable syscall checks */
	if (context.fpstate) {
		struct _fpstate * buf = context.fpstate;
		if (verify_area(VERIFY_READ, buf, sizeof(*buf)))
			goto badframe;
		restore_i387(buf);
	}
	return context.eax;
badframe:
	do_exit(SIGSEGV);
}


/* Set up a signal frame... Make the stack look the way iBCS2 expects
   it to look.  (adapted from arch/i386/kernel/signal.c) */
static void setup_frame(struct sigaction * sa,
			struct pt_regs * regs, int signr,
			unsigned long oldmask)
{
  unsigned long * frame;

  frame = (unsigned long *) regs->esp;
#if 0				/* XXX sa_restorer ??? */
  if (regs->ss != USER_DS && sa->sa_restorer)
    frame = (unsigned long *) sa->sa_restorer;
#endif
  frame -= 64;
  if (verify_area(VERIFY_WRITE,frame,64*4))
    do_exit(SIGSEGV);

  /* set up the "normal" stack seen by the signal handler (iBCS2);
     this mirrors the layout defined by struct sigcontext_struct
     (<asm/sigcontext.h>) */
#define __CODE ((unsigned long)(frame+24))
#define CODE(x) ((unsigned long *) ((x)+__CODE))
  put_user(__CODE,frame);
  if (current->exec_domain && current->exec_domain->signal_invmap)
    put_user(current->exec_domain->signal_invmap[signr], frame+1);
  else
    put_user(signr, frame+1);
  put_user(0, frame+2);		/* gs */
  put_user(0, frame+3);		/* fs */
  put_user(0, frame+4);		/* es */
  put_user(0, frame+5);		/* ds */
  put_user(regs->edi, frame+6);
  put_user(regs->esi, frame+7);
  put_user(regs->ebp, frame+8);
  put_user(regs->esp, frame+9);
  put_user(regs->ebx, frame+10);
  put_user(regs->edx, frame+11);
  put_user(regs->ecx, frame+12);
  put_user(regs->eax, frame+13);
  put_user(current->tss.ex_page->exception_number, frame+14); /* trap nr */
  put_user(current->tss.ex_page->error_code, frame+15);	/* error code */
  put_user(regs->eip, frame+16);
#if 0
  if (regs->cs != 0x1b)
    {
      printk("setup_frame: cs != 0x1b in %s (%x)\n", current->comm, 
	     current->tss.user_thread_id.lh.low);
      enter_kdebug("setup_frame: cs != 0x1b");
    }
#endif
  put_user(regs->cs, frame+17);	/* cs */
  put_user(regs->eflags, frame+18);
  put_user(regs->esp, frame+19);
  put_user(0, frame+20);	/* ss */
  put_user(save_i387((struct _fpstate *)(frame+32)),frame+21);
/* non-iBCS2 extensions.. */
  put_user(oldmask, frame+22);
  put_user(current->tss.page_fault_addr, frame+23);

  /* set up the return code... */
  put_user(0x0000b858, CODE(0));	/* popl %eax ; movl $,%eax */
  put_user(0x80cd0000, CODE(4));	/* int $0x80 */
  put_user(__NR_sigreturn, CODE(2));
#undef __CODE
#undef CODE

  /* Set up registers for signal handler */
  regs->esp = (unsigned long) frame;
  regs->eip = (unsigned long) sa->sa_handler;
  regs->eflags &= ~TF_MASK;
}


/* OK, we're invoking a handler (from arch/i386/kernel/signal.c) */
static void handle_signal(unsigned long signr, struct sigaction *sa,
	unsigned long oldmask, struct pt_regs * regs)
{
#ifdef DEBUG
      printk("(%s)handle_signal(%lx): %s, ptregs: %p\n",
	     current->comm, signr, get_signal_name(signr), regs);
#endif
	/* are we from a system call? */
	if (regs->orig_eax >= 0) {
		/* If so, check system call restarting.. */
		switch (regs->eax) {
			case -ERESTARTNOHAND:
				regs->eax = -EINTR;
				break;

			case -ERESTARTSYS:
				if (!(sa->sa_flags & SA_RESTART)) {
					regs->eax = -EINTR;
					break;
				}
			/* fallthrough */
			case -ERESTARTNOINTR:
				regs->eax = regs->orig_eax;
				regs->eip -= 2;
		}
	}

	/* set up the stack frame */
	setup_frame(sa, regs, signr, oldmask);

	if (sa->sa_flags & SA_ONESHOT)
		sa->sa_handler = NULL;
	current->blocked |= sa->sa_mask;
}


/* Post a signal. (from arch/i386/kernel/signal.c) */
/*
 * Note that 'init' is a special process: it doesn't get signals it doesn't
 * want to handle. Thus you cannot kill init even with a SIGKILL even by
 * mistake.
 */
int do_signal(unsigned long oldmask, struct pt_regs * regs)
{
	unsigned long mask = ~current->blocked;
	unsigned long signr;
	struct sigaction * sa;
#ifdef DEBUG
      printk("(%s)do_signal: ptregs: %p\n",
	     current->comm, regs);
#endif

	while ((signr = current->signal & mask)) {
		/*
		 *	This stops gcc flipping out. Otherwise the assembler
		 *	including volatiles for the inline function to get
		 *	current combined with this gets it confused.
		 */
	        struct task_struct *t=current;
		__asm__("bsf %3,%1\n\t"
			"btrl %1,%0"
			:"=m" (t->signal),"=r" (signr)
			:"0" (t->signal), "1" (signr));
		sa = current->sig->action + signr;
		signr++;
		if ((current->flags & PF_PTRACED) && signr != SIGKILL) {
			current->exit_code = signr;
			current->state = TASK_STOPPED;
			notify_parent(current);
			schedule();
			if (!(signr = current->exit_code))
				continue;
			current->exit_code = 0;
			if (signr == SIGSTOP)
				continue;
			if (_S(signr) & current->blocked) {
				current->signal |= _S(signr);
				continue;
			}
			sa = current->sig->action + signr - 1;
		}
		if (sa->sa_handler == SIG_IGN) {
			if (signr != SIGCHLD)
				continue;
			/* check for SIGCHLD: it's special */
			while (sys_waitpid(-1,NULL,WNOHANG) > 0)
				/* nothing */;
			continue;
		}
		if (sa->sa_handler == SIG_DFL) {
			if (current->pid == 1)
				continue;
			switch (signr) {
			case SIGCONT: case SIGCHLD: case SIGWINCH:
				continue;

			case SIGSTOP: case SIGTSTP: case SIGTTIN: case SIGTTOU:
				if (current->flags & PF_PTRACED)
					continue;
				current->state = TASK_STOPPED;
				current->exit_code = signr;
				if (!(current->p_pptr->sig->action[SIGCHLD-1].sa_flags & 
						SA_NOCLDSTOP))
					notify_parent(current);
				schedule();
				continue;

			case SIGQUIT: case SIGILL: case SIGTRAP:
			case SIGABRT: case SIGFPE: case SIGSEGV:
				if (current->binfmt && current->binfmt->core_dump) {
					if (current->binfmt->core_dump(signr, regs))
						signr |= 0x80;
				}
				/* fall through */
			default:
				current->signal |= _S(signr & 0x7f);
				current->flags |= PF_SIGNALED;
				do_exit(signr);
			}
		}
		handle_signal(signr, sa, oldmask, regs);
		return 1;
	}

	/* Did we come from a system call? */
	if (regs->orig_eax >= 0) {
		/* Restart the system call - no handlers present */
		if (regs->eax == -ERESTARTNOHAND ||
		    regs->eax == -ERESTARTSYS ||
		    regs->eax == -ERESTARTNOINTR) {
			regs->eax = regs->orig_eax;
			regs->eip -= 2;
		}
	}
	return 0;
}
