#include <stdio.h>
#include <unistd.h>

#include <l4/syscalls.h>
#include <l4/ipc.h>

#include "init.h"
#include "config.h"
#include "exec.h"
#include "globals.h"
#include "memmap.h"

/* the interface we define */
#include <l4/server/rmgr.h>
#include "rmgr.h"

#define MGR_BUFSIZE 1024

#define MGR_STACKSIZE 1024
static char mgr_stack[MGR_STACKSIZE];
static void mgr(void) L4_NORETURN;

/*
 *  we better make this static because of limited stack size
 */
static char str_buf[MGR_BUFSIZE];

/* started from init() in startup.c */
void
rmgr(void)
{
  l4_threadid_t t, pa, pr;
  dword_t dummy;

  /* we (the main thread of this task) will serve as a pager for our
     children. */

  /* before that, we start threads for other services, e.g. resource
     management */
  t = myself;
  t.id.lthread = RMGR_LTHREAD_SUPER;
  pa = my_pager; pr = my_preempter;
  l4_thread_ex_regs(t, (dword_t) mgr, (dword_t)(mgr_stack + MGR_STACKSIZE), 
		    &pr, &pa, &dummy, &dummy, &dummy);

  /* now start serving the subtasks */
  pager();
}

/* resource manager */

/* defined here because of the lack of a better place */
quota_t quota[TASK_MAX];
owner_t __task[TASK_MAX];
owner_t __small[SMALL_MAX];

static void mgr(void)
{
  int a;
  l4_proto_t p;
  dword_t arg, d1, d2, e1, e2;
  void *desc;
  int err, unhandled, do_log;
  l4_threadid_t sender, partner, n, s;
  l4_msgdope_t result;
  l4_sched_param_t sched;

  struct {
	l4_fpage_t fp;
	l4_msgdope_t size_dope;
	l4_msgdope_t send_dope;
	dword_t      dw2[2];
	l4_strdope_t data;
  } msg;

  for (;;)
    {
      msg.size_dope = L4_IPC_DOPE(0, 1);
      msg.send_dope = L4_IPC_DOPE(0, 1);
      msg.data.rcv_size = MGR_BUFSIZE;
      msg.data.rcv_str  = (dword_t)str_buf;
      err = l4_i386_ipc_wait(&sender, &msg, &p.request, &arg, L4_IPC_NEVER, 
			     &result);
      
      while (!err)
	{
	  /* we received a request here */

	  do_log = ((debug_log_mask & (1L << p.proto.proto))
		    && (! (debug_log_types & 2) 
			|| ((debug_log_types >> 16) == p.proto.action)));

	  if (do_log && (debug_log_types & 1))
	    printf("RMGR: Log: rcv from %x:%x: [0x%x, 0x%x], r = 0x%x\n",
		   sender.lh.high, sender.lh.low, p.request, arg,
		   result.msgdope);
	  
	  /* who's the allocation unit? */
	  if (L4_IPC_MSG_DECEITED(result))
	    check(l4_nchief(sender, &partner) == L4_NC_INNER_CLAN);
	  else
	    partner = sender;

#if 0 /* currently, this assertion doesn't hold when using RMGR_TASK_CREATE */
	  assert(task_owner(partner.id.task) == myself.id.task);
#endif
	  /* handle the supervisor protocol */
	  
	  desc = L4_IPC_SHORT_MSG;
	  d1 = d2 = (dword_t) -1;

	  unhandled = 0;

          if (partner.id.task > O_MAX)
            {
              /* OOPS.. can't help this sender. */
	      unhandled = 1;
	      goto reply;
            }

	  switch (p.proto.proto)
	    {
	    case RMGR_RMGR:
	      switch (p.proto.action)
		{
		case RMGR_RMGR_PING:
		  d1 = 0; d2 = ~(p.proto.param);
		  break;
		default: 
		  unhandled = 1;
		}
	      break;
	    case RMGR_TASK:
	      switch (p.proto.action)
		{
		case RMGR_TASK_DELETE:
		  /* delete a task we've created on behalf of this user */

		case RMGR_TASK_GET:
		  /* transfer task create right for task no
		     p.proto.param to sender */
		  if (p.proto.param >= TASK_MAX)
		    break;
		  
		  if (p.proto.action == RMGR_TASK_GET)
		    {
		      if (!task_alloc(p.proto.param, partner.id.task))
			break;
		    }
		  else
		    {
		      if (task_owner(p.proto.param) != partner.id.task)
			break;
		    }
			  
		  n = sender;
		  n.id.task = p.proto.param;
		  n.id.chief = myself.id.task; /* XXX I hope the args
						  suffice for l4_task_new() */
		  n = l4_task_new(n, 
				  p.proto.action == RMGR_TASK_GET
				  ? sender.lh.low : myself.lh.low, 
				  0, 0, L4_NIL_ID);

		  if (l4_is_nil_id(n) || p.proto.action == RMGR_TASK_DELETE)
		    {
		      task_free(p.proto.param, partner.id.task);

		      if (l4_is_nil_id(n) && p.proto.action == RMGR_TASK_GET)
			break;	/* failed */
		    }
		  
		  d1 = 0;

		  break;

		case RMGR_TASK_SET_SMALL:
		case RMGR_TASK_SET_PRIO:
		  n.lh.low = arg;

 		  /* try to guess the high word of the thread id */
		  n.lh.high = partner.lh.high;
		  if (! task_equal(n, partner))
		    {
		      /* XXX only works for childs of partner */
		      n.id.chief = partner.id.task;
		      n.id.nest++;
		    }

		  s = L4_INVALID_ID;
		  l4_thread_schedule(n, (dword_t) -1, &s, &s, &sched);
		  if (sched.sched_param == 0xffffffff) /* error? */
		    {
		      /* XXX another try... we talk about a child of
                         sender */
		      if (task_equal(sender, partner))
			break;

		      n.id.chief = sender.id.task;
		      n.id.nest = sender.id.nest + 1;

		      s = L4_INVALID_ID;
		      l4_thread_schedule(n, (dword_t) -1, &s, &s, &sched);

		      if (sched.sched_param == 0xffffffff) /* error? */
			break;	/* give up. */
		    }

		  if (p.proto.action == RMGR_TASK_SET_SMALL)
		    {
		      if (! small_space_size || p.proto.param == 0
			  || p.proto.param >= 128/small_space_size
			  || ! small_alloc(p.proto.param, partner.id.task))
			break;

		      sched.sp.small = small_space_size 
			| (p.proto.param * small_space_size * 2);
		      s = L4_INVALID_ID;
		      l4_thread_schedule(n, sched, &s, &s, &sched);
		    }
		  else		/* p.proto.action == RMGR_TASK_SET_PRIO */
		    {
		      if (quota[partner.id.task].log_mcp < p.proto.param)
			break;

		      sched.sp.prio = p.proto.param;
		      s = L4_INVALID_ID;
		      l4_thread_schedule(n, sched, &s, &s, &sched);
		    }

		  d1 = 0;
		  break;

		case RMGR_TASK_CREATE:
		  /* create a task; p.proto.param = task number, arg = mcp,
		     *(dword_t*)str_buf = esp, 
		     *(dword_t*)(str_buf + 4) = eip,
		     *(l4_threadid_t*)(str_buf + 8) = pager */

		  if (result.msgdope != L4_IPC_DOPE(0, 1).msgdope)
		    break;	/* must have 8 bytes indirect string data */

		  if (p.proto.param >= TASK_MAX 
		      || l4_is_nil_id(* (l4_threadid_t*)(str_buf + 8))
		      || !task_alloc(p.proto.param, partner.id.task))
		    break;

		  n = myself;
		  n.id.chief = myself.id.task;
		  n.id.task = p.proto.param;
		  /* n.id.nest++; */

		  if (do_log)
		    printf("RMGR: Log: creating task=%x:%x, esp=%x, "
			   "eip=%x, pager=%x:%x\n",
			   n.lh.high, n.lh.low,
			   * (dword_t*)str_buf, * (dword_t*)(str_buf + 4),
			   ((l4_threadid_t*)(str_buf + 8))->lh.high,
			   ((l4_threadid_t*)(str_buf + 8))->lh.low);

		  /* XXX for now, use an L4-mcp of 0 */
		  n = l4_task_new(n, 0, * (dword_t*)str_buf, 
				  * (dword_t*)(str_buf + 4),
				  * (l4_threadid_t*)(str_buf + 8));

		  if (l4_is_nil_id(n))
		    {
		      task_free(p.proto.param, partner.id.task);
		      break;
		    }
		  
		  /* XXX for now, just reset all quotas to their
                     minimal value */
		  quota[p.proto.param].mem.max = 0;
		  quota[p.proto.param].himem.max = 0;
		  quota[p.proto.param].task.max = 0;
		  quota[p.proto.param].small.max = 0;
		  quota[p.proto.param].irq.max = 0;
		  quota[p.proto.param].log_mcp = 
		    arg > quota[partner.id.task].log_mcp 
		    ? quota[partner.id.task].log_mcp : arg;
;

		  d1 = n.lh.low;
		  d2 = n.lh.high;

		  break;

		case RMGR_TASK_GET_ID:
		  if (result.msgdope == L4_IPC_DOPE(0, 1).msgdope) {
		    /* printf("RMGR: got string: %s\n", str_buf); */

		    /*
		     *  fixme: configfile name should be ignored
		     */
		    for (a = 1;a < mb_info.mods_count;a++)
		      {
			char *ptr, *str;

			/*
			 *  fixme: comparison fails if commandline
			 *         parameter are given
			 */
			str = (char *)mb_mod[a].string;
			/* printf("RMGR: name: %s\n", str); */
			ptr = strrchr(str, '/');
			if (ptr)
			    ptr++; /* skip '/' */
			else
			    ptr = (char *)mb_mod[a].string;
			/* printf("RMGR: comparing: %s\n", ptr); */
			/* printf("             to: %s\n", str_buf); */
			if (strcmp(ptr, str_buf) == 0) {
			  /* printf("RMGR: found\n"); */
			  d1 = mb_mod_ids[a].lh.low;
			  d2 = mb_mod_ids[a].lh.high;
			  break;
			}
		      }
		  } else {
		    enter_kdebug("RMGR: no string");
		  }

		  break;

		default:
		  unhandled = 1;
		}
	      break;

	    case RMGR_IRQ:
	      switch (p.proto.action)
		{
		case RMGR_IRQ_GET:
		  if (p.proto.param < IRQ_MAX)
		    {
		      if (irq_alloc(p.proto.param, partner.id.task))
			{
			  /* the irq was free -- detach from it */
			  n = myself;
			  n.id.lthread = LTHREAD_NO_IRQ(p.proto.param);
			  err = l4_i386_ipc_call(n, L4_IPC_SHORT_MSG, 1, 0,
						 L4_IPC_SHORT_MSG, &e1, &e2,
						 L4_IPC_NEVER, &result);
			  check(err == 0);
			  
			  if (e1 == 0)
			    d1 = 0; /* success! */
			  else
			    /* fucked! (couldn't attach to irq number) */
			    irq_free(p.proto.param, partner.id.task);
			}
		    }
		  break;

		case RMGR_IRQ_FREE:
		  if (p.proto.param < IRQ_MAX
		      && irq_free(p.proto.param, partner.id.task))
		    {
		      n = myself;
		      n.id.lthread = LTHREAD_NO_IRQ(p.proto.param);
		      err = l4_i386_ipc_call(n, L4_IPC_SHORT_MSG, 0, 0,
					     L4_IPC_SHORT_MSG, &e1, &e2,
					     L4_IPC_NEVER, &result);
		      check(err == 0);

		      if (e1 == 0)
			d1 = 0;	/* success! */
		    }
		  break;

		default:
		  unhandled = 1;
		}
	      break;

	    default:
	      unhandled = 1;
	    }

	reply:
	  if (verbose && unhandled)
	    {
	      printf("RMGR: can't handle request=0x%x, data=0x%x "
		     "from thread=%x\n", p.request, arg, sender.lh.low);
	    }

	  
	  if ((do_log && (debug_log_types & 1))
	      || ((verbose || debug) && d1 == (dword_t)-1))
	    {
	      if (d1 != 0 && !do_log) /* error, and request not yet logged */
		printf("RMGR: Log: rcv from %x:%x: [0x%x, 0x%x], r = 0x%x\n",
		       sender.lh.high, sender.lh.low, p.request, arg,
		       result.msgdope);
		
	      printf("RMGR: Log: snd to   %x:%x: [0x%x, 0x%x]\n",
		     sender.lh.high, sender.lh.low, d1, d2);
	    }
	  
	  if ((do_log && (debug_log_types & 4))
	       || (debug && d1 == (dword_t)-1))
	    enter_kdebug("rmgr>");
	  
	  /* send reply and wait for next message */
	  msg.size_dope = L4_IPC_DOPE(0, 1);
	  msg.send_dope = L4_IPC_DOPE(0, 0);
	  msg.data.rcv_size = MGR_BUFSIZE;
	  msg.data.rcv_str  = (dword_t)str_buf;
	  err = l4_i386_ipc_reply_and_wait(sender, desc, d1, d2,
					   &sender, &msg, &p.request, &arg,
					   L4_IPC_NEVER, &result);
	}
    }
}

