/* -*- indented-text -*- */

%{
#include <unistd.h>

#include "quota.h"
#include "init.h"
#include "cfg.h"

#define yyparse __cfg_parse

static void yyerror(const char *s);

unsigned __cfg_task;
static unsigned ctask;
static int assert_mod = -1;

static unsigned c_max = -1, c_low = 0, c_high = -1, c_mask = 0xffffffff;

#ifndef min
#define min(x,y) ((x)<(y)?(x):(y))
#endif
#ifndef max
#define max(x,y) ((x)>(y)?(x):(y))
#endif

#define minset(x, y) ((x) = min((x),(y)))
#define maxset(x, y) ((x) = max((x),(y)))

%}

%union {
  char *string;
  unsigned number;
  struct {
    unsigned low, high;
  } interval;
}

%token TASK MODNAME CHILD IRQ MAX IN MASK MEMORY HIMEM LOGMCP BOOTMCP BOOTPRIO
%token BOOTWAIT RMGR SIGMA0 DEBUGFLAG VERBOSE LOG
%token SMALLSIZE SMALL BOOTSMALL MODULE
%token <string> UNSIGNED STRING
%type <number> number
%type <interval> setspec
%type <string> string

/* grammer rules follow */
%%

file	: rules
	;

rules	: rule rules
	|
	;

rule	: taskspec constraints modules	{ quota_t *q = & quota[ctask];
					  bootquota_t *b = 
					    & bootquota[ctask];
					  printf("RMGR: configured "
						 "task 0x%x: "
						 "[ m:%x,%x,%x "
						 "hm:%x,%x,%x\n      "
						 "t:%x,%x,%x "
						 "i:%x lmcp:%x "
						 "s:%x,%x,%x\n      "
						 "mcp:%x prio:%x "
						 "small:%x ]\n", 
						 ctask,
						 q->mem.low, q->mem.high,
						 q->mem.max, q->himem.low, 
						 q->himem.high, q->himem.max, 
						 q->task.low,
						 q->task.high, q->task.max,
						 q->irq.max, q->log_mcp,
						 q->small.low, q->small.high,
						 q->small.max,
						 b->mcp, b->prio, 
						 b->small_space);
					  assert_mod = -1; }
	| smallsizerule
	| flag
	;

smallsizerule : SMALLSIZE number	{ small_space_size = $2; }

flag	: BOOTWAIT			{ boot_wait++; }
	| VERBOSE			{ verbose++; }
	| DEBUGFLAG			{ debug++; }
	| LOG number number		{ debug_log_mask = $2;
					  debug_log_types = $3; }
	;

modules	: 
	| nextmod asserts		{ bootquota[ctask].mods++; }
	;

nextmod	: MODULE			{ assert_mod++; }
	;

taskspec : nexttask asserts
	| TASK RMGR			{ ctask = myself.id.task; }
	| TASK SIGMA0			{ ctask = my_pager.id.task; }
	| TASK number			{ ctask = $2; }
	;

nexttask: TASK				{ ctask = __cfg_task++;
					  assert_mod = ctask - 3; }

asserts : assert asserts
	|
	;

assert	: MODNAME string		{ char *n;
					  if (assert_mod < 1)
					    {
					      printf("RMGR: ERROR: no boot "
					             "module associated with "
						     "modname %s\n",
						     $2);
					      boot_errors++;
					      goto assert_end;
					    }
					  n = (char *)
					    mb_mod[assert_mod].string;
					  if (! n)
					    {
					      printf("RMGR: WARNING: cmdlines "
						     "unsupported, can't "
						     "verify modname %s\n",
						     $2);
					    }
					  else if (! strstr(n, $2))
					    {
					      printf("RMGR: ERROR: modname "
						     "%s doesn't match "
						     "cmdline %s\n",
						     $2, n);
					      boot_errors++;
					    }
					assert_end:
					  free($2);
					}
	;

constraints : constraint constraints
	| 
	;

constraint : CHILD childconstraints	{ maxset(quota[ctask].task.low, c_low);
					  minset(quota[ctask].task.high, c_high);
					  minset(quota[ctask].task.max, c_max);
                                          c_low = 0; c_high = -1; c_max = -1; }
	| MEMORY memoryconstraints	{ maxset(quota[ctask].mem.low, c_low);
					  minset(quota[ctask].mem.high, c_high);
					  minset(quota[ctask].mem.max, c_max);
                                          c_low = 0; c_high = -1; c_max = -1; }
	| HIMEM memoryconstraints	{ maxset(quota[ctask].himem.low, c_low);
					  minset(quota[ctask].himem.high, c_high);
					  minset(quota[ctask].himem.max, c_max);
                                          c_low = 0; c_high = -1; c_max = -1; }
	| IRQ irqconstraints		{ unsigned mask = 0, i;
					  for (i = max(0, c_low);
					       i <= min(15, c_max);
					       i++)
					    {
                                              mask |= 1L << i;
					    }
					  quota[ctask].irq.max &= 
					    c_mask & mask;
                                          c_low = 0; c_high = -1; c_max = -1;
					  c_mask = 0xffffffff; }
	| SMALL smallconstraints	{ maxset(quota[ctask].small.low, c_low);
					  minset(quota[ctask].small.high, c_high);
					  minset(quota[ctask].small.max, c_max);
                                          c_low = 0; c_high = -1; c_max = -1; }
	| LOGMCP number			{ quota[ctask].log_mcp = $2; }
	| BOOTMCP number		{ bootquota[ctask].mcp = $2; }
	| BOOTPRIO number		{ bootquota[ctask].prio = $2; }
	| BOOTSMALL number		{ bootquota[ctask].small_space
					    = $2; }
	;

childconstraints : numconstraints
	;

smallconstraints : numconstraints
	;

memoryconstraints : numconstraints
	;

irqconstraints : numconstraints
	| maskconstraints
	;

numconstraints : numconstraint numconstraints
	| numconstraint
	;

numconstraint : MAX number		{ c_max = min(c_max, $2); }
	| IN setspec			{ c_low = max(c_low, $2.low); 
					  c_high = min(c_high, $2.high); }
	;

setspec : '[' number ',' number ']'	{ $$.low = $2; $$.high = $4; }
	;

maskconstraints : maskconstraint
	;

maskconstraint : MASK number		{ c_mask &= $2; }
	;

number	: UNSIGNED			{ $$ = strtoul($1, 0, 0); }
	;

string	: STRING			{ $$ = strdup($1 + 1);
					  $$[strlen($$) - 1] = 0; }

%%
/* end of grammer -- misc C source code follows */

#include "cfg-scan.c"

static void yyerror(const char *s)
{
  printf("RMGR: ERROR: while parsing config file: %s\n"
	 "             at line %d, col %d\n", s, line, col);
}


#ifdef TEST

char cfg[] = "task modname 'foo'\n"
             "#a comment\n"
             "child in [10,30] max 100\n";

int main(void)
{
  cfg_setup_input(cfg, cfg + sizeof(cfg));

  return cfg_parse();
}

#endif /* TEST */
