/* scstr.c	- Smartcard related functions - string stuff
 *
 * Copyright 1993-1997, Tim Hudson. All rights reserved.
 *
 * You can pretty much do what you like with this code except pretend that 
 * you wrote it provided that any derivative of this code includes the
 * above comments unchanged. If you put this in a product then attribution
 * is mandatory. See the details in the COPYING file.
 *
 * Tim Hudson
 * tjh@cryptsoft.com
 *
 */

#include "platform.h"

#ifdef USE_STDIO
#include <stdio.h>
#endif /* USE_STDIO */

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "sio.h"
#include "sct0.h"
#include "sc.h"
#include "slog.h"

static struct {
  int code;
  char *desc;
  char *alias;
} reader2string[]={
{SC_READER_UNKNOWN, "UNKNOWN", NULL },
{SC_READER_DUMBMOUSE, "DUMBMOUSE", NULL }, 
{SC_READER_SCT, "SCT", "SCR"}, 
{SC_READER_REFLEX, "RFX", "REFLEX"}, 
{SC_READER_GCR, "GCR", "GEMPLUS" },
{SC_READER_PE, "PE", "PHILIPS" }
};
#define N_reader2string (sizeof(reader2string)/sizeof(reader2string[0]))

static struct {
  int code;
  char *desc;
  char *alias;
} card2string[]={
{SC_CARD_UNKNOWN,"UNKNOWN", NULL},
{SC_CARD_CRYPTOFLEX,"CRYPTOFLEX", "CF"},
{SC_CARD_DX,"DX",NULL},
{SC_CARD_CHIPKNIP,"CHIPKNIP","CC"},
{SC_CARD_GSM,"GSM","SIM" }
};
#define N_card2string (sizeof(card2string)/sizeof(card2string[0]))

char *SC_Reader2String(int code)
{
  int i;

  for(i=0;i<N_reader2string;i++)
    if (reader2string[i].code==code)
      return reader2string[i].desc;
  return("UNKNOWN");
}

int SC_String2Reader(char *desc)
{
  int i;

  for(i=0;i<N_reader2string;i++) {
    if (STRCASECMP(reader2string[i].desc,desc)==0)
      return reader2string[i].code;
    /* allow for aliases */
    if (reader2string[i].alias!=NULL) {
      if (STRCASECMP(reader2string[i].alias,desc)==0)
	return reader2string[i].code;
    }
  }
  return(SC_READER_UNKNOWN);
}

char *SC_Card2String(int code)
{
  int i;

  for(i=0;i<N_card2string;i++)
    if (card2string[i].code==code)
      return card2string[i].desc;
  return("UNKNOWN");
}

int SC_String2Card(char *desc)
{
  int i;

  for(i=0;i<N_card2string;i++) {
    if (STRCASECMP(card2string[i].desc,desc)==0)
      return card2string[i].code;
    /* allow for aliases */
    if (card2string[i].alias!=NULL) {
      if (STRCASECMP(card2string[i].alias,desc)==0)
	return card2string[i].code;
    }
  }
  return(SC_CARD_UNKNOWN);
}

static struct {
  int code;
  char *token;
  char *text;
} err2string[] = {
{ SC_ERR_NONE, "NONE", NULL },
{ SC_ERR_UNSPECIFIED, "UNSPECIFIED", NULL }, 
{ SC_ERR_BAD_ATR, "BAD_ATR", NULL }, 
{ SC_ERR_PROBE_FAILED, "PROBE_FAILED", NULL }, 
{ SC_ERR_NO_REPLY, "NO_REPLY", NULL }, 
{ SC_ERR_NO_INS_ON_REPLY, "NO_INS_ON_REPLY", NULL }, 
{ SC_ERR_BAD_CMD_FORMAT, "BAD_CMD_FORMAT", NULL }, 
{ SC_ERR_BAD_CMD_TOO_SHORT, "CMD_TOO_SHORT", NULL }, 
{ SC_ERR_BAD_CMD_DATA_MISSING, "CMD_DATA_MISSING", NULL }, 
{ SC_ERR_EXE_CMD_NOT_FOUND, "EXE_CMD_NOT_FOUND", NULL }, 
{ SC_ERR_EXE_NO_CMD_TABLE, "EXE_NO_CMD_TABLE", NULL },
{ SC_ERR_EXE_MISSING_PARAM, "EXE_MISSING_PARAM", NULL },
{ SC_ERR_MSG_TOO_SHORT, "MSG_TOO_SHORT", NULL },
{ SC_ERR_WRITE_FAILED, "WRITE_FAILED", NULL },
{ SC_ERR_READ_FAILED, "READ_FAILED", NULL },
{ SC_ERR_MSG_TRAILER_INVALID, "MSG_TRAILER_INVALID", NULL },
{ SC_ERR_MSG_BAD, "MSG_BAD", NULL },
{ SC_ERR_VAR_NO_SPACE, "VAR_NO_SPACE", NULL },
{ SC_ERR_VAR_SHORT_DATA, "VAR_SHORT_DATA", NULL },
{ SC_ERR_PARSE_FAILED, "PARSE_FAILED", NULL },
{ SC_ERR_BAD_CONVERT_ASCII, "BAD_CONVERT_ASCII", NULL },
{ SC_ERR_BAD_CONVERT_VARS, "BAD_CONVERT_VARS", NULL },
{ SC_ERR_MSG_HEADER_INVALID, "MSG_HEADER_INVALID", NULL },
{ SC_ERR_NOT_SUPPORTED, "NOT_SUPPORTED", NULL }
};
#define N_err2string (sizeof(err2string)/sizeof(err2string[0]))

char *SC_ErrorCode2String(int code)
{
  int i;

  for(i=0;i<N_err2string;i++)
    if (err2string[i].code==code)
      return err2string[i].token;
  return("UNKNOWN");
}

static struct {
  int code;
  char *token;
  char *text;
} rsp2string[] = {
{ SC_RSP_OK, "OK" },
{ SC_RSP_UNKNOWN, "UNKNOWN" },
{ SC_RSP_DENIED, "DENIED" },
{ SC_RSP_NOTFOUND, "NOTFOUND" },
{ SC_RSP_BADCLA, "BADCLA" },
{ SC_RSP_BADINS, "BADINS" },
{ SC_RSP_BADP1, "BADP1" },
{ SC_RSP_BADP2, "BADP2" },
{ SC_RSP_BADP1P2, "BADP1P2" },
{ SC_RSP_BADLEN, "BADLEN" },
{ SC_RSP_BADPASSWORD, "BADPASSWORD" },
{ SC_RSP_WRONGTYPE, "WRONGTYPE" },
{ SC_RSP_BLOCKED, "BLOCKED" },
{ SC_RSP_NOTSELECTED, "NOTSELECTED" },
{ SC_RSP_BADADDR, "BADADDR" }
};
#define N_rsp2string (sizeof(rsp2string)/sizeof(rsp2string[0]))

char *SC_RSPCode2String(int code)
{
  int i;

  for(i=0;i<N_rsp2string;i++)
    if (rsp2string[i].code==code)
      return rsp2string[i].token;
  return("UNKNOWN");
}

