/* sccards.c	- Smartcard related functions - card cmd tables
 *
 * Copyright 1993-1997, Tim Hudson. All rights reserved.
 *
 * You can pretty much do what you like with this code except pretend that 
 * you wrote it provided that any derivative of this code includes the
 * above comments unchanged. If you put this in a product then attribution
 * is mandatory. See the details in the COPYING file.
 *
 * Tim Hudson
 * tjh@cryptsoft.com
 *
 */

#include "platform.h"

#ifdef USE_STDIO
#include <stdio.h>
#endif /* USE_STDIO */

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "sio.h"
#include "sct0.h"
#include "sc.h"
#include "scint.h"
#include "strutil.h"

/* these errors are the ones that are common amongst *all* cards ... if
 * the error is not common then it should be in the card specific list
 * and not in here!
 */
static SC_CMD_ERR_CVT global_errs[]={
{ 0x90, 0x00, SC_RSP_OK },
{ 0x6d, 0x00, SC_RSP_BADINS },
{ 0x6e, 0x00, SC_RSP_BADCLA }
};
#define N_global_errs (sizeof(global_errs)/sizeof(global_errs[0]))

static SC_CMD_TABLE cmd_table[]={
{ SC_CARD_CRYPTOFLEX, "cryptoflex", 
            cryptoflex_cmds, &N_cryptoflex_cmds,
            cryptoflex_icmds, &N_cryptoflex_icmds,
            cryptoflex_errs, &N_cryptoflex_errs,
	    cryptoflex_pdata, &N_cryptoflex_pdata,
	    cryptoflex_pconst, &N_cryptoflex_pconst
},
{ SC_CARD_DX, "dx", 
            dx_cmds, &N_dx_cmds, 
            dx_icmds, &N_dx_icmds, 
            dx_errs, &N_dx_errs,
	    dx_pdata, &N_dx_pdata,
	    dx_pconst, &N_dx_pconst
},
{ SC_CARD_CHIPKNIP, "chipknip", 
            chipknip_cmds, &N_chipknip_cmds,
            chipknip_icmds, &N_chipknip_icmds,
            chipknip_errs, &N_chipknip_errs,
	    chipknip_pdata, &N_chipknip_pdata,
	    chipknip_pconst, &N_chipknip_pconst
},
{ SC_CARD_GSM, "gsm", 
            gsm_cmds, &N_gsm_cmds,
            gsm_icmds, &N_gsm_icmds,
            gsm_errs, &N_gsm_errs,
	    gsm_pdata, &N_gsm_pdata,
	    gsm_pconst, &N_gsm_pconst
},
};
#define N_cmd_table (sizeof(cmd_table)/sizeof(cmd_table[0]))

/* fwd decl */
static int SC_ExecCmd(SC_STATE *sc,SC_CMD_ENT *ent,
                                    char *cmdargs,char *uout);
static char *getname(char *buf,char **pos);
static char *getval(SC_STATE *sc, char *var, int vallen, 
                          char **cmdargs,char *prompt);
static char *convert_ascii(char *inbuf,char *outbuf);
static char *convert_vars(SC_STATE *sc, char *inbuf,char *outbuf);

int SC_ExecuteCommand(SC_STATE *sc,char *thecmd,char *uout,int chain)
{
  int i,j,k,len,ret;
  char ch;
  SC_CMD_ENT *ent;
  SC_CMD_INTERNAL *ient;
  SC_CMD_PDATA *pdata;
  char cmdbuf[SC_MAX_HEX_DATA_LEN];
  char tmpbuf1[SC_MAX_HEX_DATA_LEN];
  char tmpbuf2[SC_MAX_HEX_DATA_LEN];
  char tmpbuf3[SC_MAX_HEX_DATA_LEN];
  char *p1,*p2;
  char *cmd;

  /* watch for rubbish */
  if ((sc==NULL) || (thecmd==NULL) || (uout==NULL))
    return (0);

  /* failure unless told otherwise */
  ret=(0);

  if (sc->slog)
    SLOG_printf(sc->slog,"SC_ExecuteCommand: \"%s\"\n",thecmd);

  /* allow for compiling without writeable strings */
  strcpy(cmdbuf,thecmd);
  cmd=cmdbuf;

  /* command depth is increasing */
  sc->exe_depth++;

  /* check for ASCII->HEX conversions that may be required for this
   * command ... <ASCII> => HEX HEX HEX with \ able to escape the closing
   * > and itself
   */
  cmd=convert_ascii(cmd,tmpbuf1);
  if (cmd==NULL) {
    sc->error_code=SC_ERR_BAD_CONVERT_ASCII;
    goto done;
  }

  /* now convert any vars/patterns across too */
  cmd=convert_vars(sc,cmd,tmpbuf2);
  if (cmd==NULL) {
    sc->error_code=SC_ERR_BAD_CONVERT_VARS;
    goto done;
  }

  /* find the matching entry for this card */
  for(i=0;i<N_cmd_table;i++) {
    if (sc->card_type==cmd_table[i].card_type) {
      /* get first word */
      len=strlen(cmd);
      for(j=0;j<len;j++)
	if (isspace(cmd[j]))
	  break;
      ch=cmd[j];
      cmd[j]='\0';

      /* handle any internal commands directly */
      if (cmd[0]=='!') {
	if (STRCASECMP(cmd+1,"list")==0) {
	  cmd[j]=ch;
	  uout[0]='\0';
	  for(k=0;k<(*cmd_table[i].N_cmds);k++) {
	    ent=cmd_table[i].cmds+k;
	    if (uout[0]!='\0')
	      strcat(uout,",");
	    strcat(uout,ent->name);
	  }
	} else if (STRCASECMP(cmd+1,"ilist")==0) {
	  cmd[j]=ch;
	  uout[0]='\0';
	  /* catch the internal table first */
	  for(k=0;k<N_sc_icmds;k++) {
	    ient=sc_icmds+k;
	    if (uout[0]!='\0')
	      strcat(uout,",");
	    strcat(uout,ient->name);
	  }
	  /* then the command specific one */
	  for(k=0;k<(*cmd_table[i].N_icmds);k++) {
	    ient=cmd_table[i].icmds+k;
	    if (uout[0]!='\0')
	      strcat(uout,",");
	    strcat(uout,ient->name);
	  }
	} else if (STRCASECMP(cmd+1,"desc")==0) {
	  cmd[j]=ch;
	  uout[0]='\0';
	  for(k=0;k<(*cmd_table[i].N_cmds);k++) {
	    if (STRCASECMP(cmd_table[i].cmds[k].name,cmd+j+1)==0) {
	      ent=cmd_table[i].cmds+k;

	      /* how should I describe a command? */
	      p2=ent->P1P2;
	      while ((p1=getname(p2,&p2))!=NULL) {
		if (uout[0]!='\0')
		  strcat(uout,",");
		strcat(uout,p1);
		if (strchr(p1,':')==NULL)
		  strcat(uout,":2");
	      }
	      p2=ent->P1;
	      while ((p1=getname(p2,&p2))!=NULL) {
		if (uout[0]!='\0')
		  strcat(uout,",");
		strcat(uout,p1);
	      }
	      p2=ent->P2;
	      while ((p1=getname(p2,&p2))!=NULL) {
		if (uout[0]!='\0')
		  strcat(uout,",");
		strcat(uout,p1);
	      }
	      p2=ent->len;
	      while ((p1=getname(p2,&p2))!=NULL) {
		if (uout[0]!='\0')
		  strcat(uout,",");
		strcat(uout,p1);
	      }
	      p2=ent->data;
	      while ((p1=getname(p2,&p2))!=NULL) {
		if (uout[0]!='\0')
		  strcat(uout,",");
		strcat(uout,p1);
	      }
	      break;
	    }
	  }
	  if (k==(*cmd_table[i].N_cmds))
	    sprintf(uout,"Unknown command to desc \"%s\"",cmd+j+1);
	} else if (STRCASECMP(cmd+1,"echo")==0) {
	  /* get the value of a single var */
	  cmd[j]=ch;
	  uout[0]='\0';

	  /* skip leading whitespace */
	  while(isspace(cmd[j]))
	    j++;

	  if (sc->sout)
	    SLOG_printf(sc->sout,"%s\n",cmd+j);

	} else if (STRCASECMP(cmd+1,"get")==0) {
	  /* get the value of a single var */
	  cmd[j]=ch;
	  uout[0]='\0';

	  /* skip leading whitespace */
	  while(isspace(cmd[j]))
	    j++;

	  p1=SC_GetVar(sc,cmd+j);
	  if (p1!=NULL)
	    strcpy(uout,p1);

	} else if (STRCASECMP(cmd+1,"set")==0) {
	  /* set a single var to the given value */
	  cmd[j]=ch;
	  uout[0]='\0';

	  /* skip leading whitespace */
	  while(isspace(cmd[j]))
	    j++;

	  /* should have varname<SPACES>value */
	  p1=cmd+j;
	  while(cmd[j] && !isspace(cmd[j]))
	    j++;
	  cmd[j++]='\0';

	  /* skip any extra spaces */
	  while (isspace(cmd[j]))
	    j++;
	  p2=cmd+j;

	  /* stop on first space */
	  while(cmd[j] && !isspace(cmd[j]))
	    j++;
	  cmd[j++]='\0';

	  /* simply now ... */
	  if (!SC_SetVar(sc,p1,p2)) {
	    sprintf(uout,"SET: failed for %s=%s\n",p1,p2);
	    goto done;
	  }

	} else if (STRCASECMP(cmd+1,"clear")==0) {
	  /* clear out all vars */
	  uout[0]='\0';
	  SC_ClearVars(sc);
	} else if (STRCASECMP(cmd+1,"vars")==0) {
	  int vars;

	  if (sc->sout) {
	    vars=SC_GetVarCount(sc);
	    if (vars>0) {
	      for(i=0;i<vars;i++) {
		SLOG_printf(sc->sout,"VAR %s=%s\n",SC_GetVarName(sc,i),
					    SC_GetVarValue(sc,i));
	      }
	    }
	  }

#ifdef USE_STDIO
	} else if (STRCASECMP(cmd+1,"write")==0) {
	  unsigned long fsize;
	  char fid[5];
	  char cmdbuf[SC_MAX_DATA_LEN];
	  FILE *fp;
	  int k,maxchunk;
	  unsigned int j;
	  char *p;

	  /* write the current file (after a selectfile)
	   * assumes: FILE_ID is the name
	   *          FILE_SIZE is the length
	   */
	  if ((p=SC_GetVar(sc,"FILE_ID"))==NULL) {
	    sprintf(uout,"WRITE: no file id in FILE_ID\n");
	    goto done;
	  } 
	  if (strlen(p)>4)
	    goto done;
	  strcpy(fid,p);
	  if ((p=SC_GetVar(sc,"FILE_SIZE"))==NULL) {
	    sprintf(uout,"WRITE: no file size in FILE_SIZE\n");
	    goto done;
	  }
	  fsize=strtoul(p,NULL,16);

	  fp=fopen(fid,"wb");
	  if (fp==NULL) {
	    sprintf(uout,"WRITE: create %s failed\n",fid);
	    goto done;
	  }

	  /* now grab chunks of SC_MAX_TRANSFER_LEN bytes */
	  maxchunk=SC_MAX_TRANSFER_LEN;
	  maxchunk=16;
	  for(j=0;j<fsize;j+=maxchunk) {
	    k=fsize-j;
	    if (k>maxchunk)
	      k=maxchunk;
	    sprintf(cmdbuf,"ReadBinary %04x %02x",j,k);
	    if (!SC_ExecuteCommand(sc,cmdbuf,uout,0)) {
	      sprintf(uout,"WRITE: exec readbinary failed\n");
	      fclose(fp);
	      goto done;
	    } else {
	      fwrite(sc->last_rsp.data,sc->last_rsp.len,1,fp);
	    }
	  }
	  fclose(fp);
#endif /* USE_STDIO */
	} else if (STRCASECMP(cmd+1,"scan")==0) {
	  unsigned long fid;
	  char *p;
	  char cmdbuf[SC_MAX_DATA_LEN];
	  char startdir[64];
	  int all,finishpoint;
	  char *token;

	  /* by default we are scanning the root level */
	  strcpy(startdir,"/");

	  /* 
	   * scan
	   * scan [fileid]
	   * scan [all [start [finish]]]
	   * scan [[dir=startdir] [all [start [finish]]]]
	   *
	   */

	  /* no output to start with */
	  uout[0]='\0';

	  /* grab the starting point ... */
	  token=cmd+1+strlen("scan ");
next_word: ;
	  p=mstrtok(token," \t\n");
	  if (p!=NULL) {
	    if (strncmp(p,"dir=",strlen("dir="))==0) {
	      strcpy(startdir,p+strlen("dir="));
	      goto next_word;
	    }
	    if (STRCASECMP(p,"all")==0) {
	      all=1;
	      /* the user can specify a starting point ... in which 
	       * case we can from there until the also optional
	       * ending point 
	       */
	      p=mstrtok(token," \t\n");
	      if (p==NULL) {
	        fid=0;
		finishpoint=255;
	      } else {
	        fid=strtoul(p,NULL,16);
		p=mstrtok(token," \t\n");
		if (p==NULL)
		  finishpoint=255;
		else
		  finishpoint=strtoul(p,NULL,16);
	      }
	    } else {
	      all=1;
	      fid=strtoul(p,NULL,16);
	    }

	    /* we must start in a known position ... either the 
	     * directory the user specified or the root otherwise
	     * the results may not be what is expected
	     */
	    sprintf(cmdbuf,"sc_sel %s",startdir);
	    if (!SC_ExecuteCommand(sc,cmdbuf,tmpbuf3,0)) {
	      sprintf(uout,"SCAN: initial selectfile %s failed",startdir);
	      mstrtok_clear(token);
	      goto done;
	    }
	    /* the command must work with no errors for us to be sure
	     * the dir exists 
	     */
	    if (SC_LAST_RSP_CODE(sc)!=SC_RSP_OK) {
	      sprintf(uout,"SCAN: initial selectfile %s failed",startdir);
	      mstrtok_clear(token);
	      goto done;
	    }

	    if (fid<=255) {
next_file: ;
	      for(k=0;k<=255;k++) {
		sprintf(cmdbuf,"SelectFile %02x%02x",(int)(fid & 0xff),k);
		if (!SC_ExecuteCommand(sc,cmdbuf,tmpbuf3,0)) {
		  sprintf(uout,"SCAN: selectfile failed");
		  mstrtok_clear(token);
		  goto done;
		}

		if (SC_LAST_RSP_CODE(sc)==SC_RSP_OK) {
		    /* add the file id into the output list */
		    sprintf(uout+strlen(uout),"%02x%02x ",(int)(fid & 0xff),k);

		    /* string must be writeable! */
		    sprintf(cmdbuf,"sc_sel %s",startdir);
		    if (!SC_ExecuteCommand(sc,cmdbuf,tmpbuf3,0)) {
		      sprintf(uout,"SCAN: selectfile back to startdir failed\n");
		      mstrtok_clear(token);
		      goto done;
		    }
 		    if (SC_LAST_RSP_CODE(sc)!=SC_RSP_OK) {
		      sprintf(uout,"SCAN: selectfile back to startdir failed\n");
		      mstrtok_clear(token);
		      goto done;
		    }
		}

		if (sc->slog!=NULL) 
		  SLOG_printf(sc->slog,"SCANRES: %s\n",uout);

	      }
	      if (all) {
		fid++;
		/* stop at the end of the scan */
		if (fid>(unsigned long)finishpoint)
		  goto donescan;
		goto next_file;
	      }
	    } else {
	      /* scan of a subdir - select the dir then scan */
	      sprintf(cmdbuf,"SelectFile %04x",(int)(fid & 0xffff));
	      if (!SC_ExecuteCommand(sc,cmdbuf,tmpbuf3,0)) {
		sprintf(uout,"SCAN: selectfile failed");
		mstrtok_clear(token);
		goto done;
	      }
  	      if (SC_LAST_RSP_CODE(sc)!=SC_RSP_OK) {
		sprintf(uout,"SCAN: initial dir of %04x does not exist",
		          (int)(fid & 0xffff));
		mstrtok_clear(token);
		goto done;
	      }
	      sprintf(startdir,"/%04x",(int)(fid & 0xffff));
	      /* now we start looking for files inside the DF
	       * starting with 0 and working our way up all the
	       * files ... if the user wants to subset the search
	       * then they can use the other form of the command
	       */
	      fid=0;
	      all=1;
	      goto next_file;
	    }
	  }

donescan: ;
          /* cleanup */
	  mstrtok_clear(token);

	  /* if we get to here it worked */
	  sc->error_code=SC_ERR_NONE;

	  /* also blank the last response code too ... as it isn't
	   * something that it makes sense to keep for the moment
	   */
	  sc->last_rsp.rsp_code=SC_ERR_NONE;

	} else if (STRCASECMP(cmd+1,"plist")==0) {
	  cmd[j]=ch;
	  uout[0]='\0';

	  for(k=0;k<(*cmd_table[i].N_pdata);k++) {
	    pdata=cmd_table[i].pdata+k;
	    if (uout[0]!='\0')
	      strcat(uout,",");
	    strcat(uout,pdata->name);
	  }
	} else if (STRCASECMP(cmd+1,"pdesc")==0) {
	  cmd[j]=ch;
	  uout[0]='\0';

	  /* skip leading whitespace */
	  while(isspace(cmd[j]))
	    j++;

	  for(k=0;k<(*cmd_table[i].N_pdata);k++) {
	    pdata=cmd_table[i].pdata+k;

	    if (STRCASECMP(pdata->name,cmd+j)==0) {
	      sprintf(uout,"%s",pdata->data);
	      break;
	    }
	  }

	} else if (STRCASECMP(cmd+1,"parse")==0) {
	  cmd[j]=ch;
	  uout[0]='\0';

	  /* skip leading whitespace */
	  while(isspace(cmd[j]))
	    j++;

	  /* no pattern name means we want a list of patterns */
	  if (strlen(cmd+j)==0) {
	    for(k=0;k<(*cmd_table[i].N_pdata);k++) {
	      pdata=cmd_table[i].pdata+k;
	      if (uout[0]!='\0')
		strcat(uout,",");
	      strcat(uout,pdata->name);
	    }
	  } else {
	    /* call the parsing logic to convert the last response into
	     * more user friendly output
	     */
	    if (!SC_DoParse(sc,cmd+j))
	      sprintf(uout,"PARSE:OK");
	    else {
	      /* we must indicate a command error */
	      sc->error_code=SC_ERR_PARSE_FAILED;
	      sprintf(uout,"PARSE:FAILED");
	      goto done;
	    }
	  }
	} else if ( (STRCASECMP(cmd+1,"text")==0) ||
	            (STRCASECMP(cmd+1,"ascii")==0) ) {
	  cmd[j]=ch;
	  uout[0]='\0';
	  for(k=0;k<sc->last_rsp.len;k++) {
	    ch=sc->last_rsp.data[k];
	    if (!isprint(ch))
	      ch='.';
	    sprintf(uout+strlen(uout),"%c",ch);
	  }
	  if (sc->sout) {
	    SLOG_dump(sc->sout,sc->last_rsp.data,sc->last_rsp.len,1);
	    SLOG_printf(sc->sout,"\n");
	  }
	} else if (STRCASECMP(cmd+1,"dump")==0) {
	  cmd[j]=ch;
	  uout[0]='\0';
	  if (sc->sout) {
	    SLOG_dump(sc->sout,sc->last_rsp.data,sc->last_rsp.len,2);
	    SLOG_printf(sc->sout,"\n");
	  }
	} else {
	  sprintf(uout,"Unknown internal command \"%s\"",cmd+1);
	}

	/* this is successful - well good enough for me */
	ret=1;
	goto done;
      }

      /* first check card specific "internal" commands */
      for(k=0;k<(*cmd_table[i].N_icmds);k++) {
	if (STRCASECMP(cmd_table[i].icmds[k].name,cmd)==0) {
	  cmd[j]=ch;
	  ient=cmd_table[i].icmds+k;

	  if (ient->func!=NULL)
	    ret=(*ient->func)(sc,(void *)ient,cmd+j,uout);
	  goto done;
	}
      }

      /* then match "internal" global commands - that way we can
       * have a default implementation of a command which can be
       * overridden in a card specific manner which makes things
       * easier to handle for lots of commands
       */
      for(k=0;k<N_sc_icmds;k++) {
	if (STRCASECMP(sc_icmds[k].name,cmd)==0) {
	  cmd[j]=ch;
	  ient=sc_icmds+k;

	  if (ient->func!=NULL)
	    ret=(*ient->func)(sc,(void *)ient,cmd+j,uout);
	  goto done;
	}
      }


      /* find command name match */
      for(k=0;k<(*cmd_table[i].N_cmds);k++) {
	if (STRCASECMP(cmd_table[i].cmds[k].name,cmd)==0) {
	  cmd[j]=ch;
	  ent=cmd_table[i].cmds+k;

	  /* at the beginning of a command sequence where we
	   * are talking to the card itself we clear out
	   * any existing vars 
	   */
	  if (sc->exe_depth==1) {
	    if (!SC_ClearVars(sc))
	      return(0);
	  }

	  ret=SC_ExecCmd(sc,ent,cmd+j,uout);
	  if ((ret!=0) && chain && (ent->chain_SW1!=NULL) ) {
	    /* check to see if we should be chaining to another command
	     * which is probably going to be something like GetResponse
	     * but could actually be anything
	     */
	    if (strncmp(ent->chain_SW1,uout+2,strlen(ent->chain_SW1))==0) {
	      strcpy(tmpbuf1,ent->chain_cmd);
	      strcat(tmpbuf1," ");
	      strcpy(tmpbuf2,ent->chain_args);
	      p1=mstrtok(tmpbuf2," \t\n");
	      while (p1!=NULL) {
		p2="";
		strcat(tmpbuf1,getval(sc,p1,1,&p2,""));
	        p1=mstrtok(tmpbuf2," \t\n");
	      }
	      if (sc->slog)
		SLOG_printf(sc->slog,"CHAIN: %s\n",tmpbuf1);
	      ret=SC_ExecuteCommand(sc,tmpbuf1,uout,chain);
	      if (sc->slog)
		SLOG_printf(sc->slog,"CHAIN: RET %d\n",ret);
	      mstrtok_clear(tmpbuf2);
	    }
	  }

	  /* if no errors then check for parse commands that should be
	   * run on completion to pull appart responses into useful
	   * vars (assuming we get good data back)
	   */
	  if (ret!=0) {
	    /* we want this to always happen now so that internal
	     * commands can also benefit from the "cracking" of structs
	     * into easy to handle variables
	     * --tjh 
	     */
	    if (1 || (sc->exe_depth==1)) {
	      if (sc->last_rsp.len>0) {
		for(i=0;i<SC_MAX_AUTO_PARSE_ENTRIES;i++) {
		  /* first null name is the end of the parse list */
		  if (ent->auto_parse[i].name==NULL)
		    break;
		  /* if we don't care about length or we have an
		   * exact match then we know what to do
		   */
		  if ( (ent->auto_parse[i].len<=0) ||
		       (ent->auto_parse[i].len==sc->last_rsp.len) ) {
		    if (sc->slog)
		      SLOG_printf(sc->slog,"AUTOPARSE %s\n",
			        ent->auto_parse[i].name);
		    (void)SC_DoParse(sc,ent->auto_parse[i].name);
		    /* we should only every invoke *one* of the 
		     * autoparsing functions!
		     */
		    break;
		  }
		}
	      }
	    }
	  }
          goto done;
	}
      }

      /* command not found */
      sc->error_code=SC_ERR_EXE_CMD_NOT_FOUND;
      goto done;
    }
  }

  /* no command table for that type */
  sc->error_code=SC_ERR_EXE_NO_CMD_TABLE;

done: ;

  sc->exe_depth--;
  return (ret);
}

/* getname - extract a varname from the buffer containing ${VARNAME:LEN}
 *           constructs where :LEN is optional
 *
 * returns NULL if no name is found
 *
 */
static char *getname(char *buf, char **pos)
{
  static char varname[32];
  char *p1,*p2;
  int vlen;

  if (pos!=NULL)
    (*pos)=NULL;

  /* one good turn deserves another */
  if (buf==NULL)
    return(NULL);

  if ((p1=strstr(buf,"${"))!=NULL) {
    p2=strstr(p1,"}");

    /* if no termination of the name then ignore it */
    if (p2==NULL)
      return (NULL);

    if (pos!=NULL)
      (*pos)=p2+1;

    vlen=((unsigned long)p2-(unsigned long)p1)+1-3;
    strncpy(varname,p1+2,vlen);
    varname[vlen]='\0';

    return (varname);
  }
  return (NULL);
}

/* getdata - hook for prompting for the value of a variable when the
 * 	     data has not been provided in the input buffer.
 *
 *	     For the moment this isn't implemented and no external
 *	     hook has been exported - though this is planned
 */
static char *getdata(SC_STATE *sc, char *prompt, char *varname, int vallen)
{
  /* we don't support prompting */
  return (NULL);
}

/* getval - get the "value" of a given expression buffer where it may
 *	    contain variable references of the form ${VARNAME[:LEN]}
 *
 */
static char *getval(SC_STATE *sc, char *var, int vallen, 
                          char **cmdargs,char *prompt) 
{
  static char buf[SC_MAX_DATA_LEN];
  char varname[32];
  char *p1,*p2;
  int len,got,ofs,havevar;
  int totalgot,totallen;

  /* handle the simple cases */
  if (var==NULL)
    return "";
  len=strlen(var);
  if (len==0)
    return "";

  /* extract the symbolic stuff */
  ofs=0;
  havevar=0;
  totallen=0;
  totalgot=0;
  p2=var;
  while ((p1=getname(p2,&p2))!=NULL) {
    havevar=1;
    strcpy(varname,p1);

    /* now extract the length (if it is there) */
    if ((p1=strchr(varname,':'))!=NULL) {
      *p1='\0';
      p1++;
      vallen=(int)strtol(p1,NULL,0);
    }
    totallen+=vallen;

    /* now get the value */
    p1=(*cmdargs);
    got=0;

    /* first check to see if it is one of the reserved names
     * that we handle specially - which are used during chaining
     * of args as they are the only output that can normally be
     * passed between one command execution and the next in the
     * chain (if auto-chaining is being used)
     */
    if (STRCASECMP(varname,"SW1")==0) {
      sprintf(buf+ofs,"%02x",sc->last_rsp.SW1);
      got=2;
    } else if (STRCASECMP(varname,"SW2")==0) {
      sprintf(buf+ofs,"%02x",sc->last_rsp.SW2);
      got=2;
    } else {
      /* copy bytes ignoring white space */
      while ( (*p1!='\0') && (got<(vallen*2)) ) {
	if (!isspace(*p1))
	  buf[ofs+got++]=(*p1);
	p1++;
      }
    }
    /* terminate the string */
    buf[ofs+got]='\0';

    /* we have got extra bytes now */
    ofs+=got;
    totalgot+=got;

    /* we update the args pointer for whatever we consumed
     * whether or not we think it is enough data
     */
    (*cmdargs)=p1;

    /* not enough data means something went wrong */
    if (got!=(vallen*2)) {
      p1=getdata(sc,prompt,varname,vallen);
      return (NULL);
    }
  }
  
  if (havevar) {
    /* we have the value - but still check the size */
    if (totalgot==(totallen*2)) {
      return (buf);
    } else {
      p1=getdata(sc,prompt,varname,vallen);
      return (NULL);
    }
  } else {
    /* if we don't have a var name simply return what was passed in */
    return (var);
  }
}

/* SC_ExecCmd - execute a particular command by assembling the parameters
 *		and calling SC_DoAsciiCommand
 */
static int SC_ExecCmd(SC_STATE *sc,SC_CMD_ENT *ent,
                                    char *cmdargs,char *uout)
{
  char tmpbuf[32];
  char buf1[5];
  char buf2[5];
  char buf4[SC_MAX_HEX_DATA_LEN];
  char cmd[SC_MAX_HEX_DATA_LEN];
  char *p;
  char *P1, *P2, *data;
  char *len;
  char *llen;
  int dlen;
  int ret;

  memset(buf1,'\0',sizeof(buf1));
  memset(buf2,'\0',sizeof(buf2));
  memset(buf4,'\0',sizeof(buf4));

  sc->error_code=SC_ERR_NONE;

  /* first check for any parameters that are required and extract
   * them as we go along
   */
  if (ent->P1P2!=NULL) {
    p=getval(sc,ent->P1P2,2,&cmdargs,ent->name);
    if (p==NULL)
      goto bad_param;
    strcpy(buf1,p);
    p=buf1;

    /* now split it into P1 and P2 */
    if (strlen(p)>=4) {
      P1=p;
      strcpy(buf2,p+2);
      P2=buf2;
      p[2]='\0';
    } else {
      P1="00";
      P2="00";
    }
  } else {
    /* we have individual P1 and P2 rather than a combined 
     * use of both bytes ... this is the normal case
     */
    P1=getval(sc,ent->P1,1,&cmdargs,ent->name);
    if (P1==NULL)
      goto bad_param;
    strcpy(buf1,P1);
    P1=buf1;
    P2=getval(sc,ent->P2,1,&cmdargs,ent->name);
    if (P2==NULL)
      goto bad_param;
    strcpy(buf2,P2);
    P2=buf2;
  }

  /* if we have a len parameter then it specifies the expected
   * length of data ... as the length can be variable so we 
   * have to watch this at this point so we know how much
   * data we should expect to collect
   */
  dlen=1;
  if (ent->direction==SC_DIR_IN) {
    if (ent->len!=NULL) {
      if (strlen(ent->len)!=0) {
	llen=getval(sc,ent->len,1,&cmdargs,ent->name);
	if (sc->slog)
	  SLOG_printf(sc->slog,"SC_ExecCmd llen %s\n",llen==NULL?"<NULL>":llen);
	if (llen==NULL)
	  goto bad_param;
	dlen=(int)strtol(llen,NULL,16);
      }
    }
  }

  /* any user data */
  data=getval(sc,ent->data,dlen,&cmdargs,ent->name);
  if (data==NULL)
    goto bad_param;
  strcpy(buf4,data);
  data=buf4;

  if (ent->direction==SC_DIR_OUT) {
    len=getval(sc,ent->len,1,&cmdargs,ent->name);
    if (len==NULL)
      goto bad_param;
  } else {
    /* we set things according to how much data we actually
     * have rather than anything else
     */
    sprintf(tmpbuf,"%02x",strlen(data)/2);
    len=tmpbuf;
  }

  /* now we can construct the command to send */
  sprintf(cmd,"%02x%02x%2s%2s%2s%s",ent->CLA,ent->INS,P1,P2,len,data);

  /* and the rest is the responsibility of the command handler now */
  if (sc->slog) {
    SLOG_printf(sc->slog,"SC_ExecCmd %s\n",cmd);
    /* might as well make this easily readable */
    SLOG_printf(sc->slog,"P1=%s\n",P1);
    SLOG_printf(sc->slog,"P2=%s\n",P2);
    SLOG_printf(sc->slog,"len=%s\n",len);
    SLOG_printf(sc->slog,"data=%s\n",data);
  }

  /* execute the command */
  ret=SC_DoAsciiCommand(sc,ent->direction,cmd,uout);
  return (ret);

bad_param: ;
  sc->error_code=SC_ERR_EXE_MISSING_PARAM;
  return (0);
}

/* SC_SW1SW2_RSPCode - convert from the card-specific SW1, SW2 values
 *		       into the card-independent RSPCodes
 */
int SC_SW1SW2_RSPCode(SC_STATE *sc,unsigned char SW1,unsigned char SW2)
{
  int i,j,ret;
  SC_CMD_ERR_CVT *errs;
  int N_errs;

  /* watch for rubbish */
  if (sc==NULL) 
    return (0);

  ret=SC_RSP_UNKNOWN;

  /* check the global error table first */
  errs=global_errs;
  N_errs=N_global_errs;
  /* now try to match */
  for(j=0;j<N_errs;j++) {
    /* check SW1 first */
    if (errs[j].SW1!=-1)
      if ((errs[j].SW1 & 0xff)!=SW1)
	continue;
    /* then check SW2 if required */
    if (errs[j].SW2!=-1)
      if ((errs[j].SW2 & 0xff)!=SW2)
	continue;
    /* if we get to here then we matched */
    ret=errs[j].rsp_code;
    goto done;
  }

  /* find the matching entry for this card and check the
   * card specific error table 
   */
  for(i=0;i<N_cmd_table;i++) {
    if (sc->card_type==cmd_table[i].card_type) {
      errs=cmd_table[i].errs;
      N_errs=(*cmd_table[i].N_errs);

      /* now try to match */
      for(j=0;j<N_errs;j++) {
	/* check SW1 first */
	if (errs[j].SW1!=-1)
	  if ((errs[j].SW1 & 0xff)!=SW1)
	    continue;
	/* then check SW2 if required */
	if (errs[j].SW2!=-1)
	  if ((errs[j].SW2 & 0xff)!=SW2)
	    continue;
	/* if we get to here then we matched */
	ret=errs[j].rsp_code;
	goto done;
      }

      /* if we match the card type we have done */
      break;
    }
  }
done: ;
  return (ret);
}

/* SC_ClearVars - remove all variable definitions 
 */
int SC_ClearVars(SC_STATE *sc)
{
  int ret;

  /* watch for rubbish */
  ret=(0);
  if (sc==NULL)
    goto done;

  /* clear out things - we don't want to leak any info
   * between commands
   */
  sc->N_vars=0;
  memset(sc->vardata,'\0',sc->vardata_used);
  memset(sc->valdata,'\0',sc->valdata_used);
  memset(sc->vars,'\0',sizeof(sc->vars));
  memset(sc->vals,'\0',sizeof(sc->vals));
  sc->vardata_used=0;
  sc->valdata_used=0;
  ret=1;

done:;
  return(ret);
}

/* SC_SetVar - set (or overwrite) the value of the specified variable 
 */
int SC_SetVar(SC_STATE *sc,char *name,char *val)
{
  int ret;
  int nlen,vlen;
  char *p;

  /* watch for rubbish */
  ret=(0);
  if ((sc==NULL) || (name==NULL) || (val==NULL))
    goto done;

  /* check to see if the var is already set and if so then we
   * overwrite it now 
   */
  if ((p=SC_GetVar(sc,name))!=NULL) {
    if (strlen(p)!=strlen(val)) {
      sc->error_code=SC_ERR_VAR_NO_SPACE;
      goto done;
    } else {
      /* update the value */
      memcpy(p,val,strlen(val));
    }
  }

  /* check that we have a slot available to hold the var */
  if (sc->N_vars==SC_VALUE_MAX_VARS) {
    sc->error_code=SC_ERR_VAR_NO_SPACE;
    goto done;
  }

  /* now check for space being available */
  nlen=strlen(name);
  vlen=strlen(val);
  if ((sc->vardata_used+nlen+1)>=SC_VALUE_BUF_LEN) {
    sc->error_code=SC_ERR_VAR_NO_SPACE;
    goto done;
  }
  if ((sc->valdata_used+vlen+1)>=SC_VALUE_BUF_LEN) {
    sc->error_code=SC_ERR_VAR_NO_SPACE;
    goto done;
  }

  /* now copy the values into the right places */
  sc->vars[sc->N_vars]=sc->vardata+sc->vardata_used;
  memcpy(sc->vardata+sc->vardata_used,name,nlen+1);
  sc->vardata_used+=nlen+1; 
  sc->vals[sc->N_vars]=sc->valdata+sc->valdata_used;
  memcpy(sc->valdata+sc->valdata_used,val,vlen+1);
  sc->valdata_used+=vlen+1; 

  /* it worked */
  sc->N_vars++;
  ret=1;

done: ;
  return(ret);
}

/* SC_GetVar - get the value of the specified variable or return NULL
 *	       if it doesn't exist
 */
char *SC_GetVar(SC_STATE *sc,char *name)
{
  int i,j;
  char tmpbuf[32];
  char *varname;
  int nlen;

  if ((sc==NULL) || (name==NULL))
    return (NULL);

  /* handle array index of [0] being the same as the
   * base name of a var so that we can be consistant
   */
  varname=name;
  nlen=strlen(name);
  if (nlen>3) {
    if (strcmp(name+nlen-3,"[0]")==0) {
      strcpy(tmpbuf,name);
      tmpbuf[nlen-3]='\0';
      varname=tmpbuf;
    }
  }

  /* now check for the constants */
  for(i=0;i<N_cmd_table;i++) {
    if (sc->card_type==cmd_table[i].card_type) {
      for (j=0;j<(*cmd_table[i].N_pconst);j++) {
	if (STRCASECMP(name,cmd_table[i].pconst[j].name)==0)
	  return (cmd_table[i].pconst[j].data);
      }
      break;
    }
  }

  /* search for the var by name */
  for(i=0;i<sc->N_vars;i++) {
    if (STRCASECMP(name,sc->vars[i])==0)
      return sc->vals[i];
  }

  return (NULL);
}

/* SC_GetVarCount - return the number of variables 
 */
int SC_GetVarCount(SC_STATE *sc)
{
  if (sc==NULL)
    return (0);
  else
    return (sc->N_vars);
}

/* SC_GetVarName - get the name of the idx-th variable */
char *SC_GetVarName(SC_STATE *sc,int idx)
{
  if (sc==NULL)
    return (NULL);
  if ((idx<0)||(idx>sc->N_vars))
    return (NULL);
  return (sc->vars[idx]);
}

/* SC_GetVarValue - get the value of the idx-th variable */
char *SC_GetVarValue(SC_STATE *sc,int idx)
{
  if (sc==NULL)
    return (NULL);
  if ((idx<0)||(idx>sc->N_vars))
    return (NULL);
  return (sc->vals[idx]);
}

int SC_GetIntVar(SC_STATE *sc,char *name)
{
  char *p;

  p=SC_GetVar(sc,name);
  if (p==NULL)
    return -1;
  else
    return (int)strtoul(p,NULL,16);
}

int SC_SetIntVar(SC_STATE *sc,char *name,int val,int len)
{
  char buf[64];

  sprintf(buf,"%0.*X",val,len);
  return SC_SetVar(sc,name,buf);
}

/* SC_DoParse - parse the last respond buffer using the specified parsing
 *		pattern name
 *
 * Usually this is invoked automatically via the card-specific command table 
 * or explicitly via the !parse internal command rather than via a direct
 * call to this function
 */
int SC_DoParse(SC_STATE *sc,char *cmd)
{
  int ret;
  int i,j;
  SC_CMD_PDATA *pdata;
  int N_pdata;
  char *p1,*p2;
  char varname[32];
  char tmpname[32];
  char tmpbuf[SC_VALUE_BUF_LEN];
  int vallen,used;
  int base_used=(-1);
  int idx=0;

  /* start with nothing */
  pdata=NULL;

  /* watch for rubbish */
  ret=(0);
  if ((sc==NULL) || (cmd==NULL))
    goto done;

  /* parsing requires that we clear things out first */
  if (!SC_ClearVars(sc))
    goto done;

  for(i=0;i<N_cmd_table;i++) {
    if (sc->card_type==cmd_table[i].card_type) {
      pdata=cmd_table[i].pdata;
      N_pdata=(*cmd_table[i].N_pdata);

      for(j=0;j<N_pdata;j++) {
	if (STRCASECMP(pdata[j].name,cmd)==0) {

          if (sc->slog)
	    SLOG_printf(sc->slog,"SC_DoParse %s\n",cmd);

	  /* now go and do things with the pattern 
	   * which should be a list of vars to copy the data
	   * from sc->last_rsp into
	   */
	  used=0;
do_again: ;
	  p2=pdata[j].data;
	  while ((p1=getname(p2,&p2))!=NULL) {
	    /* should have var[:len] */
	    strcpy(varname,p1);

	    /* default length is 1 byte */
	    vallen=1;

	    /* now extract the length (if it is there) */
	    if ((p1=strchr(varname,':'))!=NULL) {
	      *p1='\0';
	      p1++;
	      /* we can specify that the length is unknown and
	       * is the remainder of the data ... naturally this
	       * only works for the last catch-all :-)
	       */
	      if (strcmp(p1,"N")==0) {
		vallen=(sc->last_rsp.len-used);
	      } else {
	        vallen=(int)strtol(p1,NULL,0);
	      }
	    }

	    /* we abort with error if we run out of input
	     * data to match the parse pattern
	     */
            if (used+vallen>sc->last_rsp.len) {
	      sc->error_code=SC_ERR_VAR_SHORT_DATA;
	      goto done;
	    }

            /* now extract the value bytes out of the last
	     * response and store in the temp buffer
	     */
	    SC_BufferToHex(sc->last_rsp.data+used,vallen,tmpbuf,
	                          sizeof(tmpbuf),0);
            used+=vallen;

	    /* now set it */
	    if (idx>0) {
	      strcpy(tmpname,varname);
	      sprintf(varname,"%s[%d]",tmpname,idx);
	    }
	    if (!SC_SetVar(sc,varname,tmpbuf))
	      goto done;
	  }

	  /* if we get to here and there is still data to be consumed
	   * then we check to see if it is a multiple of what we have
	   * already used first time though and if so get another set
	   * of vars but indicate array indicies in the names
	   */
	  if (used<sc->last_rsp.len) {
	    if (base_used==-1)
	      base_used=used;
	    /* make sure we will have enough left for a whole round */
	    if ((used+base_used)<=sc->last_rsp.len) {
	      idx++;
	      goto do_again;
	    }
	  }

	  /* we need to indicate to the user how many "array" entries 
	   * there are in this result set
	   */
	  sprintf(tmpbuf,"%d",idx+1);
	  if (!SC_SetVar(sc,"$COUNT",tmpbuf))
	    goto done;

	  if (used!=sc->last_rsp.len) {
	    sc->error_code=SC_ERR_VAR_SHORT_DATA;
	    if (sc->slog)
	      SLOG_printf(sc->slog,"SC_DoParse: used %d rsp %d\n",used,
		                            sc->last_rsp.len);
	    goto done;
	  }

	  ret=1;

	  /* it worked ... check for a post-parsing function that 
	   * we should be calling to do work that isn't easily expressed
	   * in simple tables
	   */
	  if (pdata[j].func!=NULL) {
	    if (sc->slog)
	      SLOG_printf(sc->slog,"SC_DoParse -> post-parsing %s\n",cmd);
	    ret=(*pdata[j].func)(sc,pdata+j);
	  }
	  break;
	}
      }

      /* if we match the card type we have done */
      break;
    }
  }

done: ;
  /* if there is an error we clear out whatever vars we did
   * get around to setting so that we can be sure that the
   * error is noticed!
   */
  if (!ret) {
    if (sc->slog)
      SLOG_printf(sc->slog,"SC_DoParse: failed %d (%s)\n",
	        sc->error_code,SC_ErrorCode2String(sc->error_code));

    (void)SC_ClearVars(sc);
  }

  return (ret);
}

/* convert_ascii -> replace occurences of <ASCII> with the appropriate
 *		    hex values that correspond to the ASCII
 */
static char *convert_ascii(char *inbuf,char *outbuf)
{
  char *p1,*p2,*uout;
  int have_ascii;
  int len;
  char ch;
  
  /* ignore silly things */
  if ((inbuf==NULL)||(outbuf==NULL))
    return (inbuf);

  have_ascii=0;
  p1=inbuf;
  uout=outbuf;
  while((p2=strchr(p1,'<'))!=NULL) {
    have_ascii=1;

    /* copy from start of search until the '<' into the output buffer */
    len=(p2-p1);
    memcpy(uout,p1,len);
    uout+=len;

    /* find the end of the block */
    p2++;
    while (*p2) {
      if (*p2=='\\') {
	p2++;
	ch=*(p2++);
	/* allow for "special" interpretation of certain chars */
	switch(ch) {
	  case '0': ch='\0'; break;
	  case 'n': ch='\n'; break;
	  case 'r': ch='\r'; break;
	  /* it is useful to be able to add 0xff when doing
	   * padding stuff ... which is a bit unusual in terms
	   * of the way we are doing this but it is handy!
	   */
	  case 'z': ch=(char)0xff; break;
	  break;
	}
      } else if (*p2=='>') {
	/* we have hit the end of the block */
	p1=p2+1;
	break;
      } else {
	ch=*(p2++);
      }
      /* covert ascii into hex */
      SC_BufferToHex(&ch,1,uout,4,0);
      uout+=2;
    }
    *uout='\0';
  }

  if (have_ascii) {
    /* now copy the remainder across too */
    while (*p1) {
      *uout++=*p1++;
    }
    return(outbuf);
  } else
    return(inbuf);
}

#define P_START '{'
#define P_END   '}'

/* convert_vars -> replace occurences of {pattern} with the appropriate
 *		   values of the variables that are referenced in the pattern
 */
static char *convert_vars(SC_STATE *sc, char *inbuf,char *outbuf)
{
  char *p,*p1,*p2,*p3,*p4,*uout;
  int have_vars,have_a_var;
  int len;
  char varname[32];
  char pbuf[64];
  int i,j,k,plen,match,vallen;
  SC_CMD_PDATA *pdata;
  
  /* ignore silly things */
  if ((inbuf==NULL)||(outbuf==NULL))
    return (inbuf);

  have_vars=0;
  p1=inbuf;
  uout=outbuf;
  while((p2=strchr(p1,P_START))!=NULL) {
    have_a_var=0;
    /* if it is ${VAR} then we need to handle that differently to {PATTERN} */
    if (p2>inbuf) {
      if (p2[-1]=='$') {
	have_a_var=1;
	p2--;
      }
    }
    have_vars=1;

    /* copy from start of search until the curly brace into the output buffer */
    len=(p2-p1);
    memcpy(uout,p1,len);
    uout+=len;
    *uout='\0';

    /* skip the $ as well (as we went back a byte to share the logic with
     * the pattern handling code) 
     */
    if (have_a_var)
      p2++;

    /* find the end of the block */
    p2++;
    plen=0;
    while (*p2) {
      if (*p2==P_END) {
	/* we have hit the end of the block */
	p1=p2+1;
	break;
      } else {
	pbuf[plen++]=*(p2++);
      }
    }
    /* terminate the string */
    pbuf[plen]='\0';

    /* handle simple var lookups now */
    if (have_a_var) {
      p=SC_GetVar(sc,pbuf);
      if (p==NULL) {
	if (sc->slog)
	  SLOG_printf(sc->slog,"convert_vars: failed to find VAR %s\n",pbuf);
	return (NULL);
      }

      /* copy the value */
      vallen=strlen(p);
      memcpy(uout,p,vallen);
      uout+=vallen;
      *uout='\0';

      /* now continue substituting values */
      continue;
    }

    /* should have the pattern name in pbuf - now find a match */
    match=0;
    for(i=0;i<N_cmd_table;i++) {
      if (sc->card_type==cmd_table[i].card_type) {
	for(k=0;k<(*cmd_table[i].N_pdata);k++) {
	  pdata=cmd_table[i].pdata+k;

	  if (STRCASECMP(pdata->name,pbuf)==0) {
	    if (sc->slog)
	      SLOG_printf(sc->slog,"convert_vars: MATCH on %s\n",pbuf);
	    match=1;

	    /* now get the value of each of the vars in the pattern 
	     * and insert them into the data stream 
	     */

	    p4=pdata->data;
	    while ((p3=getname(p4,&p4))!=NULL) {
	      /* should have var[:len] */
	      strcpy(varname,p3);

	      /* default length is 1 byte */
	      vallen=1;

	      /* now extract the length (if it is there) */
	      if ((p3=strchr(varname,':'))!=NULL) {
		*p3='\0';
		p3++;
		vallen=(int)strtol(p3,NULL,0);
	      }

	      /* now we have the name in varname ... get the value */
	      p=SC_GetVar(sc,varname);
	      if (p==NULL) {
		/* no value means we initialise to zeros ... which is
		 * why we actually need the length 
		 */
		for(j=0;j<vallen;j++) {
		  /* two "hex" bytes */
		  *uout++='0';
		  *uout++='0';
		}
	      } else {
		/* sanity check */
		if (strlen(p)!=(unsigned int)(vallen*2)) {
		  if (sc->slog)
		    SLOG_printf(sc->slog,"convert_vars: length mismatch on %s - %d instead of %d\n",varname,strlen(p),vallen*2);
		  return(NULL);
		}
		memcpy(uout,p,vallen*2);
		uout+=vallen*2;
		*uout='\0';
	      }
	    }
	  }
	}
      }
    }

    /* oops ... unknown pattern ... this is a serious error! */
    if (match==0) {
      if (sc->slog)
	SLOG_printf(sc->slog,"convert_vars: failed to find %s\n",pbuf);
      return (NULL);
    }
  }

  if (have_vars) {
    /* now copy the remainder across too */
    while (*p1) {
      *uout++=*p1++;
    }
    return(outbuf);
  } else
    return(inbuf);

}

int SC_ExecuteCommands(SC_STATE *sc,char **cmds,char *uout,int chain)
{
  int ret;

  if ((sc==NULL)||(cmds==NULL)||(uout==NULL))
    return 0;

  ret=1;
  /* loop through all the commands until NUL is seen */
  while (*cmds!=NULL) {
    /* execute a single command */
    if (sc->slog)
      SLOG_printf(sc->slog,"SC_ExecuteCommands: cmd %s\n",*cmds);
    ret=SC_ExecuteCommand(sc,*cmds,uout,chain);
    /* if it fails to execute we stop */
    if (!ret) {
      if (sc->slog)
	SLOG_printf(sc->slog,"SC_ExecuteCommands: cmd %s exec failed\n",*cmds);
      break;
    }
    /* if it doesn't succeed executing then we stop */
    if (sc->last_rsp.rsp_code!=SC_RSP_OK) {
      ret=(0);
      if (sc->slog)
	SLOG_printf(sc->slog,"SC_ExecuteCommands: cmd %s cmd failed %d (%s)\n",
	                *cmds,sc->last_rsp.rsp_code,
			SC_RSPCode2String(sc->last_rsp.rsp_code));
      break;
    }
    /* move to next cmd */
    cmds++;
  }
  return(ret);
}

