/* sc.c	- Smartcard related functions
 *
 * Copyright 1993-1997, Tim Hudson. All rights reserved.
 *
 * You can pretty much do what you like with this code except pretend that 
 * you wrote it provided that any derivative of this code includes the
 * above comments unchanged. If you put this in a product then attribution
 * is mandatory. See the details in the COPYING file.
 *
 * Tim Hudson
 * tjh@cryptsoft.com
 *
 */

#include "platform.h"

/* you can switch off the dependancy on the higher-level card
 * interpreting stuff that handles response code conversion
 * by commenting out the following define
 */
#define USE_RSP_CODE_CONVERT

#ifdef USE_STDIO
#include <stdio.h>
#endif /* USE_STDIO */

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "sio.h"
#include "sct0.h"
#include "sc.h"
#include "slog.h"

/* fwd decl */
int SC_GCR_Command(SC_STATE *sc,char *inbuf,int len,char *outbuf,
                  int maxsize,int delay);

int SC_PE_Command(SC_STATE *sc,char *inbuf,int len,char *outbuf,
                  int maxsize,int delay);

int SC_SCT_Command(SC_STATE *sc,char *inbuf,int nonhex, int len,char *outbuf,
                  int maxsize,int delay);


int SC_GetVersion(int *vmajor, int *vminor)
{
  (*vmajor)=SC_VERSION_MAJOR;
  (*vminor)=SC_VERSION_MINOR;
  return(1);
}

SC_STATE *SC_Init(SIO_INFO *s,int reader_type,int card_type,int probe,
                    SLOG *log)
{
  SC_STATE st, *stp;

  /* make sure we have sane parameters */
  if (s==NULL)
    return (NULL);

  /* start with a blank entry */
  memset(&st,'\0',sizeof(st));

  /* explicitly set things */
  st.sio=s;
  st.reader_type=reader_type;
  st.card_type=card_type;
  st.error_code=SC_ERR_NONE;
  st.slog=log;
  st.sout=NULL;
  st.N_vars=0;
  st.exe_depth=0;
  st.noexec=0;
  st.noparse=0;

  /* if the caller wants us to do some probing to get card details
   * then now is the right time to do it 
   */
  if (probe) {
    /* if the probe fails then we return an error in error_code and
     * the user can ignore or handle it as they see fit
     * (and yes I know this wipes out the error set in SC_GetATR 
     * but until I put an error stack in place it will have to 
     * do as that error is important to other callers of SC_GetATR)
     */
    if (!SC_GetATR(&st))
      st.error_code=SC_ERR_PROBE_FAILED;
  }

  /* copy things now that it has worked */
  stp=(SC_STATE *)malloc(sizeof(SC_STATE));
  if (stp==NULL)
    goto err;

  memcpy(stp,&st,sizeof(SC_STATE));
  return(stp);

err:; 
  return (NULL);
}

int SC_Final(SC_STATE *sc)
{
  /* do we need to do any finalisation stuff? perhaps later? */
  if (sc!=NULL) {
    /* we should power the card down on final so that the user
     * knows we have finished with it ... GCRs have an active
     * light on them so it does make a difference
     */
    if (sc->reader_type==SC_READER_GCR) {
      /* TODO */
    }
  }
  return(1);
}

int SC_SetLog(SC_STATE *sc,SLOG *slog)
{
  if (sc!=NULL) {
    if (sc->slog!=NULL)
      SLOG_close(sc->slog);
    sc->slog=slog;
  }
  return (1);
}

int SC_SetOutput(SC_STATE *sc,SLOG *sout)
{
  if (sc!=NULL) {
    if (sc->sout!=NULL)
      SLOG_close(sc->sout);
    sc->sout=sout;
  }
  return (1);
}

int SC_GetATR(SC_STATE *sc)
{
  SC_ATR *atr;
  char buf[SC_MAX_DATA_LEN];
  char origbuf[SC_MAX_DATA_LEN];
  int len,ret,i,hack,origlen;
  unsigned char ch;

  sc->error_code=0;
  if (sc->reader_type==SC_READER_DUMBMOUSE) {
    /* the dumbmouse is a simple RTS toggle and
     * the ATR is returned in a straight forward 
     * manner - easy (and sensible) stuff
     */
    if ((len=SCT0_GetATR(sc->sio,buf,sizeof(buf)))>0) {
      atr=SC_DecodeATR(buf,len);
      if (atr!=NULL) {
	/* it worked! */
	sc->atr=atr;
      } else {
	sc->error_code=SC_ERR_BAD_ATR;
	goto failed;
      }
    } else {
      goto failed;
    }
  } 

  if (sc->reader_type==SC_READER_GCR) {
    len=0;
    /* send POWER ON CARD */
    buf[len++]=0x6e;
    buf[len++]=0x01;
    buf[len++]=0x00;
    buf[len++]=0x00;

    /* now perform the command and get the response */
    ret=SC_GCR_Command(sc,buf,len,buf,sizeof(buf),1);
    if (!ret)
      goto failed;

    if (ret<5) {
      sc->error_code=SC_ERR_MSG_TOO_SHORT;
      goto failed;
    }

    /* now we have to handle the return format of the GCR reader
     * which seems to want to insert a rubbish byte in the middle
     * of the ATR which we have to parse out
     */
    len=(buf[5] & 0xff);

    memcpy(origbuf,buf,len+6);
    origlen=len;

    if (sc->slog) {
      SLOG_printf(sc->slog,"ATR ORIG\n");
      SLOG_dump(sc->slog,buf+6,len,0);
      SLOG_printf(sc->slog,"\n");
    }

    /* if we have the funny byte included then parsing it out 
     * sounds like a good idea to me ... this is a complete total
     * hack as there is obviously something about this reader that
     * I don't understand as what it is doing is seriously broken
     */
    hack=0;
    if ( (buf[6+2]==0x11) && (buf[6+2+1]==0x00) ) {
      if (sc->slog)
	SLOG_printf(sc->slog,"ATR HACK - removing 0x11 0x00\n");
      hack=1;
      for(i=2;i<len;i++)
	buf[6+i]=buf[6+i+2];
      len-=2;
      if (sc->slog) {
	SLOG_dump(sc->slog,buf+6,len,0);
	SLOG_printf(sc->slog,"\n");
      }
    }

    atr=SC_DecodeATR(buf+6,len);
    if (atr!=NULL) {
      /* it worked! */
      sc->atr=atr;
    } else {
      /* if we didn't play silly buggers with the ATR then it
       * really is BAD
       */
      if (!hack) {
	sc->error_code=SC_ERR_BAD_ATR;
	goto failed;
      }

      /* restore the buffer and try without 0x11 0x00 */
      len=origlen;
      memcpy(buf,origbuf,len+6);

      if (sc->slog) {
	SLOG_printf(sc->slog,"ATR ORIG\n");
	SLOG_dump(sc->slog,buf+6,len,0);
	SLOG_printf(sc->slog,"\n");
      }

      /* if we have the funny byte included then parsing it out 
       * sounds like a good idea to me ... this is a complete total
       * hack as there is obviously something about this reader that
       * I don't understand as what it is doing is seriously broken
       */
      hack=0;
      if (buf[6+2]==0x11) {
	if (sc->slog)
	  SLOG_printf(sc->slog,"ATR HACK - removing 0x11\n");
	hack=1;
	for(i=2;i<len;i++)
	  buf[6+i]=buf[6+i+1];
	len--;
	if (sc->slog) {
	  SLOG_dump(sc->slog,buf+6,len,0);
	  SLOG_printf(sc->slog,"\n");
	}
      }

      if (hack) {
        atr=SC_DecodeATR(buf+6,len);
	if (atr!=NULL) {
	  sc->atr=atr;
	} else {

	  /* restore the buffer and try it in the original form */
	  len=origlen;
	  memcpy(buf,origbuf,len+6);

	  if (sc->slog)
	    SLOG_printf(sc->slog,"ATR HACK - trying original\n");
	  if (sc->slog) {
	    SLOG_dump(sc->slog,buf+6,len,0);
	    SLOG_printf(sc->slog,"\n");
	  }

	  atr=SC_DecodeATR(buf+6,len);
	  if (atr!=NULL) {
	    sc->atr=atr;
	  } else {
	    sc->error_code=SC_ERR_BAD_ATR;
	    goto failed;
	  }
	}
      } else {
	sc->error_code=SC_ERR_BAD_ATR;
	goto failed;
      }
    }

  }

  if (sc->reader_type==SC_READER_PE) {
    len=0;

    /* send POWER ON CARD */
    buf[len++]=0x00;
    buf[len++]=0x60;
    buf[len++]=0x00;
    buf[len++]=0x01;
    buf[len++]=0x00;

    /* now perform the command and get the response */
    ret=SC_PE_Command(sc,buf,len,buf,sizeof(buf),1);
    if (!ret)
      goto failed;

    if (ret<4) {
      sc->error_code=SC_ERR_MSG_TOO_SHORT;
      goto failed;
    }

    /* now we have to handle the return format of the GCR reader
     * which seems to want to insert a rubbish byte in the middle
     * of the ATR which we have to parse out
     */
    len=(buf[2] & 0xff);

    memcpy(origbuf,buf,len+3);
    origlen=len;

    if (sc->slog) {
      SLOG_printf(sc->slog,"ATR ORIG\n");
      SLOG_dump(sc->slog,buf+3,len,0);
      SLOG_printf(sc->slog,"\n");
    }

    /* something two bytes long is an error indication rather than
     * a card ATR response
     */
    if (len==2) {
      sc->error_code=SC_ERR_PROBE_FAILED;
      goto failed;
    }

    /* translate the simple mapping on non-reversed byte info
     * back into what it should be
     */
    ch=(buf[3+0] & 0xff);
    if ((ch!=0x3b) && (ch!=0x3f)) {
      /* convert from native to translated */
      if (ch==0xdc)
	buf[3+0]=0x3b;
      else if (ch==0xc0)
	buf[3+0]=0x3f;
    }

    atr=SC_DecodeATR(buf+3,len);
    if (atr!=NULL) {
      /* it worked! */
      sc->atr=atr;
    } else {
      sc->error_code=SC_ERR_BAD_ATR;
      goto failed;
    }
  }

  if (sc->reader_type==SC_READER_SCT) {
    len=0;

    /* send General STatus */
    buf[len++]='S';
    buf[len++]='T';

    /* now perform the command and get the response */
    ret=SC_SCT_Command(sc,buf,2,len,buf,sizeof(buf),1);
    if (!ret)
      goto failed;

    if (ret<2) {
      sc->error_code=SC_ERR_MSG_TOO_SHORT;
      goto failed;
    }

    if (sc->slog) {
      SLOG_printf(sc->slog,"STATUS\n");
      SLOG_dump(sc->slog,buf,ret,0);
      SLOG_printf(sc->slog,"\n");
    }

    /* what comes back is CIS RPS RFU <DATA> ME1 ME2 ETX CHK */
    if (buf[1]!=0x30) {
      sc->error_code=SC_ERR_PROBE_FAILED;
      goto failed;
    }

    /* should get {00,02} 30 00 or something like that back */

    /* now we know the reader is operational we ask for the card ATR */
    len=0;

    /* send Select Handler */
    buf[len++]='S';
    buf[len++]='H';
    buf[len++]=' ';
    buf[len++]=0x05;    /* asynch card */
    buf[len++]=0x10;    /* read up to 16 bytes of data from card */

    /* now perform the command and get the response */
    ret=SC_SCT_Command(sc,buf,3,len,buf,sizeof(buf),1);
    if (!ret)
      goto failed;

    if (ret<2) {
      sc->error_code=SC_ERR_MSG_TOO_SHORT;
      goto failed;
    }

    if (sc->slog) {
      SLOG_printf(sc->slog,"ATR DATA\n");
      SLOG_dump(sc->slog,buf,ret,0);
      SLOG_printf(sc->slog,"\n");
    }

    len=ret-3;

    /* what comes back is CIS RPS RFU <DATA> ME1 ME2 ETX CHK */
    if (buf[1]!=0x30) {
      sc->error_code=SC_ERR_PROBE_FAILED;
      goto failed;
    }

    atr=SC_DecodeATR(buf+3,len);
    if (atr!=NULL) {
      /* it worked! */
      sc->atr=atr;
    } else {
      sc->error_code=SC_ERR_BAD_ATR;
      goto failed;
    }
  }

  return(1);

failed: ;
  /* there was an error ... */
  if (sc->error_code==SC_ERR_NONE)
    sc->error_code=SC_ERR_UNSPECIFIED;
  return(0);
}


int SC_DoAsciiCommand(SC_STATE *sc,int direction,char *uin,char *uout)
{
  unsigned char inbuf[SC_MAX_DATA_LEN];
  unsigned char outbuf[SC_MAX_DATA_LEN];
  int inlen,outlen,i;
  SC_CMD cmd;
  SC_RSP rsp;

  /* convert in command into a raw buffer */
  inlen=SC_HexToBuffer(uin,inbuf,sizeof(inbuf));
  if (inlen<=0) {
    sc->error_code=SC_ERR_BAD_CMD_FORMAT;
    return(0);
  }

  /* we have to have at least 5 bytes or the command is not
   * anything valid to send so we log that explicitly
   */
  if (inlen<5) {
    sc->error_code=SC_ERR_BAD_CMD_TOO_SHORT;
    return(0);
  }

  /* now pull this appart into the right things */
  memset(&cmd,'\0',sizeof(cmd));
  cmd.CLA=inbuf[0];
  cmd.INS=inbuf[1];
  cmd.P1=inbuf[2];
  cmd.P2=inbuf[3];
  /*
  cmd.Lc=(int)(unsigned char)(inbuf[4] & 0xff);
  */
  cmd.Lc=inbuf[4];

  if (direction==SC_DIR_IN) {
    /* now the rest of the data is "data" except perhaps the 
     * last byte which may exist and if so it is the expected
     * length of the response
     */
    if ((inlen-5-cmd.Lc)<0) {
      sc->error_code=SC_ERR_BAD_CMD_DATA_MISSING;
      return(0);
    } 
    memcpy(cmd.data,inbuf+5,cmd.Lc);

    /* if this is an "IN" command then it has data that is passed and
     * so if it isn't there then the user has sent the wrong stuff so
     * we complain without sending the command to the reader
     */
    if ((inlen-5-cmd.Lc)>1) {
      sc->error_code=SC_ERR_BAD_CMD_DATA_MISSING;
      return(0);
    }

    /* copy Le if it is present */
    if ((inlen-5-cmd.Lc)==1)
      cmd.Le=inbuf[5+cmd.Lc] & 0xff;
  }

  /* now use the "normal" command processor */
  if (SC_DoCommand(sc,direction,&cmd,&rsp)) {
    /* now convert the RSP into a straight hex string */
    sprintf(outbuf,"%02x%02x%02x",rsp.rsp_code & 0xff,
                      rsp.SW1 & 0xff,rsp.SW2 & 0xff);
    outlen=strlen(outbuf);

    /* if we have data then we tack the length and the data on 
     * then end of the output
     */
    if (rsp.len>0) {
      sprintf(outbuf+outlen,"%02x",rsp.len & 0xff);
      outlen+=2;
      i=SC_BufferToHex(rsp.data,rsp.len,outbuf+outlen,
                              sizeof(outbuf)-outlen,0);
      if (i!=-1) {
	if (sc->slog)
	  SLOG_printf(sc->slog,"ASCII: len=%d outbuf %s\n",i,outbuf+outlen);
        outlen+=i;
      }
    } 

    if ((outlen>0) && (outlen<512)) {
      memcpy(uout,outbuf,outlen+1);
      return(outlen);
    } else
      return(0);
  } else {
    return(0);
  }
}

int SC_DoCommand(SC_STATE *sc,int direction, SC_CMD *uin, SC_RSP *uout)
{
  int j,len,newlen;
  unsigned char thebuf[BUFSIZ];
  unsigned char *buf=thebuf;
  SIO_INFO *s;
  int ret;

  /* sanity check */
  if (sc==NULL)
    return(0);

  /* clear the output ... we really don't want to leave
   * rubbish around even if it is the users rubbish
   */
  memset(uout,'\0',sizeof(SC_RSP));

  sc->error_code=SC_ERR_NONE;
  s=sc->sio;

  /* bypass command execution if that is what the
   * user wants ... probably means we are testing
   * command parsing etc
   */
  if (sc->noexec) 
    return (0);

  if (sc->reader_type==SC_READER_DUMBMOUSE) {
    /* flush data ... read until nothing available */
    while (SIO_ReadChar(s)!=-1)
      ;

    /* send out the first bit of the command */
    buf[0]=uin->CLA;
    buf[1]=uin->INS;
    buf[2]=uin->P1;
    buf[3]=uin->P2;
    buf[4]=(uin->Lc & 0xff);
    len=SIO_WriteBuffer(s,buf,5);

    /* check that it worked */
    if (len!=5) {
      sc->error_code=SC_ERR_WRITE_FAILED;
      goto err;
    }

    /* we need to delay before reading to give the reader time
     * to start responding
     */
    SIO_Delay(s,100);

    /* now read in some data - which should be the INS */
    for(j=0;j<2;j++) {
      len=SIO_ReadBuffer(s,buf,sizeof(thebuf));
      if (sc->slog) {
        SLOG_printf(sc->slog,"SC_DoCommand: read %d\n",len);
	if (len>0) {
	  SLOG_dump(sc->slog,buf,len,0);
	  SLOG_printf(sc->slog,"\n");
	}
      }
      if (len>0)
	break;
    }

    /* if we didn't get back the INS byte then something
     * went wrong ... 
     */

    if (len<=0) {
      sc->error_code=SC_ERR_NO_REPLY;
      return (0);
    }

    /* we must get the INS back or things are broken! */
    if (buf[0]!=uin->INS) {
      if (len==2) {
        /* we probably have SW1 and SW2 which we can report
         * to the user to sort out 
         */
        goto handle_reply;
      }
      sc->error_code=SC_ERR_NO_INS_ON_REPLY;
      return (0);
    }

    /* if this is an "IN" command then it has data that is passed 
     * and we need to send it now 
     */
    if (direction==SC_DIR_IN) {
      /* now send the data if there is any */
      if (uin->Lc>0) {
	len=SIO_WriteBuffer(s,uin->data,uin->Lc);
	/* check that it worked */
	if (len!=uin->Lc) {
	  sc->error_code=SC_ERR_WRITE_FAILED;
	  goto err;
	}
      } else {
	/* I guess sending out the Le now would be sensible and
	 * is probably required for commands that have no optional
	 * parameters?
	 */
	len=SIO_WriteChar(s,'\0');
	/* check that it worked */
	if (len!=1) {
	  sc->error_code=SC_ERR_WRITE_FAILED;
	  goto err;
	}
      }
      len=0;
    } else {
      /* it is normal to have all the data available now ... */
      if (len>1) {
	buf++;
        len--;
      } else {
        len=0;
      }
    }

    /* we need to delay before reading to give the reader time
     * to start responding
     */
    SIO_Delay(s,100);

    /* now read in some data - which should be the response code 
     * followed by command specific response data
     */
    for(j=0;j<2;j++) {
      newlen=SIO_ReadBuffer(s,buf+len,sizeof(thebuf)-len);
      if (sc->slog) {
        SLOG_printf(sc->slog,"SC_DoCommand: read %d\n",newlen);
	if (newlen>0) {
	  SLOG_dump(sc->slog,buf+len,newlen,0);
	  SLOG_printf(sc->slog,"\n");
	}
      }
      if (newlen>0)
        len+=newlen;
      SIO_Delay(s,100);
    }

    if (len<2) {
      sc->error_code=SC_ERR_NO_REPLY;
      return (0);
    }

handle_reply: ;
    /* now copy the data to the output buffer */
    if (direction==SC_DIR_IN) {
      /* we have SW1 SW2 and nothing else? */
      uout->SW1=buf[0];
      uout->SW2=buf[1];
      uout->len=(len-2);
      if (uout->len>0) 
	memcpy(uout->data,buf+2,uout->len);
    } else {
      /* we have the data response followed by SW1 SW2 */
      uout->len=(len-2);
      if (uout->len>0) 
	memcpy(uout->data,buf,uout->len);
      uout->SW1=buf[uout->len];
      uout->SW2=buf[uout->len+1];
    }

#ifdef USE_RSP_CODE_CONVERT
    uout->rsp_code=SC_SW1SW2_RSPCode(sc,uout->SW1,uout->SW2);
#else /* !USE_RSP_CODE_CONVERT */
    uout->rsp_code=SC_RSP_UNKNOWN;
#endif /* USE_RSP_CODE_CONVERT */

  } else if (sc->reader_type==SC_READER_GCR) {
    /* things are a little more complex than the DUMBMOUSE 
     * but basically we assemble the binary of the message here
     * and then pass the problem off to the GCR message handler
     * which will convert to hex and frame and get the reply
     * and decode it back to binary
     */
    len=0;
    if (direction==SC_DIR_IN)
      buf[len++]=0xda;
    if (direction==SC_DIR_OUT)
      buf[len++]=0xdb;
    buf[len++]=uin->CLA;
    buf[len++]=uin->INS;
    buf[len++]=uin->P1;
    buf[len++]=uin->P2;
    buf[len++]=(uin->Lc & 0xff);
    /* input commands have data to send */
    if (direction==SC_DIR_IN) {
      memcpy(buf+len,uin->data,uin->Lc);
      len+=uin->Lc;
    }

    /* now perform the command and get the response */
    ret=SC_GCR_Command(sc,buf,len,buf,sizeof(thebuf),0);
    if (!ret)
      goto err;

    /* all replies have to be five bytes or longer */
    if (ret<5) {
      sc->error_code=SC_ERR_MSG_TOO_SHORT;
      goto err;
    }

    /* check the packaging of the message */
    if ((buf[0]!=0x60)||(buf[ret-1]!=0x03)) {
      sc->error_code=SC_ERR_MSG_BAD;
      goto err;
    }

    /* extract the message body length */
    len=buf[1] & 0xff;

    /* convert into "normal" terms */
    buf+=3;
    len--;
    goto handle_reply;

#if 0
    /* now extract the status code */
    if ((buf[2]==0x00)||(buf[2]==0xe7)) {
      uout->SW1=buf[3];
      uout->SW2=buf[4];
    } else {
      /* we make something up I guess */
      uout->SW1=0xff;
      uout->SW2=0xff;
    }

    /* copy the data */
    if (uout->len>0) 
      memcpy(uout->data,buf+5,uout->len);
#endif

  } else if (sc->reader_type==SC_READER_PE) {
    /* things are a little more complex than the DUMBMOUSE 
     * but basically we assemble the binary of the message here
     * and then pass the problem off to the PE message handler
     * which will convert to hex and frame and get the reply
     * and decode it back to binary
     */
    len=0;
    buf[len++]=uin->CLA;
    buf[len++]=uin->INS;
    buf[len++]=uin->P1;
    buf[len++]=uin->P2;
    buf[len++]=(uin->Lc & 0xff);

    /* input commands have data to send */
    if (direction==SC_DIR_IN) {
      memcpy(buf+len,uin->data,uin->Lc);
      len+=uin->Lc;
    }

    /* now perform the command and get the response */
    ret=SC_PE_Command(sc,buf,len,buf,sizeof(thebuf),0);
    if (!ret)
      goto err;

    /* all replies have to be four bytes or longer */
    if (ret<4) {
      sc->error_code=SC_ERR_MSG_TOO_SHORT;
      goto err;
    }

    /* extract the message body length */
    len=buf[2] & 0xff;

    /* convert into "normal" terms */
    buf+=3;
    goto handle_reply;

  } else if (sc->reader_type==SC_READER_SCT) {
    /* things are a little more complex than the DUMBMOUSE 
     * but basically we assemble the binary of the message here
     * and then pass the problem off to the SCT message handler
     * which will convert to hex and frame and get the reply
     * and decode it back to binary
     */
    len=0;
    buf[len++]='I';
    buf[len++]='S';
    buf[len++]=' ';
    buf[len++]='D';
    if (direction==SC_DIR_IN) {
      buf[len++]='A';
    } else {
      buf[len++]='B';
    }
    buf[len++]=uin->CLA;
    buf[len++]=uin->INS;
    buf[len++]=uin->P1;
    buf[len++]=uin->P2;
    buf[len++]=(uin->Lc & 0xff);

    /* input commands have data to send */
    if (direction==SC_DIR_IN) {
      memcpy(buf+len,uin->data,uin->Lc);
      len+=uin->Lc;
    }

    /* now perform the command and get the response */
    ret=SC_SCT_Command(sc,buf,5,len,buf,sizeof(thebuf),1);
    if (!ret)
      goto err;

    /* all replies have to be four bytes or longer */
    if (ret<4) {
      sc->error_code=SC_ERR_MSG_TOO_SHORT;
      goto err;
    }

    /* convert into "normal" terms */
    buf+=3;
    len=ret-3;

    goto handle_reply;
  } else {
    goto err;
  }

  /* keep a copy of the result of the last command */
  memcpy(&sc->last_rsp,uout,sizeof(SC_RSP));

  /* it worked */
  return(1);

err: ;
  /* it failed */
  return(0);
}

/* SC_HexToRSP -> convert ASCII hex buffer into binary SC_RSP format */
int SC_HexToRSP(char *inbuf, SC_RSP *uout)
{
  char SW1[3];
  char SW2[3];
  char len[3];
  char tmpbuf[SC_MAX_DATA_LEN];
  int inlen,l;

  /* protect from silly usage */
  if ((inbuf==NULL)||(uout==NULL))
    return 0;

  /* clean slate to start with */
  memset(uout,'\0',sizeof(SC_RSP));
  SW1[0]='\0';
  SW2[0]='\0';
  len[0]='\0';

  inlen=strlen(inbuf);

  /* must have at least SW1 and SW2 */
  if (inlen<4)
    goto err;

  /* now do the conversion - should have SW1 SW2 [LEN DATA] */
  strncpy(SW1,inbuf,2);
  SW1[2]='\0';

  strncpy(SW2,inbuf+2,2);
  SW2[2]='\0';

  if (inlen>=6) {
    strncpy(len,inbuf+4,2);
    len[2]='\0';
  }

  /* now convert each in turn */
  l=SC_HexToBuffer(SW1,tmpbuf,sizeof(tmpbuf));
  if (l!=1)
    goto err;
  uout->SW1=tmpbuf[0];
  l=SC_HexToBuffer(SW2,tmpbuf,sizeof(tmpbuf));
  if (l!=1)
    goto err;
  uout->SW2=tmpbuf[0];
  l=SC_HexToBuffer(len,tmpbuf,sizeof(tmpbuf));
  if (l!=1)
    goto err;
  uout->len=(int)tmpbuf[0];

  if (uout->len>0) {
    l=SC_HexToBuffer(inbuf+6,tmpbuf,sizeof(tmpbuf));
    if (l!=uout->len)
      goto err;
  }

  return (1);
err: ;
  return (0);
}

int SC_GCR_Command(SC_STATE *sc,char *inbuf,int inlen,char *outbuf,
                    int maxsize,int delay)
{
  char buf[SC_MAX_HEX_DATA_LEN];
  char tmpbuf[3];
  unsigned long chk;
  int len,addlen,wlen,i,j,newlen;
  SIO_INFO *s;

  /* lazy */
  s=sc->sio;

  if (sc->slog) {
    SLOG_printf(sc->slog,"SC_GCR_Command:\n");
    if (inlen>0) {
      SLOG_dump(sc->slog,inbuf,inlen,0);
      SLOG_printf(sc->slog,"\n");
    }
  }

  len=0;
  chk=0;

  /* add the message header */
  tmpbuf[0]=0x60;
  tmpbuf[1]=(inlen & 0xff);
  chk^=tmpbuf[0];
  chk^=tmpbuf[1];

  /* convert to hex */
  len=SC_BufferToHex(tmpbuf,2,buf,sizeof(buf),0);

  /* add in the data */
  addlen=SC_BufferToHex(inbuf,inlen,buf+len,sizeof(buf)-len,0);
  len+=addlen;

  /* add the data into the checksum */
  for(i=0;i<inlen;i++)
    chk^=inbuf[i];

  /* add the checksum */
  tmpbuf[0]=(char)(chk & 0xff);
  addlen=SC_BufferToHex(tmpbuf,1,buf+len,sizeof(buf)-len,0);
  len+=addlen;

  /* finally add the raw (non-hex) message terminator */
  buf[len++]=0x03;

  if (sc->slog) {
    SLOG_printf(sc->slog,"SC_GCR_Command: RAW\n");
    if (len>0) {
      SLOG_dump(sc->slog,buf,len,0);
      SLOG_printf(sc->slog,"\n");
    }
  }

  /* the smallest valid command is at least seven bytes long */
  if (len<7) {
    sc->error_code=SC_ERR_MSG_TOO_SHORT;
    goto err;
  }

  /* now we have a command we can send ... the usual serial transfer
   * dance can begin
   */

  /* flush data ... read until nothing available */
  while (SIO_ReadChar(s)!=-1)
    ;

  /* write the bulk of it out */
  wlen=SIO_WriteBuffer(s,buf,len);

  /* check that it worked */
  if (wlen!=len) {
    sc->error_code=SC_ERR_WRITE_FAILED;
    goto err;
  }

  /* now read in the reply ... */
  len=0;
  for(j=0;j<2;j++) {
    newlen=SIO_ReadBuffer(s,buf+len,sizeof(buf)-len);
    if (sc->slog) {
      SLOG_printf(sc->slog,"SC_GCR_Command: read %d\n",newlen);
      if (newlen>0) {
	SLOG_dump(sc->slog,buf+len,newlen,0);
	SLOG_printf(sc->slog,"\n");
      }
    }
    if (newlen>0)
      len+=newlen;
    SIO_Delay(s,100);

    /* if we have no data then we need a decent delay */
    if ((len==0)&&(delay)) {
      SIO_Delay(s,SIO_READ_WAIT_DEFAULT);
    }

  }

  /* the smallest valid command is at least seven bytes long */
  if (len<7) {
    sc->error_code=SC_ERR_MSG_TOO_SHORT;
    goto err;
  }

  /* process the reply turning the HEX back into ASCII */
  if (buf[len-1]!=0x03) {
    sc->error_code=SC_ERR_MSG_TRAILER_INVALID;
    goto err;
  }

  /* change trailer to NUL to terminate the string */
  buf[len-1]='\0';
  /* convert to binary */
  i=SC_HexToBuffer(buf,outbuf,maxsize);
  if (i>0) {
    /* put the trailer back */
    outbuf[i++]=0x03;
  }

  if (sc->slog) {
    SLOG_printf(sc->slog,"SC_GCR_Command: OUTPUT\n");
    if (i>0) {
      SLOG_dump(sc->slog,outbuf,i,0);
      SLOG_printf(sc->slog,"\n");
    }
  }

  return (i);

err: ;
  return (0);
}

int SC_PE_Command(SC_STATE *sc,char *inbuf,int inlen,char *outbuf,
                    int maxsize,int delay)
{
  char buf[SC_MAX_HEX_DATA_LEN];
  int len,wlen,j,newlen;
  SIO_INFO *s;

  /* lazy */
  s=sc->sio;

  if (sc->slog) {
    SLOG_printf(sc->slog,"SC_PE_Command:\n");
    if (inlen>0) {
      SLOG_dump(sc->slog,inbuf,inlen,0);
      SLOG_printf(sc->slog,"\n");
    }
  }

  len=0;

  /* for the moment we hardwire device address details for the PE122 
   * which we could generalise for the whole PE series but I cannot
   * be bothered with that at the moment
   */
  buf[len++]=SC_READER_PE_DEVICE_ID;
  buf[len++]=0x00;
  buf[len++]=(inlen & 0xff);

  /* copy the data on to the end */
  memcpy(buf+len,inbuf,inlen);
  len+=inlen;

  if (sc->slog) {
    SLOG_printf(sc->slog,"SC_PE_Command: RAW\n");
    if (len>0) {
      SLOG_dump(sc->slog,buf,len,0);
      SLOG_printf(sc->slog,"\n");
    }
  }

  /* the smallest valid command is at least four bytes long */
  if (len<4) {
    sc->error_code=SC_ERR_MSG_TOO_SHORT;
    goto err;
  }

  /* now we have a command we can send ... the usual serial transfer
   * dance can begin
   */

  /* flush data ... read until nothing available */
  while (SIO_ReadChar(s)!=-1)
    ;

  /* write the bulk of it out */
  wlen=SIO_WriteBuffer(s,buf,len);

  /* check that it worked */
  if (wlen!=len) {
    sc->error_code=SC_ERR_WRITE_FAILED;
    goto err;
  }

  /* the reader needs quite a bit of time to wake
   * up and respond it seems
   */
  SIO_Delay(s,SIO_READ_WAIT_DEFAULT);

  /* now read in the reply ... */
  len=0;
  for(j=0;j<2;j++) {
    newlen=SIO_ReadBuffer(s,buf+len,sizeof(buf)-len);
    if (sc->slog) {
      SLOG_printf(sc->slog,"SC_PE_Command: read %d\n",newlen);
      if (newlen>0) {
	SLOG_dump(sc->slog,buf+len,newlen,0);
	SLOG_printf(sc->slog,"\n");
      }
    }
    if (newlen>0)
      len+=newlen;
    SIO_Delay(s,100);

    /* if we have no data then we need a decent delay */
    if ((len==0)&&(delay)) {
      SIO_Delay(s,SIO_READ_WAIT_DEFAULT);
    }

  }

  /* put things into the right place */
  memcpy(outbuf,buf,len);

  /* the smallest valid command is at least four bytes long */
  if (len<4) {
    sc->error_code=SC_ERR_MSG_TOO_SHORT;
    goto err;
  }

  /* process the reply turning the HEX back into ASCII */
  if (buf[0]!=SC_READER_PE_DEVICE_ID) {
    sc->error_code=SC_ERR_MSG_BAD;
    goto err;
  }

  if (sc->slog) {
    SLOG_printf(sc->slog,"SC_PE_Command: OUTPUT\n");
    if (len>0) {
      SLOG_dump(sc->slog,outbuf,len,0);
      SLOG_printf(sc->slog,"\n");
    }
  }

  return (len);

err: ;
  return (0);
}

int SC_SCT_Command(SC_STATE *sc,char *inbuf,int nonhex,int inlen,char *outbuf,
                    int maxsize,int delay)
{
  char buf[SC_MAX_HEX_DATA_LEN];
  unsigned long chk;
  int len,addlen,wlen,i,j,newlen;
  SIO_INFO *s;

  /* lazy */
  s=sc->sio;

  if (sc->slog) {
    SLOG_printf(sc->slog,"SC_SCT_Command:\n");
    if (inlen>0) {
      SLOG_dump(sc->slog,inbuf,inlen,0);
      SLOG_printf(sc->slog,"\n");
    }
  }

  len=0;
  chk=0;

  /* header STX */
  buf[len++]=0x02;

  /* copy non-hex bytes as is */
  for(i=0;i<nonhex;i++) {
    buf[len++]=inbuf[i];
  }

  /* adjust for non-hex bytes */
  inbuf+=nonhex;
  inlen-=nonhex;

  /* add in the data */
  addlen=SC_BufferToHex(inbuf,inlen,buf+len,sizeof(buf)-len,0);
  len+=addlen;

  /* add the trailer ETX */
  buf[len++]=0x03;

  /* add the data into the checksum */
  for(i=1;i<(len-1);i++)
    chk^=buf[i];

  /* add the checksum */
  buf[len++]=(char)(chk & 0xff);

  if (sc->slog) {
    SLOG_printf(sc->slog,"SC_SCT_Command: RAW\n");
    if (len>0) {
      SLOG_dump(sc->slog,buf,len,0);
      SLOG_printf(sc->slog,"\n");
      SLOG_dump(sc->slog,buf,len,1);
      SLOG_printf(sc->slog,"\n");
    }
  }

  /* the smallest valid command is at least four bytes long */
  if (len<4) {
    sc->error_code=SC_ERR_MSG_TOO_SHORT;
    goto err;
  }

  /* now we have a command we can send ... the usual serial transfer
   * dance can begin
   */

  /* flush data ... read until nothing available */
  while (SIO_ReadChar(s)!=-1)
    ;

  /* write the bulk of it out */
  wlen=SIO_WriteBuffer(s,buf,len);

  /* check that it worked */
  if (wlen!=len) {
    sc->error_code=SC_ERR_WRITE_FAILED;
    goto err;
  }

  /* now read in the reply ... */
  len=0;
  for(j=0;j<4;j++) {
    newlen=SIO_ReadBuffer(s,buf+len,sizeof(buf)-len);
    if (sc->slog) {
      SLOG_printf(sc->slog,"SC_SCT_Command: read %d\n",newlen);
      if (newlen>0) {
	SLOG_dump(sc->slog,buf+len,newlen,0);
	SLOG_printf(sc->slog,"\n");
      }
    }
    if (newlen>0)
      len+=newlen;
    SIO_Delay(s,100);

    /* if we have no data then we need a decent delay */
    if ((len==0)&&(delay)) {
      SIO_Delay(s,2*SIO_READ_WAIT_DEFAULT);
    }

  }

  /* the smallest valid command is at least four bytes long */
  if (len<3) {
    sc->error_code=SC_ERR_MSG_TOO_SHORT;
    goto err;
  }

  /* process the reply turning the HEX back into ASCII */

  if (sc->slog) {
    SLOG_printf(sc->slog,"SC_SCT_Command: READ RAW\n");
    if (len>0) {
      SLOG_dump(sc->slog,buf,len,0);
      SLOG_printf(sc->slog,"\n");
    }
  }

  /* check the header */
  if (buf[0]!=0x02) {
    sc->error_code=SC_ERR_MSG_HEADER_INVALID;
    goto err;
  }

  /* check the trailer */
  if (buf[len-2]!=0x03) {
    sc->error_code=SC_ERR_MSG_TRAILER_INVALID;
    goto err;
  }

  /* change trailer to NUL to terminate the string */
  buf[len-2]='\0';

  /* convert to binary */
  i=SC_HexToBuffer(buf+1,outbuf,maxsize);

  if (sc->slog) {
    SLOG_printf(sc->slog,"SC_SCT_Command: OUTPUT\n");
    if (i>0) {
      SLOG_dump(sc->slog,outbuf,i,0);
      SLOG_printf(sc->slog,"\n");
      SLOG_dump(sc->slog,outbuf,i,1);
      SLOG_printf(sc->slog,"\n");
    }
  }

  return (i);

err: ;
  return (0);
}


int SC_SetTestMode(SC_STATE *sc,int testval)
{
  if (sc==NULL)
    return (0);
  sc->noexec=testval;
  return (1);
}

int SC_SetParseMode(SC_STATE *sc,int parseval)
{
  if (sc==NULL)
    return (0);
  sc->noparse=parseval;
  return (1);
}

