/* sct0.c	- Smartcard T=0 related functions
 *
 * Copyright 1993-1997, Tim Hudson. All rights reserved.
 *
 * You can pretty much do what you like with this code except pretend that 
 * you wrote it provided that any derivative of this code includes the
 * above comments unchanged. If you put this in a product then attribution
 * is mandatory. See the details in the COPYING file.
 *
 * Tim Hudson
 * tjh@cryptsoft.com
 *
 */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "sio.h"
#include "sct0.h"

static int dbglev=0;
static SLOG *outf=NULL;

/* max number of times we retry a probe when we don't get back
 * something that is a valid response 
 */
#define MAX_RETRIES  2

int SCT0_SetDebug(int lev)
{
  int oldlev;

  oldlev=dbglev;
  dbglev=lev;
  return(oldlev);
}

int SCT0_GetDebug(void)
{
  return(dbglev);
}

int SCT0_SetDebugLog(SLOG *slog)
{
  if (outf!=NULL)
    SLOG_close(outf);
  outf=slog;
  return(1);
}

int SCT0_Reset(SIO_INFO *s)
{
  int ch;

  /* flush any data in the buffer - this is important */
  while ((ch=SIO_ReadChar(s))!=-1)
    ;

  /* toggle RTS to trigger reset */
#if 0
  SIO_DropRTS(s);
  SIO_Delay(s,25);
  SIO_RaiseRTS(s);
  SIO_Delay(s,25);
#endif
#if 1
  SIO_RaiseRTS(s);
  SIO_Delay(s,25);
  SIO_DropRTS(s);
  SIO_Delay(s,25);
#endif

  return(1);
}

int SCT0_GetATR(SIO_INFO *s,char *buf,int bufsize)
{
  int i,n,ch;

  SCT0_Reset(s);

  /* now read in whatever data is available */
  n=0;
  /* we have two shots at doing this ... some cards are a little 
   * slow at responding 
   */
  for(i=0;i<2;i++) {
    while ((ch=SIO_ReadChar(s))!=-1) {
      if (buf!=NULL) {
	if (n>=bufsize)
	  break;
	buf[n++]=(ch & 0xff);
      }
    }
  }
  return(n);
}

/* we maintain a table of settings that it makes sense to scan through
 * when looking for the communication settings for use with a particular
 * smartcard ... the order here is setup to match the most likely things
 * first so you probably shouldn't play with it.
 */
static long all_speed[]={9600L,19200L,38400L,115200L};
static int all_parity[]={SIO_PARITY_EVEN,SIO_PARITY_ODD,SIO_PARITY_NONE};
static int all_databits[]={8,7};
static int all_stopbits[]={2,1};
static int all_iomode[]={SIO_IOMODE_DIRECT,SIO_IOMODE_INDIRECT};

#define N_all_speed (sizeof(all_speed)/sizeof(all_speed[0]))
#define N_all_parity (sizeof(all_parity)/sizeof(all_parity[0]))
#define N_all_databits (sizeof(all_databits)/sizeof(all_databits[0]))
#define N_all_stopbits (sizeof(all_stopbits)/sizeof(all_stopbits[0]))
#define N_all_iomode (sizeof(all_iomode)/sizeof(all_iomode[0]))

int SCT0_ScanForATR(SIO_INFO *s,char *ubuf,int ubufsiz)
{
  long aspeed;
  int aparity,adatabits,astopbits,aiomode;
  char buf[256];
  int len;

  for(aspeed=0;aspeed<N_all_speed;aspeed++)
    for(adatabits=0;adatabits<N_all_databits;adatabits++)
      for(astopbits=0;astopbits<N_all_stopbits;astopbits++)
        for(aparity=0;aparity<N_all_parity;aparity++)
          for(aiomode=0;aiomode<N_all_iomode;aiomode++) {
	    SIO_SetSpeed(s,all_speed[aspeed]);
	    SIO_SetDataBits(s,all_databits[adatabits]);
	    SIO_SetStopBits(s,all_stopbits[astopbits]);
	    SIO_SetParity(s,all_parity[aparity]);
	    SIO_SetIOMode(s,all_iomode[aiomode]);
	    SIO_WriteSettings(s);
	    len=SCT0_GetATR(s,buf,sizeof(buf));

	    if (dbglev)
	      SLOG_printf(outf,"settings: %s\n",SIO_GetSettingsString(s));
	    if (len>0) {
	      if (dbglev) {
		/* dump the record */
		SLOG_printf(outf,"ATR-DATA: ");
		SLOG_dump(outf,buf,len,0);
		SLOG_printf(outf,"\n");
		SLOG_printf(outf,"ATR-TEXT: ");
		SLOG_dump(outf,buf,len,1);
		SLOG_printf(outf,"\n");
	      }
	      /* it has to be a valid ATR response before we
	       * consider that it has worked 
	       */
	      if ((buf[0]==0x3f)||(buf[0]==0x3b)) {
		/* copy into the users buffer */
	        if ((ubuf!=NULL)&&(len<=ubufsiz))
		  memcpy(ubuf,buf,len);
		return(len);
	      }
	    }
	  }
  return(0);
}

int SCT0_ScanClasses(SIO_INFO *s,char *classlist)
{
  int i,j,len;
  unsigned char buf[BUFSIZ];
  int retries,reset_done;

  /* flush data ... read until nothing available */
  while (SIO_ReadChar(s)!=-1)
    ;

  reset_done=retries=0;
  for(i=0;i<=255;i++) {
do_again: ;
    buf[0]=i;	/* CLA = 0 */
    buf[1]=0x0;
    buf[2]=0x0;
    buf[3]=0x0;
    buf[4]=0x0;
    len=SIO_WriteBuffer(s,buf,5);

    /* we need to delay before reading to give the reader time
     * to start responding
     */
    SIO_Delay(s,100);

    /* now read in some data */
    for(j=0;j<2;j++) {
      len=SIO_ReadBuffer(s,buf,sizeof(buf));
      if (len>0)
	break;
    }

    if (dbglev) {
      SLOG_printf(outf,"CLA 0x%02x: (%d) ",i,len);
      SLOG_dump(outf,buf,len,0);
      SLOG_printf(outf,"\n");
    }

    if (len!=2) {
      if (retries++>MAX_RETRIES) {
	if (!reset_done) {
	  /* reset things ... some cards defend against 
	   * a probe like this
	   */
	  SCT0_Reset(s);
	  while (SIO_ReadChar(s)!=-1)
	    ;
	  reset_done++;

	  if (dbglev) {
	    SLOG_printf(outf,"CLA 0x%02x: SKIPPING with RESET\n",i);
	  }

	  /* skip to next INS */
	  continue;
	}
	return (-1);
      }
      goto do_again;
    }

    /* process the result ... 6e 00 means the class doesn't exist */
    if (buf[0]!=0x6e)
      classlist[i]=1;
    else
      classlist[i]=0;

    reset_done=0;

  }
  return(1);
}

int SCT0_ScanInstructions(SIO_INFO *s,char theclass,char *inslist)
{
  int i,j,len;
  unsigned char buf[BUFSIZ];
  int retries;

  if (dbglev)
    SLOG_printf(outf,"INS scan for CLASS 0x%02x\n",theclass & 0xff);

  /* flush data ... read until nothing available */
  while (SIO_ReadChar(s)!=-1)
    ;

  retries=0;

  for(i=0;i<=255;i++) {
do_again: ;
    buf[0]=theclass;	/* CLA = 0 */
    buf[1]=i;           /* instruction */
    buf[2]=0x0;
    buf[3]=0x0;
    buf[4]=0x0;
    len=SIO_WriteBuffer(s,buf,5);

    /* we need to delay before reading to give the reader time
     * to start responding
     */
    SIO_Delay(s,100);

    /* now read in some data */
    for(j=0;j<2;j++) {
      len=SIO_ReadBuffer(s,buf,sizeof(buf));
      if (len>0)
	break;
    }

    if (dbglev) {
      SLOG_printf(outf,"INS 0x%02x: (%d) ",i,len);
      SLOG_dump(outf,buf,len,0);
      SLOG_printf(outf,"\n");
    }

    /* we could get back a response to a command that didn't
     * need any parameters ... which we should handle
     */
    if (len==3) {
      if ((buf[0] & 0xff)==i) {
	/* process the result ... 6e 00 or 6d 00 means the instruction
	 * doesn't exist ... 6d=unknown, 6e=wrong
	 */
	if ((buf[1]!=0x6e)&&(buf[1]!=0x6d))
	  inslist[i]=1;
	else
	  inslist[i]=0;
	continue;
      } else {
	/*
	SLOG_printf(outf,"len=3 i=0x%x buf[0]=0x%x\n",i,buf[0]);
	*/
      }
    }

    if (len!=2) {
      if (retries++>MAX_RETRIES)
	return (-1);
      goto do_again;
    }

    /* process the result ... 6e 00 or 6d 00 means the instruction
     * doesn't exist ... 6d=unknown, 6e=wrong
     */
    if ((buf[0]!=0x6e)&&(buf[0]!=0x6d))
      inslist[i]=1;
    else
      inslist[i]=0;

  }
  return(1);
}

int SCT0_ScanFiles(SIO_INFO *s,char theclass,char thefile,char *filelist)
{
  int i,j,len;
  unsigned char buf[BUFSIZ];
  int retries;

  if (dbglev)
    SLOG_printf(outf,"FILE scan for CLASS 0x%02x file 0x%02x\n",
	            theclass & 0xff, thefile & 0xff);

  /* flush data ... read until nothing available */
  while (SIO_ReadChar(s)!=-1)
    ;

  retries=0;

  for(i=0;i<=255;i++) {
do_again: ;
    buf[0]=theclass;	/* CLA = 0 */
    buf[1]=0xa4;        /* instruction */
    buf[2]=0x0;
    buf[3]=0x0;
    buf[4]=0x2;
    len=SIO_WriteBuffer(s,buf,5);

    /* we need to delay before reading to give the reader time
     * to start responding
     */
    SIO_Delay(s,100);

    /* now read in some data - which should be the INS */
    for(j=0;j<2;j++) {
      len=SIO_ReadBuffer(s,buf,sizeof(buf));
      if (len>0)
	break;
    }

    if (dbglev) {
      SLOG_printf(outf,"FILE 0x%02x: (%d) [INS] ",i,len);
      SLOG_dump(outf,buf,len,0);
      SLOG_printf(outf,"\n");
    }

    if (len!=1) {
      if (retries++>MAX_RETRIES)
	return (-1);
      goto do_again;
    }

    /* we must get the INS back or things are broken! */
    if (buf[0]!=0xa4) {
      if (dbglev)
	SLOG_printf(outf,"ScanFiles - didn't get back INS - aborting scan\n");
      return (-1);
    }

    /* now send the file ID */
    buf[0]=thefile;    
    buf[1]=i;
    len=SIO_WriteBuffer(s,buf,2);

    /* we need to delay before reading to give the reader time
     * to start responding
     */
    SIO_Delay(s,100);

    /* now read in some data - which should be the response code */
    for(j=0;j<2;j++) {
      len=SIO_ReadBuffer(s,buf,sizeof(buf));
      if (len>0)
	break;
    }

    if (dbglev) {
      SLOG_printf(outf,"FILE 0x%02x: (%d) [RES] ",i,len);
      SLOG_dump(outf,buf,len,0);
      SLOG_printf(outf,"\n");
    }

    if (len!=2) {
      if (retries++>MAX_RETRIES)
	return (-1);
      goto do_again;
    }

    /* process the result 
     * 61 xx is okay with length (Schlumberger Cryptoflex)
     * 9x 00 is okay (Chipknip?) 
     */
    if ( (((buf[0] & 0xf0)==0x90) && (buf[1]==0)) ||
         (buf[0]==0x61) ) {
      filelist[i]=1;
      /* now we need to select the MF again otherwise we can end
       * up confused with going down a directory level which isn't
       * what we want
       */
    } else
      filelist[i]=0;

  }
  return(1);
}

int SCT0_ScanForResponse(SIO_INFO *s,char *ubuf,int ubufsiz)
{
  long aspeed;
  int aparity,adatabits,astopbits,aiomode;
  char buf[256];
  int len,j;

/*  for(aspeed=0;aspeed<N_all_speed;aspeed++) */

    aspeed=0;

    for(adatabits=0;adatabits<N_all_databits;adatabits++)
      for(astopbits=0;astopbits<N_all_stopbits;astopbits++)
        for(aparity=0;aparity<N_all_parity;aparity++)
          for(aiomode=0;aiomode<N_all_iomode;aiomode++) {
	    SIO_SetSpeed(s,all_speed[aspeed]);
	    SIO_SetDataBits(s,all_databits[adatabits]);
	    SIO_SetStopBits(s,all_stopbits[astopbits]);
	    SIO_SetParity(s,all_parity[aparity]);
	    SIO_SetIOMode(s,all_iomode[aiomode]);
	    SIO_WriteSettings(s);

	    len=SIO_WriteBuffer(s,ubuf,ubufsiz);

	    /* we need to delay before reading to give the reader time
	     * to start responding
	     */
	    SIO_Delay(s,100);

	    /* now read in some data - which should be the INS */
	    for(j=0;j<2;j++) {
	      len=SIO_ReadBuffer(s,buf,sizeof(buf));
	      if (len>0)
		break;
	    }

	    if (1||dbglev) {
	      SLOG_printf(outf,"settings: %s\n",SIO_GetSettingsString(s));
	      SLOG_printf(outf,"RESPONSE TO COMMAND\n");
	      SLOG_dump(outf,buf,len,0);
	      SLOG_printf(outf,"\n");
	    }

	  }
  return(0);
}
