#!/usr/local/bin/perl -w 

use strict;
use FileHandle;
use Fcntl;
use POSIX qw(strftime);
use Tie::DB_Lock;
use PriorityQueue;

my $configFile = shift @ARGV;

if (not defined $configFile) {
    die "usage: $0 <configfile>\n";
}

my $scoresDBFile = "data/scores.db";
my $clients = readConfig($configFile);
my $probes = readProbes();

# IP-Adressen der Clients 'rausfinden
my $client;
my %clientsByAddress;
foreach $client (@$clients) {
    my $inetAddr = gethostbyname($client->{'name'});
    my $addr = inet_ntoa($inetAddr);
    if (not defined $addr) {
	die"$0: could not resolve client host name $client->{'name'} to IP address\n";
    }
    $client->{'addr'} = $addr;
    $clientsByAddress{$addr} = $client;
}

# Scores-Datenbank initialisieren

umask(002);
my %scores;
tie %scores, 'Tie::DB_Lock', $scoresDBFile, 'rw'
    or die "$0: could not die to scores DB $scoresDBFile: $!\n";

my @allClientNames;
foreach (@$clients) {
    push @allClientNames, $_->{'name'};
}
$scores{'__allClientNames__'} = join(':', @allClientNames);

my @allProbeNames;
foreach (@$probes) {
    push @allProbeNames, $_->{'name'};
}
$scores{'__allProbeNames__'} = join(':', @allProbeNames);

my ($clientName, $probeName);
foreach $clientName (@allClientNames) {
    foreach $probeName (@allProbeNames) {
	if (not defined $scores{"$clientName:$probeName"}) {
	    $scores{"$clientName:$probeName"} = 0;
	}
    }
}

untie %scores;

# Signal-Handler fuer SIGCLD aufsetzen, damit wir nicht auf sterbende
# Subprozesse warten muessen

sub REAPER {
    my $pid = wait;
#    warn "$0: child $pid died\n";
    $SIG{CHLD} = \&REAPER;
}
$SIG{CHLD} = \&REAPER;

# TimerQueue aufsetzen.  In der TimerQueue werden alle zeitlich
# gesteuerten Events abgelegt.  Jedes Event wird mit der gewuenschten
# Startzeit abgelegt.  Die Hauptschleife wartet so lange, bis die Zeit
# fuer das vorderste Element der Queue gekommen ist, zieht dieses von
# der Queue und fuehrt die in {'item'} abgelegte Codereferenz aus.

my $queue = new PriorityQueue;
my $now = time;
my $probe;
foreach $probe (@$probes) {
    $queue->enqueue($now + $probe->{'interval'}, sub { runProbe(\%clientsByAddress, $probe); } );
}

while (1) {
    my $now = time;

    while ($queue->front
	   and $queue->front->{'priority'} <= $now) {
	&{$queue->dequeue->{'item'}};
    }

    my $next = $queue->front->{'priority'} - time;
    print "$0: next event in $next seconds\n";

    sleep($next) if ($next > 0);
}

sub runProbe {
    my ($clientsByAddress, $probe) = @_;

    # Probe gleich wieder auf die Queue tun.
    $queue->enqueue(time + $probe->{'interval'}, sub { runProbe(\%clientsByAddress, $probe); } );

    my @activeProbes;		# Aktive Probe-Prozesse (liste mit Hashes)

    # Anstarten der Probes.  Alle Clients werden gleichzeitig geprobed.

    print "$0: running probe $probe->{'name'}\n";

    my $client;
    foreach $client (values %$clientsByAddress) {
	my ($readHandle, $writeHandle) = FileHandle::pipe;
	my $pid = fork;
	if (not defined $pid) {
	    die "$0: can't fork: $!\n";
	}
	if ($pid) {
	    fcntl($readHandle, F_SETFL, O_NDELAY)
		or die "$0: fcntl failed: $!\n";
	    $writeHandle->close;
	    push(@activeProbes,
		 {
		     'pid' => $pid, 
		     'handle' => $readHandle,
		     'client' => $client
		     }
		 );
	} else {
	    $0 = "dm-probe $probe->{'name'}: $client->{'name'}";
	    setpgrp;
	    $readHandle->close;
	    my $workingDir = "data/$client->{'name'}";
	    if (not -d $workingDir) {
		mkdir ($workingDir, 0755)
		    or die "$0: can't create working directory $workingDir: $!\n";
	    }
	    chdir($workingDir)
		or die "$0: can't chdir to working directory $workingDir: $!\n";
	    open(STDOUT, ">&".fileno($writeHandle))
		or die "$0: can't fdup WRITEHANDLE: $!\n";
	    my $logfile = $probe->{'name'}.".log";
	    open(STDERR, ">>logfile")
		or die "$0: can't open log file $logfile: $!\n";
	    print STDERR 
		"-------------------------------------------------\n",
		strftime("%D %T running probe $probe->{'name'}\n",
			 localtime time);
	    $| = 1;
	    &{$probe->{'code'}}($client);
	    exit(0);
	}
    }

    $queue->enqueue(time + $probe->{'timeout'}, sub {

	print "$0: collecting results for probe $probe->{'name'}\n";

	# Einsammeln der Resultate von den Probes und Verteilen der Punkte
	my $activeProbe;
	foreach $activeProbe (@activeProbes) {
	    my $result = read($activeProbe->{'handle'}, $_, 1024*64);
#	print "$0 read [$_] from $activeProbe->{'client'}->{'name'} on probe $probe->{'name'}\n";
	    if (not defined $result) {
		warn "$0: bad read for result from $activeProbe->{'client'}->{'name'}: $!\n";
	    } else {
		foreach (split(/\n/)) {
		    if (/^(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}) (\d+)$/) {
			my ($addr, $points) = ($1, $2);
			if (defined $clientsByAddress->{$addr}) {
			    if ($addr ne $activeProbe->{'client'}->{'addr'}) {
				warn "$0: client $activeProbe->{'client'}->{'name'} lost $points "
				    . "points "
					. "on probe $probe->{'name'} to "
					    . "$clientsByAddress->{$addr}->{'name'}\n";
			    }
			    my $winner = $clientsByAddress->{$addr};
			    warn "$0: client $winner->{'name'} won $points points\n";

			    # Notieren der Punkte in der Punktedatenbank
			    my %scores;
			    tie %scores, 'Tie::DB_Lock', $scoresDBFile, 'rw'
				or die "$0: could not die to scores DB $scoresDBFile: $!\n";
			    $scores{$winner->{'name'} . ":" . $probe->{'name'}} += $points;
			    untie %scores;

			} else {
			    warn "$0: client $activeProbe->{'client'}->{'name'} sent bogus address, no score for this probe\n";
			}
		    } else {
			warn "$0: no points for $activeProbe->{'client'}->{'name'} on probe "
			    . "$probe->{'name'}\n";
		    }
		}
	    }
	    (kill -9, $activeProbe->{'pid'})
		or warn "$0: can't kill $activeProbe->{'pid'} ($!)\n";
	}
	print "$0: results for probe $probe->{'name'} collected\n";
    });
}

sub readProbes {

    my $probesDir = "probes";

    my $probes = [];
    my $probeFile;

  probe:
    foreach $probeFile (<$probesDir/*.pl>) {

	if (not open(PROBE, $probeFile)) {
	    warn "$0: can't open probe $_: $!\n";
	    next probe;

	}

	print STDERR "$0: loading probe $probeFile\n";

	$probeFile =~ m-^$probesDir/(.*)\.pl$-;
	my $probe = { name => $1 };

	my @keywords = qw(timeout interval);

	my $code = "sub {";
	while (<PROBE>) {
	    my $matchReg = join("|", @keywords);
	    if (/#@# ($matchReg):\s*(.*)\n$/) {
		$probe->{$1} = $2;
	    } else {
		$code .= $_;
	    }
	}
	$code .= "}";

	my $missingKeyword = 0;
	foreach (@keywords) {
	    if (not defined $probe->{$_}) {
		print STDERR "$0: missing keyword $_ in probe $probeFile\n";
		$missingKeyword++;
	    }
	}
	if ($missingKeyword) {
	    print STDERR "$0: probe not loaded\n";
	    next probe;
	}

	close(PROBE);

	$probe->{code} = eval $code;

	if (not defined $probe->{code}) {
	    print STDERR "$0: problem with probe $probeFile\n";
	    print STDERR "$0: (probe not loaded)\n";
	    next probe;
	}

	if ($probe->{interval} < $probe->{timeout}) {
	    print STDERR "$0: interval shorter than timeout in probe $probeFile\n";
	    print STDERR "$0: probe not loaded\n";
	    next probe;
	}

	push @$probes, $probe;
    }

    print "$0: ", $#$probes+1, " probes loaded\n";
    return $probes;
}

sub readConfig {
    my $filename = shift;
    my @retval;

    open(CONFIG, $filename)
	or die "$0: can't open configuration file $filename: $!\n";
    while (<CONFIG>) {
	chomp;
	my $realConfigLine = $_;
	s/#.*//;
	s/^\s*//;
	s/\s*$//;
	next if (/^$/);
	if (not /^(\S+)\s+(\S+)\s+(\S+)$/) {
	    die "$0: bad configuration line: [$realConfigLine]\n";
	}
	my ($host, $user, $password) = ($1, $2, $3);
	push @retval, { 'name' => $host,
			'user' => $user,
			'password' => $password
			};
    }
    close(CONFIG);
    return \@retval;
}
