"use strict";
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.registerPPLLanguage = void 0;
const tslib_1 = require("tslib");
const monaco_1 = require("../monaco");
const constants_1 = require("./constants");
const worker_store_1 = require("../worker_store");
const worker_proxy_service_1 = require("./worker_proxy_service");
const ppl_language_analyzer_1 = require("./ppl_language_analyzer");
const ppl_documentation_1 = require("./ppl_documentation");
const formatter_1 = require("./formatter");
// @ts-ignore
const ppl_editor_worker_js_1 = tslib_1.__importDefault(require("!!raw-loader!../../target/public/ppl.editor.worker.js"));
const PPL_LANGUAGE_ID = constants_1.ID;
const OWNER = 'PPL_WORKER';
// Register ppl worker to the worker map first
(0, worker_store_1.registerWorker)(constants_1.ID, ppl_editor_worker_js_1.default);
// PPL worker proxy service for worker-based syntax highlighting
const pplWorkerProxyService = new worker_proxy_service_1.PPLWorkerProxyService();
// PPL analyzer for synchronous tokenization (lazy initialization)
let pplAnalyzer;
/**
 * Map PPL Language Analyzer tokens to Monaco editor token classes
 * Based on ANTLR-generated token types from OpenSearchPPLLexer
 */
const mapPPLTokenToMonacoTokenType = (tokenType) => {
    const type = tokenType.toUpperCase();
    // Use optimized Set lookups from constants
    for (const [monacoType, tokenSet] of Object.entries(constants_1.PPL_TOKEN_SETS)) {
        if (tokenSet.has(type)) {
            return monacoType;
        }
    }
    // Default case
    return 'identifier';
};
/**
 * Create Monaco language configuration for PPL
 */
const createPPLLanguageConfiguration = () => ({
    comments: {
        lineComment: '//',
        blockComment: ['/*', '*/'],
    },
    brackets: [
        ['{', '}'],
        ['[', ']'],
        ['(', ')'],
    ],
    autoClosingPairs: [
        { open: '{', close: '}' },
        { open: '[', close: ']' },
        { open: '(', close: ')' },
        { open: '"', close: '"', notIn: ['string'] },
        { open: "'", close: "'", notIn: ['string', 'comment'] },
    ],
    surroundingPairs: [
        { open: '{', close: '}' },
        { open: '[', close: ']' },
        { open: '(', close: ')' },
        { open: '"', close: '"' },
        { open: "'", close: "'" },
    ],
});
/**
 * Set up synchronous tokenization for PPL
 */
const setupPPLTokenization = () => {
    monaco_1.monaco.languages.setTokensProvider(PPL_LANGUAGE_ID, {
        getInitialState: () => {
            const state = {
                clone: () => state,
                equals: () => true,
            };
            return state;
        },
        tokenize: (line, state) => {
            // Use PPL Language Analyzer for accurate tokenization
            const tokens = [];
            try {
                // Only process if line contains potential PPL content
                if (line.trim()) {
                    // Lazy initialize the PPL analyzer only when needed
                    if (!pplAnalyzer) {
                        pplAnalyzer = (0, ppl_language_analyzer_1.getPPLLanguageAnalyzer)();
                    }
                    const pplTokens = pplAnalyzer.tokenize(line);
                    for (const pplToken of pplTokens) {
                        const tokenType = mapPPLTokenToMonacoTokenType(pplToken.type);
                        tokens.push({
                            startIndex: pplToken.startIndex,
                            scopes: tokenType,
                        });
                    }
                }
            }
            catch (error) {
                // If ANTLR fails, return empty tokens
            }
            return {
                tokens,
                endState: state,
            };
        },
    });
};
/**
 * Process syntax highlighting for PPL models
 */
const processSyntaxHighlighting = async (model) => {
    // Only process if the model is still set to PPL language
    if (model.getLanguageId() !== PPL_LANGUAGE_ID) {
        // Clear any existing PPL markers if language changed
        monaco_1.monaco.editor.setModelMarkers(model, OWNER, []);
        return;
    }
    try {
        const content = model.getValue();
        // Ensure worker is set up before validation - always call setup as it has internal check
        pplWorkerProxyService.setup(ppl_editor_worker_js_1.default);
        // Get validation result from worker with timeout protection
        const validationResult = (await pplWorkerProxyService.validate(content));
        if (validationResult.errors.length > 0) {
            // Convert errors to Monaco markers
            const markers = validationResult.errors.map((error) => {
                // Map SyntaxError properties to Monaco marker properties
                const startLineNumber = error.line || 1;
                const endLineNumber = error.endLine || error.line || startLineNumber;
                const startColumn = (error.column || 0) + 1; // Monaco is 1-based, ANTLR is 0-based
                const endColumn = (error.endColumn || error.column + 1 || startColumn) + 1;
                const safeStartLine = Math.max(1, startLineNumber);
                const safeEndLine = Math.max(safeStartLine, endLineNumber);
                const safeStartColumn = Math.max(1, startColumn);
                const safeEndColumn = Math.max(safeStartColumn, endColumn);
                const docLink = (0, ppl_documentation_1.getPPLDocumentationLink)(error.message);
                return {
                    severity: monaco_1.monaco.MarkerSeverity.Error,
                    message: error.message,
                    startLineNumber: safeStartLine,
                    startColumn: safeStartColumn,
                    endLineNumber: safeEndLine,
                    endColumn: safeEndColumn,
                    // Add error code for better categorization
                    code: {
                        value: 'View Documentation',
                        target: monaco_1.monaco.Uri.parse(docLink.url),
                    },
                };
            });
            monaco_1.monaco.editor.setModelMarkers(model, OWNER, markers);
        }
        else {
            // Clear markers if no errors
            monaco_1.monaco.editor.setModelMarkers(model, OWNER, []);
        }
    }
    catch (error) {
        // Silent error handling - continue without worker-based highlighting
    }
};
/**
 * Set up PPL document range formatting provider
 */
const setupPPLFormatter = () => {
    monaco_1.monaco.languages.registerDocumentRangeFormattingEditProvider(PPL_LANGUAGE_ID, formatter_1.pplRangeFormatProvider);
    monaco_1.monaco.languages.registerOnTypeFormattingEditProvider(PPL_LANGUAGE_ID, formatter_1.pplOnTypeFormatProvider);
};
/**
 * Set up syntax highlighting using PPL worker
 */
const setupPPLSyntaxHighlighting = () => {
    const disposables = [];
    const handleModel = (model) => {
        // Set up content change listener
        disposables.push(model.onDidChangeContent(async () => {
            if (model.getLanguageId() === PPL_LANGUAGE_ID) {
                await processSyntaxHighlighting(model);
            }
        }));
        // Set up language change listener
        disposables.push(model.onDidChangeLanguage(async () => {
            if (model.getLanguageId() === PPL_LANGUAGE_ID) {
                await processSyntaxHighlighting(model);
            }
            else {
                monaco_1.monaco.editor.setModelMarkers(model, OWNER, []);
            }
        }));
        // Process immediately if already PPL
        if (model.getLanguageId() === PPL_LANGUAGE_ID) {
            processSyntaxHighlighting(model);
        }
    };
    // Listen for new models
    disposables.push(monaco_1.monaco.editor.onDidCreateModel(handleModel));
    // Listen for model disposal to clear markers
    disposables.push(monaco_1.monaco.editor.onWillDisposeModel((model) => {
        monaco_1.monaco.editor.setModelMarkers(model, OWNER, []);
    }));
    // Handle existing models
    monaco_1.monaco.editor.getModels().forEach(handleModel);
    // Return cleanup function
    return () => {
        disposables.forEach((d) => d.dispose());
        pplWorkerProxyService.stop();
    };
};
/**
 * Register PPL language support with Monaco Editor
 */
const registerPPLLanguage = () => {
    // Register the PPL language
    monaco_1.monaco.languages.register({
        id: PPL_LANGUAGE_ID,
        extensions: ['.ppl'],
        aliases: ['PPL', 'ppl', 'Piped Processing Language'],
        mimetypes: ['application/ppl', 'text/ppl'],
    });
    // Set language configuration
    monaco_1.monaco.languages.setLanguageConfiguration(PPL_LANGUAGE_ID, createPPLLanguageConfiguration());
    // Set up synchronous tokenization
    setupPPLTokenization();
    // Set up PPL formatter
    setupPPLFormatter();
    // Set up syntax highlighting with worker
    const disposeSyntaxHighlighting = setupPPLSyntaxHighlighting();
    return {
        dispose: () => {
            disposeSyntaxHighlighting();
        },
    };
};
exports.registerPPLLanguage = registerPPLLanguage;
// Auto-register PPL language support
(0, exports.registerPPLLanguage)();
//# sourceMappingURL=language.js.map