﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/route53/Route53Request.h>
#include <aws/route53/Route53_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Route53 {
namespace Model {

/**
 * <p>A complex type that contains information about the resource record sets that
 * you want to create based on a specified traffic policy.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/route53-2013-04-01/CreateTrafficPolicyInstanceRequest">AWS
 * API Reference</a></p>
 */
class CreateTrafficPolicyInstanceRequest : public Route53Request {
 public:
  AWS_ROUTE53_API CreateTrafficPolicyInstanceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateTrafficPolicyInstance"; }

  AWS_ROUTE53_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID of the hosted zone that you want Amazon Route 53 to create resource
   * record sets in by using the configuration in a traffic policy.</p>
   */
  inline const Aws::String& GetHostedZoneId() const { return m_hostedZoneId; }
  inline bool HostedZoneIdHasBeenSet() const { return m_hostedZoneIdHasBeenSet; }
  template <typename HostedZoneIdT = Aws::String>
  void SetHostedZoneId(HostedZoneIdT&& value) {
    m_hostedZoneIdHasBeenSet = true;
    m_hostedZoneId = std::forward<HostedZoneIdT>(value);
  }
  template <typename HostedZoneIdT = Aws::String>
  CreateTrafficPolicyInstanceRequest& WithHostedZoneId(HostedZoneIdT&& value) {
    SetHostedZoneId(std::forward<HostedZoneIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The domain name (such as example.com) or subdomain name (such as
   * www.example.com) for which Amazon Route 53 responds to DNS queries by using the
   * resource record sets that Route 53 creates for this traffic policy instance.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateTrafficPolicyInstanceRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>(Optional) The TTL that you want Amazon Route 53 to assign to all of the
   * resource record sets that it creates in the specified hosted zone.</p>
   */
  inline long long GetTTL() const { return m_tTL; }
  inline bool TTLHasBeenSet() const { return m_tTLHasBeenSet; }
  inline void SetTTL(long long value) {
    m_tTLHasBeenSet = true;
    m_tTL = value;
  }
  inline CreateTrafficPolicyInstanceRequest& WithTTL(long long value) {
    SetTTL(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the traffic policy that you want to use to create resource record
   * sets in the specified hosted zone.</p>
   */
  inline const Aws::String& GetTrafficPolicyId() const { return m_trafficPolicyId; }
  inline bool TrafficPolicyIdHasBeenSet() const { return m_trafficPolicyIdHasBeenSet; }
  template <typename TrafficPolicyIdT = Aws::String>
  void SetTrafficPolicyId(TrafficPolicyIdT&& value) {
    m_trafficPolicyIdHasBeenSet = true;
    m_trafficPolicyId = std::forward<TrafficPolicyIdT>(value);
  }
  template <typename TrafficPolicyIdT = Aws::String>
  CreateTrafficPolicyInstanceRequest& WithTrafficPolicyId(TrafficPolicyIdT&& value) {
    SetTrafficPolicyId(std::forward<TrafficPolicyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the traffic policy that you want to use to create resource
   * record sets in the specified hosted zone.</p>
   */
  inline int GetTrafficPolicyVersion() const { return m_trafficPolicyVersion; }
  inline bool TrafficPolicyVersionHasBeenSet() const { return m_trafficPolicyVersionHasBeenSet; }
  inline void SetTrafficPolicyVersion(int value) {
    m_trafficPolicyVersionHasBeenSet = true;
    m_trafficPolicyVersion = value;
  }
  inline CreateTrafficPolicyInstanceRequest& WithTrafficPolicyVersion(int value) {
    SetTrafficPolicyVersion(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_hostedZoneId;
  bool m_hostedZoneIdHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  long long m_tTL{0};
  bool m_tTLHasBeenSet = false;

  Aws::String m_trafficPolicyId;
  bool m_trafficPolicyIdHasBeenSet = false;

  int m_trafficPolicyVersion{0};
  bool m_trafficPolicyVersionHasBeenSet = false;
};

}  // namespace Model
}  // namespace Route53
}  // namespace Aws
