﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/databrew/GlueDataBrew_EXPORTS.h>
#include <aws/databrew/model/RecipeStep.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace GlueDataBrew {
namespace Model {

/**
 * <p>Represents one or more actions to be performed on a DataBrew
 * dataset.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/Recipe">AWS API
 * Reference</a></p>
 */
class Recipe {
 public:
  AWS_GLUEDATABREW_API Recipe() = default;
  AWS_GLUEDATABREW_API Recipe(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLUEDATABREW_API Recipe& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLUEDATABREW_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the user who created the recipe.</p>
   */
  inline const Aws::String& GetCreatedBy() const { return m_createdBy; }
  inline bool CreatedByHasBeenSet() const { return m_createdByHasBeenSet; }
  template <typename CreatedByT = Aws::String>
  void SetCreatedBy(CreatedByT&& value) {
    m_createdByHasBeenSet = true;
    m_createdBy = std::forward<CreatedByT>(value);
  }
  template <typename CreatedByT = Aws::String>
  Recipe& WithCreatedBy(CreatedByT&& value) {
    SetCreatedBy(std::forward<CreatedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time that the recipe was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreateDate() const { return m_createDate; }
  inline bool CreateDateHasBeenSet() const { return m_createDateHasBeenSet; }
  template <typename CreateDateT = Aws::Utils::DateTime>
  void SetCreateDate(CreateDateT&& value) {
    m_createDateHasBeenSet = true;
    m_createDate = std::forward<CreateDateT>(value);
  }
  template <typename CreateDateT = Aws::Utils::DateTime>
  Recipe& WithCreateDate(CreateDateT&& value) {
    SetCreateDate(std::forward<CreateDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the user who last modified the recipe.</p>
   */
  inline const Aws::String& GetLastModifiedBy() const { return m_lastModifiedBy; }
  inline bool LastModifiedByHasBeenSet() const { return m_lastModifiedByHasBeenSet; }
  template <typename LastModifiedByT = Aws::String>
  void SetLastModifiedBy(LastModifiedByT&& value) {
    m_lastModifiedByHasBeenSet = true;
    m_lastModifiedBy = std::forward<LastModifiedByT>(value);
  }
  template <typename LastModifiedByT = Aws::String>
  Recipe& WithLastModifiedBy(LastModifiedByT&& value) {
    SetLastModifiedBy(std::forward<LastModifiedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The last modification date and time of the recipe.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedDate() const { return m_lastModifiedDate; }
  inline bool LastModifiedDateHasBeenSet() const { return m_lastModifiedDateHasBeenSet; }
  template <typename LastModifiedDateT = Aws::Utils::DateTime>
  void SetLastModifiedDate(LastModifiedDateT&& value) {
    m_lastModifiedDateHasBeenSet = true;
    m_lastModifiedDate = std::forward<LastModifiedDateT>(value);
  }
  template <typename LastModifiedDateT = Aws::Utils::DateTime>
  Recipe& WithLastModifiedDate(LastModifiedDateT&& value) {
    SetLastModifiedDate(std::forward<LastModifiedDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the project that the recipe is associated with.</p>
   */
  inline const Aws::String& GetProjectName() const { return m_projectName; }
  inline bool ProjectNameHasBeenSet() const { return m_projectNameHasBeenSet; }
  template <typename ProjectNameT = Aws::String>
  void SetProjectName(ProjectNameT&& value) {
    m_projectNameHasBeenSet = true;
    m_projectName = std::forward<ProjectNameT>(value);
  }
  template <typename ProjectNameT = Aws::String>
  Recipe& WithProjectName(ProjectNameT&& value) {
    SetProjectName(std::forward<ProjectNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the user who published the recipe.</p>
   */
  inline const Aws::String& GetPublishedBy() const { return m_publishedBy; }
  inline bool PublishedByHasBeenSet() const { return m_publishedByHasBeenSet; }
  template <typename PublishedByT = Aws::String>
  void SetPublishedBy(PublishedByT&& value) {
    m_publishedByHasBeenSet = true;
    m_publishedBy = std::forward<PublishedByT>(value);
  }
  template <typename PublishedByT = Aws::String>
  Recipe& WithPublishedBy(PublishedByT&& value) {
    SetPublishedBy(std::forward<PublishedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time when the recipe was published.</p>
   */
  inline const Aws::Utils::DateTime& GetPublishedDate() const { return m_publishedDate; }
  inline bool PublishedDateHasBeenSet() const { return m_publishedDateHasBeenSet; }
  template <typename PublishedDateT = Aws::Utils::DateTime>
  void SetPublishedDate(PublishedDateT&& value) {
    m_publishedDateHasBeenSet = true;
    m_publishedDate = std::forward<PublishedDateT>(value);
  }
  template <typename PublishedDateT = Aws::Utils::DateTime>
  Recipe& WithPublishedDate(PublishedDateT&& value) {
    SetPublishedDate(std::forward<PublishedDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the recipe.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  Recipe& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique name for the recipe.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  Recipe& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) for the recipe.</p>
   */
  inline const Aws::String& GetResourceArn() const { return m_resourceArn; }
  inline bool ResourceArnHasBeenSet() const { return m_resourceArnHasBeenSet; }
  template <typename ResourceArnT = Aws::String>
  void SetResourceArn(ResourceArnT&& value) {
    m_resourceArnHasBeenSet = true;
    m_resourceArn = std::forward<ResourceArnT>(value);
  }
  template <typename ResourceArnT = Aws::String>
  Recipe& WithResourceArn(ResourceArnT&& value) {
    SetResourceArn(std::forward<ResourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of steps that are defined by the recipe.</p>
   */
  inline const Aws::Vector<RecipeStep>& GetSteps() const { return m_steps; }
  inline bool StepsHasBeenSet() const { return m_stepsHasBeenSet; }
  template <typename StepsT = Aws::Vector<RecipeStep>>
  void SetSteps(StepsT&& value) {
    m_stepsHasBeenSet = true;
    m_steps = std::forward<StepsT>(value);
  }
  template <typename StepsT = Aws::Vector<RecipeStep>>
  Recipe& WithSteps(StepsT&& value) {
    SetSteps(std::forward<StepsT>(value));
    return *this;
  }
  template <typename StepsT = RecipeStep>
  Recipe& AddSteps(StepsT&& value) {
    m_stepsHasBeenSet = true;
    m_steps.emplace_back(std::forward<StepsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Metadata tags that have been applied to the recipe.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  Recipe& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  Recipe& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier for the version for the recipe. Must be one of the
   * following:</p> <ul> <li> <p>Numeric version (<code>X.Y</code>) - <code>X</code>
   * and <code>Y</code> stand for major and minor version numbers. The maximum length
   * of each is 6 digits, and neither can be negative values. Both <code>X</code> and
   * <code>Y</code> are required, and "0.0" isn't a valid version.</p> </li> <li> <p>
   * <code>LATEST_WORKING</code> - the most recent valid version being developed in a
   * DataBrew project.</p> </li> <li> <p> <code>LATEST_PUBLISHED</code> - the most
   * recent published version.</p> </li> </ul>
   */
  inline const Aws::String& GetRecipeVersion() const { return m_recipeVersion; }
  inline bool RecipeVersionHasBeenSet() const { return m_recipeVersionHasBeenSet; }
  template <typename RecipeVersionT = Aws::String>
  void SetRecipeVersion(RecipeVersionT&& value) {
    m_recipeVersionHasBeenSet = true;
    m_recipeVersion = std::forward<RecipeVersionT>(value);
  }
  template <typename RecipeVersionT = Aws::String>
  Recipe& WithRecipeVersion(RecipeVersionT&& value) {
    SetRecipeVersion(std::forward<RecipeVersionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_createdBy;
  bool m_createdByHasBeenSet = false;

  Aws::Utils::DateTime m_createDate{};
  bool m_createDateHasBeenSet = false;

  Aws::String m_lastModifiedBy;
  bool m_lastModifiedByHasBeenSet = false;

  Aws::Utils::DateTime m_lastModifiedDate{};
  bool m_lastModifiedDateHasBeenSet = false;

  Aws::String m_projectName;
  bool m_projectNameHasBeenSet = false;

  Aws::String m_publishedBy;
  bool m_publishedByHasBeenSet = false;

  Aws::Utils::DateTime m_publishedDate{};
  bool m_publishedDateHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_resourceArn;
  bool m_resourceArnHasBeenSet = false;

  Aws::Vector<RecipeStep> m_steps;
  bool m_stepsHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::String m_recipeVersion;
  bool m_recipeVersionHasBeenSet = false;
};

}  // namespace Model
}  // namespace GlueDataBrew
}  // namespace Aws
