"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.AlgorithmSuite = exports.MessageFormat = exports.CommitmentPolicySuites = exports.CommittingAlgorithmSuiteIdentifier = exports.NonCommittingAlgorithmSuiteIdentifier = exports.SignaturePolicySuites = exports.NonSigningAlgorithmSuiteIdentifier = exports.SignaturePolicy = exports.CommitmentPolicy = exports.AlgorithmSuiteIdentifier = void 0;
/*
 * This file contains information about particular algorithm suites used
 * within the encryption SDK.  In most cases, end-users don't need to
 * manipulate this structure, but it can occasionally be needed for more
 * advanced use cases, such as writing keyrings.
 *
 * Here we describe the overall shape of the Algorithm Suites used by the AWS Encryption
 * SDK for JavaScript.  Specific details for Node.js and WebCrypto can be found
 * in the respective files
 */
const immutable_class_1 = require("./immutable_class");
const needs_1 = require("./needs");
/* References to https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/algorithms-reference.html
 * These define the possible parameters for algorithm specifications that correspond
 * to the Node.js or WebCrypto environment.
 * These parameters are composed into an algorithm suite specification for each
 * environment in the respective files.
 */
var AlgorithmSuiteIdentifier;
(function (AlgorithmSuiteIdentifier) {
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES128_GCM_IV12_TAG16"] = 20] = "ALG_AES128_GCM_IV12_TAG16";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES192_GCM_IV12_TAG16"] = 70] = "ALG_AES192_GCM_IV12_TAG16";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES256_GCM_IV12_TAG16"] = 120] = "ALG_AES256_GCM_IV12_TAG16";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256"] = 276] = "ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256"] = 326] = "ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256"] = 376] = "ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256"] = 532] = "ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384"] = 838] = "ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384"] = 888] = "ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY"] = 1144] = "ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY";
    AlgorithmSuiteIdentifier[AlgorithmSuiteIdentifier["ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384"] = 1400] = "ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384";
})(AlgorithmSuiteIdentifier = exports.AlgorithmSuiteIdentifier || (exports.AlgorithmSuiteIdentifier = {}));
Object.freeze(AlgorithmSuiteIdentifier);
var CommitmentPolicy;
(function (CommitmentPolicy) {
    CommitmentPolicy["FORBID_ENCRYPT_ALLOW_DECRYPT"] = "FORBID_ENCRYPT_ALLOW_DECRYPT";
    CommitmentPolicy["REQUIRE_ENCRYPT_ALLOW_DECRYPT"] = "REQUIRE_ENCRYPT_ALLOW_DECRYPT";
    CommitmentPolicy["REQUIRE_ENCRYPT_REQUIRE_DECRYPT"] = "REQUIRE_ENCRYPT_REQUIRE_DECRYPT";
})(CommitmentPolicy = exports.CommitmentPolicy || (exports.CommitmentPolicy = {}));
Object.freeze(CommitmentPolicy);
var SignaturePolicy;
(function (SignaturePolicy) {
    SignaturePolicy["ALLOW_ENCRYPT_ALLOW_DECRYPT"] = "ALLOW_ENCRYPT_ALLOW_DECRYPT";
    SignaturePolicy["ALLOW_ENCRYPT_FORBID_DECRYPT"] = "ALLOW_ENCRYPT_FORBID_DECRYPT";
})(SignaturePolicy = exports.SignaturePolicy || (exports.SignaturePolicy = {}));
Object.freeze(SignaturePolicy);
exports.NonSigningAlgorithmSuiteIdentifier = (() => {
    const { ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256, ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384, 
    // Both the name side above, and the id side below
    [0x0214]: NAME_ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256, [0x0346]: NAME_ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, [0x0378]: NAME_ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, [0x0578]: NAME_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384, ...NonSigningAlgorithmSuiteIdentifier } = AlgorithmSuiteIdentifier;
    return NonSigningAlgorithmSuiteIdentifier;
})();
exports.SignaturePolicySuites = Object.freeze({
    isDecryptEnabled(signaturePolicy, suite, messageId) {
        const id = suite.id || suite;
        const name = suite.name || AlgorithmSuiteIdentifier[id];
        let decryption_client_name = 'decryptStream';
        let signature_description = 'signed';
        if (signaturePolicy === SignaturePolicy.ALLOW_ENCRYPT_FORBID_DECRYPT) {
            decryption_client_name = 'decryptUnsignedMessageStream';
            signature_description = 'un-signed';
        }
        /* Precondition: Only handle DecryptionMaterial for algorithm suites supported in signaturePolicy. */
        (0, needs_1.needs)(this[signaturePolicy].decryptEnabledSuites[id], `Configuration conflict. ` +
            `Cannot process message with ID ${messageId} ` +
            `due to client method ${decryption_client_name} ` +
            `requiring only ${signature_description} messages. ` +
            `Algorithm ID was ${name}. ` +
            `See: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/concepts.html#digital-sigs`);
    },
    [SignaturePolicy.ALLOW_ENCRYPT_ALLOW_DECRYPT]: Object.freeze({
        decryptEnabledSuites: AlgorithmSuiteIdentifier,
        defaultAlgorithmSuite: AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384,
    }),
    [SignaturePolicy.ALLOW_ENCRYPT_FORBID_DECRYPT]: Object.freeze({
        decryptEnabledSuites: exports.NonSigningAlgorithmSuiteIdentifier,
        defaultAlgorithmSuite: AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY,
    }),
});
exports.NonCommittingAlgorithmSuiteIdentifier = (() => {
    const { ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY, ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384, 
    // Both the name side above, and the id side below
    [0x0478]: NAME_ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY, [0x0578]: NAME_ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384, ...NonCommittingAlgorithmSuiteIdentifier } = AlgorithmSuiteIdentifier;
    return NonCommittingAlgorithmSuiteIdentifier;
})();
exports.CommittingAlgorithmSuiteIdentifier = (() => {
    const { ALG_AES128_GCM_IV12_TAG16, ALG_AES192_GCM_IV12_TAG16, ALG_AES256_GCM_IV12_TAG16, ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256, ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256, ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256, ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256, ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, 
    // Both the name side above, and the id side below
    [0x0014]: NAME_ALG_AES128_GCM_IV12_TAG16, [0x0046]: NAME_ALG_AES192_GCM_IV12_TAG16, [0x0078]: NAME_ALG_AES256_GCM_IV12_TAG16, [0x0114]: NAME_ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256, [0x0146]: NAME_ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256, [0x0178]: NAME_ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256, [0x0214]: NAME_ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256, [0x0346]: NAME_ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, [0x0378]: NAME_ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384, ...CommittingAlgorithmSuiteIdentifier } = AlgorithmSuiteIdentifier;
    return CommittingAlgorithmSuiteIdentifier;
})();
exports.CommitmentPolicySuites = Object.freeze({
    isEncryptEnabled(commitmentPolicy, suite) {
        if (!suite)
            return;
        const id = suite.id || suite;
        const name = suite.name || AlgorithmSuiteIdentifier[id];
        /* Precondition: Only handle EncryptionMaterial for algorithm suites supported in commitmentPolicy. */
        (0, needs_1.needs)(this[commitmentPolicy].encryptEnabledSuites[id], `Configuration conflict. ` +
            `Cannot encrypt due to CommitmentPolicy ${commitmentPolicy} ` +
            `requiring only non-committed messages. ` +
            `Algorithm ID was ${name}. ` +
            `See: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/troubleshooting-migration.html`);
    },
    isDecryptEnabled(commitmentPolicy, suite, messageId) {
        const id = suite.id || suite;
        const name = suite.name || AlgorithmSuiteIdentifier[id];
        /* Precondition: Only handle DecryptionMaterial for algorithm suites supported in commitmentPolicy. */
        (0, needs_1.needs)(this[commitmentPolicy].decryptEnabledSuites[id], `Configuration conflict. ` +
            `Cannot process message with ID ${messageId} ` +
            `due to CommitmentPolicy ${commitmentPolicy} ` +
            `requiring only committed messages. ` +
            `Algorithm ID was ${name}. ` +
            `See: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/troubleshooting-migration.html`);
    },
    [CommitmentPolicy.FORBID_ENCRYPT_ALLOW_DECRYPT]: Object.freeze({
        encryptEnabledSuites: exports.NonCommittingAlgorithmSuiteIdentifier,
        decryptEnabledSuites: AlgorithmSuiteIdentifier,
        defaultAlgorithmSuite: exports.NonCommittingAlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384,
    }),
    [CommitmentPolicy.REQUIRE_ENCRYPT_ALLOW_DECRYPT]: Object.freeze({
        encryptEnabledSuites: exports.CommittingAlgorithmSuiteIdentifier,
        decryptEnabledSuites: AlgorithmSuiteIdentifier,
        defaultAlgorithmSuite: exports.CommittingAlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384,
    }),
    [CommitmentPolicy.REQUIRE_ENCRYPT_REQUIRE_DECRYPT]: Object.freeze({
        encryptEnabledSuites: exports.CommittingAlgorithmSuiteIdentifier,
        decryptEnabledSuites: exports.CommittingAlgorithmSuiteIdentifier,
        defaultAlgorithmSuite: exports.CommittingAlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384,
    }),
});
var MessageFormat;
(function (MessageFormat) {
    MessageFormat[MessageFormat["V1"] = 1] = "V1";
    MessageFormat[MessageFormat["V2"] = 2] = "V2";
})(MessageFormat = exports.MessageFormat || (exports.MessageFormat = {}));
Object.freeze(MessageFormat);
class AlgorithmSuite {
    constructor(suiteValues) {
        (0, needs_1.needs)(this.constructor !== AlgorithmSuite, 'new AlgorithmSuite is not allowed');
        /* Precondition: A algorithm suite specification must be passed. */
        (0, needs_1.needs)(suiteValues, 'Algorithm specification not set.');
        /* Precondition: The Algorithm Suite Identifier must exist. */
        (0, needs_1.needs)(AlgorithmSuiteIdentifier[suiteValues.id], 'No suite by that identifier exists.');
        Object.assign(this, suiteValues);
        (0, immutable_class_1.readOnlyProperty)(this, 'keyLengthBytes', this.keyLength / 8);
        (0, immutable_class_1.readOnlyProperty)(this, 'name', AlgorithmSuiteIdentifier[this.id]);
    }
}
exports.AlgorithmSuite = AlgorithmSuite;
(0, immutable_class_1.immutableClass)(AlgorithmSuite);
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWxnb3JpdGhtX3N1aXRlcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9hbGdvcml0aG1fc3VpdGVzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSxvRUFBb0U7QUFDcEUsc0NBQXNDOzs7QUFFdEM7Ozs7Ozs7OztHQVNHO0FBRUgsdURBQW9FO0FBQ3BFLG1DQUErQjtBQUUvQjs7Ozs7R0FLRztBQUNILElBQVksd0JBWVg7QUFaRCxXQUFZLHdCQUF3QjtJQUNsQyxrSEFBb0MsQ0FBQTtJQUNwQyxrSEFBb0MsQ0FBQTtJQUNwQyxtSEFBb0MsQ0FBQTtJQUNwQywySUFBZ0QsQ0FBQTtJQUNoRCwySUFBZ0QsQ0FBQTtJQUNoRCwySUFBZ0QsQ0FBQTtJQUNoRCxpS0FBMkQsQ0FBQTtJQUMzRCxpS0FBMkQsQ0FBQTtJQUMzRCxpS0FBMkQsQ0FBQTtJQUMzRCxrS0FBMkQsQ0FBQTtJQUMzRCx3TEFBc0UsQ0FBQTtBQUN4RSxDQUFDLEVBWlcsd0JBQXdCLEdBQXhCLGdDQUF3QixLQUF4QixnQ0FBd0IsUUFZbkM7QUFDRCxNQUFNLENBQUMsTUFBTSxDQUFDLHdCQUF3QixDQUFDLENBQUE7QUFFdkMsSUFBWSxnQkFJWDtBQUpELFdBQVksZ0JBQWdCO0lBQzFCLGlGQUErRCxDQUFBO0lBQy9ELG1GQUFpRSxDQUFBO0lBQ2pFLHVGQUFxRSxDQUFBO0FBQ3ZFLENBQUMsRUFKVyxnQkFBZ0IsR0FBaEIsd0JBQWdCLEtBQWhCLHdCQUFnQixRQUkzQjtBQUNELE1BQU0sQ0FBQyxNQUFNLENBQUMsZ0JBQWdCLENBQUMsQ0FBQTtBQUUvQixJQUFZLGVBR1g7QUFIRCxXQUFZLGVBQWU7SUFDekIsOEVBQTZELENBQUE7SUFDN0QsZ0ZBQStELENBQUE7QUFDakUsQ0FBQyxFQUhXLGVBQWUsR0FBZix1QkFBZSxLQUFmLHVCQUFlLFFBRzFCO0FBQ0QsTUFBTSxDQUFDLE1BQU0sQ0FBQyxlQUFlLENBQUMsQ0FBQTtBQXVDakIsUUFBQSxrQ0FBa0MsR0FBRyxDQUFDLEdBQUcsRUFBRTtJQUN0RCxNQUFNLEVBQ0osZ0RBQWdELEVBQ2hELGdEQUFnRCxFQUNoRCxnREFBZ0QsRUFDaEQsMkRBQTJEO0lBQzNELGtEQUFrRDtJQUNsRCxDQUFDLE1BQU0sQ0FBQyxFQUFFLHFEQUFxRCxFQUMvRCxDQUFDLE1BQU0sQ0FBQyxFQUFFLHFEQUFxRCxFQUMvRCxDQUFDLE1BQU0sQ0FBQyxFQUFFLHFEQUFxRCxFQUMvRCxDQUFDLE1BQU0sQ0FBQyxFQUFFLDREQUE0RCxFQUN0RSxHQUFHLGtDQUFrQyxFQUN0QyxHQUFHLHdCQUF3QixDQUFBO0lBQzVCLE9BQU8sa0NBQWtDLENBQUE7QUFDM0MsQ0FBQyxDQUFDLEVBQUUsQ0FBQTtBQUVTLFFBQUEscUJBQXFCLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQztJQUNqRCxnQkFBZ0IsQ0FDZCxlQUFnQyxFQUNoQyxLQUFnRCxFQUNoRCxTQUFpQjtRQUVqQixNQUFNLEVBQUUsR0FBSSxLQUF3QixDQUFDLEVBQUUsSUFBSSxLQUFLLENBQUE7UUFDaEQsTUFBTSxJQUFJLEdBQUksS0FBd0IsQ0FBQyxJQUFJLElBQUksd0JBQXdCLENBQUMsRUFBRSxDQUFDLENBQUE7UUFDM0UsSUFBSSxzQkFBc0IsR0FBRyxlQUFlLENBQUE7UUFDNUMsSUFBSSxxQkFBcUIsR0FBRyxRQUFRLENBQUE7UUFDcEMsSUFBSSxlQUFlLEtBQUssZUFBZSxDQUFDLDRCQUE0QixFQUFFO1lBQ3BFLHNCQUFzQixHQUFHLDhCQUE4QixDQUFBO1lBQ3ZELHFCQUFxQixHQUFHLFdBQVcsQ0FBQTtTQUNwQztRQUVELHFHQUFxRztRQUNyRyxJQUFBLGFBQUssRUFDSCxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUMsb0JBQW9CLENBQUMsRUFBRSxDQUFDLEVBQzlDLDBCQUEwQjtZQUN4QixrQ0FBa0MsU0FBUyxHQUFHO1lBQzlDLHdCQUF3QixzQkFBc0IsR0FBRztZQUNqRCxrQkFBa0IscUJBQXFCLGFBQWE7WUFDcEQsb0JBQW9CLElBQUksSUFBSTtZQUM1QixtR0FBbUcsQ0FDdEcsQ0FBQTtJQUNILENBQUM7SUFDRCxDQUFDLGVBQWUsQ0FBQywyQkFBMkIsQ0FBQyxFQUFFLE1BQU0sQ0FBQyxNQUFNLENBQUM7UUFDM0Qsb0JBQW9CLEVBQUUsd0JBQXdCO1FBQzlDLHFCQUFxQixFQUNuQix3QkFBd0IsQ0FBQywyREFBMkQ7S0FDdkYsQ0FBQztJQUNGLENBQUMsZUFBZSxDQUFDLDRCQUE0QixDQUFDLEVBQUUsTUFBTSxDQUFDLE1BQU0sQ0FBQztRQUM1RCxvQkFBb0IsRUFBRSwwQ0FBa0M7UUFDeEQscUJBQXFCLEVBQ25CLHdCQUF3QixDQUFDLGdEQUFnRDtLQUM1RSxDQUFDO0NBQ0gsQ0FBQyxDQUFBO0FBUVcsUUFBQSxxQ0FBcUMsR0FBRyxDQUFDLEdBQUcsRUFBRTtJQUN6RCxNQUFNLEVBQ0osZ0RBQWdELEVBQ2hELDJEQUEyRDtJQUMzRCxrREFBa0Q7SUFDbEQsQ0FBQyxNQUFNLENBQUMsRUFBRSxxREFBcUQsRUFDL0QsQ0FBQyxNQUFNLENBQUMsRUFBRSxnRUFBZ0UsRUFDMUUsR0FBRyxxQ0FBcUMsRUFDekMsR0FBRyx3QkFBd0IsQ0FBQTtJQUM1QixPQUFPLHFDQUFxQyxDQUFBO0FBQzlDLENBQUMsQ0FBQyxFQUFFLENBQUE7QUFVUyxRQUFBLGtDQUFrQyxHQUFHLENBQUMsR0FBRyxFQUFFO0lBQ3RELE1BQU0sRUFDSix5QkFBeUIsRUFDekIseUJBQXlCLEVBQ3pCLHlCQUF5QixFQUN6QixxQ0FBcUMsRUFDckMscUNBQXFDLEVBQ3JDLHFDQUFxQyxFQUNyQyxnREFBZ0QsRUFDaEQsZ0RBQWdELEVBQ2hELGdEQUFnRDtJQUNoRCxrREFBa0Q7SUFDbEQsQ0FBQyxNQUFNLENBQUMsRUFBRSw4QkFBOEIsRUFDeEMsQ0FBQyxNQUFNLENBQUMsRUFBRSw4QkFBOEIsRUFDeEMsQ0FBQyxNQUFNLENBQUMsRUFBRSw4QkFBOEIsRUFDeEMsQ0FBQyxNQUFNLENBQUMsRUFBRSwwQ0FBMEMsRUFDcEQsQ0FBQyxNQUFNLENBQUMsRUFBRSwwQ0FBMEMsRUFDcEQsQ0FBQyxNQUFNLENBQUMsRUFBRSwwQ0FBMEMsRUFDcEQsQ0FBQyxNQUFNLENBQUMsRUFBRSxxREFBcUQsRUFDL0QsQ0FBQyxNQUFNLENBQUMsRUFBRSxxREFBcUQsRUFDL0QsQ0FBQyxNQUFNLENBQUMsRUFBRSxxREFBcUQsRUFDL0QsR0FBRyxrQ0FBa0MsRUFDdEMsR0FBRyx3QkFBd0IsQ0FBQTtJQUM1QixPQUFPLGtDQUFrQyxDQUFBO0FBQzNDLENBQUMsQ0FBQyxFQUFFLENBQUE7QUFFUyxRQUFBLHNCQUFzQixHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUM7SUFDbEQsZ0JBQWdCLENBQ2QsZ0JBQWtDLEVBQ2xDLEtBQWlEO1FBRWpELElBQUksQ0FBQyxLQUFLO1lBQUUsT0FBTTtRQUNsQixNQUFNLEVBQUUsR0FBSSxLQUF3QixDQUFDLEVBQUUsSUFBSSxLQUFLLENBQUE7UUFDaEQsTUFBTSxJQUFJLEdBQUksS0FBd0IsQ0FBQyxJQUFJLElBQUksd0JBQXdCLENBQUMsRUFBRSxDQUFDLENBQUE7UUFFM0Usc0dBQXNHO1FBQ3RHLElBQUEsYUFBSyxFQUNILElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLG9CQUFvQixDQUFDLEVBQUUsQ0FBQyxFQUMvQywwQkFBMEI7WUFDeEIsMENBQTBDLGdCQUFnQixHQUFHO1lBQzdELHlDQUF5QztZQUN6QyxvQkFBb0IsSUFBSSxJQUFJO1lBQzVCLHVHQUF1RyxDQUMxRyxDQUFBO0lBQ0gsQ0FBQztJQUNELGdCQUFnQixDQUNkLGdCQUFrQyxFQUNsQyxLQUFnRCxFQUNoRCxTQUFpQjtRQUVqQixNQUFNLEVBQUUsR0FBSSxLQUF3QixDQUFDLEVBQUUsSUFBSSxLQUFLLENBQUE7UUFDaEQsTUFBTSxJQUFJLEdBQUksS0FBd0IsQ0FBQyxJQUFJLElBQUksd0JBQXdCLENBQUMsRUFBRSxDQUFDLENBQUE7UUFFM0Usc0dBQXNHO1FBQ3RHLElBQUEsYUFBSyxFQUNILElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLG9CQUFvQixDQUFDLEVBQUUsQ0FBQyxFQUMvQywwQkFBMEI7WUFDeEIsa0NBQWtDLFNBQVMsR0FBRztZQUM5QywyQkFBMkIsZ0JBQWdCLEdBQUc7WUFDOUMscUNBQXFDO1lBQ3JDLG9CQUFvQixJQUFJLElBQUk7WUFDNUIsdUdBQXVHLENBQzFHLENBQUE7SUFDSCxDQUFDO0lBQ0QsQ0FBQyxnQkFBZ0IsQ0FBQyw0QkFBNEIsQ0FBQyxFQUFFLE1BQU0sQ0FBQyxNQUFNLENBQUM7UUFDN0Qsb0JBQW9CLEVBQUUsNkNBQXFDO1FBQzNELG9CQUFvQixFQUFFLHdCQUF3QjtRQUM5QyxxQkFBcUIsRUFDbkIsNkNBQXFDLENBQUMsZ0RBQWdEO0tBQ3pGLENBQUM7SUFDRixDQUFDLGdCQUFnQixDQUFDLDZCQUE2QixDQUFDLEVBQUUsTUFBTSxDQUFDLE1BQU0sQ0FBQztRQUM5RCxvQkFBb0IsRUFBRSwwQ0FBa0M7UUFDeEQsb0JBQW9CLEVBQUUsd0JBQXdCO1FBQzlDLHFCQUFxQixFQUNuQiwwQ0FBa0MsQ0FBQywyREFBMkQ7S0FDakcsQ0FBQztJQUNGLENBQUMsZ0JBQWdCLENBQUMsK0JBQStCLENBQUMsRUFBRSxNQUFNLENBQUMsTUFBTSxDQUFDO1FBQ2hFLG9CQUFvQixFQUFFLDBDQUFrQztRQUN4RCxvQkFBb0IsRUFBRSwwQ0FBa0M7UUFDeEQscUJBQXFCLEVBQ25CLDBDQUFrQyxDQUFDLDJEQUEyRDtLQUNqRyxDQUFDO0NBQ0gsQ0FBQyxDQUFBO0FBbUJGLElBQVksYUFHWDtBQUhELFdBQVksYUFBYTtJQUN2Qiw2Q0FBUyxDQUFBO0lBQ1QsNkNBQVMsQ0FBQTtBQUNYLENBQUMsRUFIVyxhQUFhLEdBQWIscUJBQWEsS0FBYixxQkFBYSxRQUd4QjtBQUNELE1BQU0sQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLENBQUE7QUE0RDVCLE1BQXNCLGNBQWM7SUFvQmxDLFlBQVksV0FBcUI7UUFDL0IsSUFBQSxhQUFLLEVBQ0gsSUFBSSxDQUFDLFdBQVcsS0FBSyxjQUFjLEVBQ25DLG1DQUFtQyxDQUNwQyxDQUFBO1FBQ0QsbUVBQW1FO1FBQ25FLElBQUEsYUFBSyxFQUFDLFdBQVcsRUFBRSxrQ0FBa0MsQ0FBQyxDQUFBO1FBQ3RELDhEQUE4RDtRQUM5RCxJQUFBLGFBQUssRUFDSCx3QkFBd0IsQ0FBQyxXQUFXLENBQUMsRUFBRSxDQUFDLEVBQ3hDLHFDQUFxQyxDQUN0QyxDQUFBO1FBQ0QsTUFBTSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQUE7UUFFaEMsSUFBQSxrQ0FBZ0IsRUFBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLFNBQVMsR0FBRyxDQUFDLENBQUMsQ0FBQTtRQUM1RCxJQUFBLGtDQUFnQixFQUNkLElBQUksRUFDSixNQUFNLEVBQ04sd0JBQXdCLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBdUIsQ0FDeEQsQ0FBQTtJQUNILENBQUM7Q0FDRjtBQXpDRCx3Q0F5Q0M7QUFDRCxJQUFBLGdDQUFjLEVBQUMsY0FBYyxDQUFDLENBQUEifQ==