﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iot/IoTRequest.h>
#include <aws/iot/IoT_EXPORTS.h>
#include <aws/iot/model/ResourceIdentifier.h>

#include <utility>

namespace Aws {
namespace IoT {
namespace Model {

/**
 */
class ListAuditFindingsRequest : public IoTRequest {
 public:
  AWS_IOT_API ListAuditFindingsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListAuditFindings"; }

  AWS_IOT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A filter to limit results to the audit with the specified ID. You must
   * specify either the taskId or the startTime and endTime, but not both.</p>
   */
  inline const Aws::String& GetTaskId() const { return m_taskId; }
  inline bool TaskIdHasBeenSet() const { return m_taskIdHasBeenSet; }
  template <typename TaskIdT = Aws::String>
  void SetTaskId(TaskIdT&& value) {
    m_taskIdHasBeenSet = true;
    m_taskId = std::forward<TaskIdT>(value);
  }
  template <typename TaskIdT = Aws::String>
  ListAuditFindingsRequest& WithTaskId(TaskIdT&& value) {
    SetTaskId(std::forward<TaskIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter to limit results to the findings for the specified audit check.</p>
   */
  inline const Aws::String& GetCheckName() const { return m_checkName; }
  inline bool CheckNameHasBeenSet() const { return m_checkNameHasBeenSet; }
  template <typename CheckNameT = Aws::String>
  void SetCheckName(CheckNameT&& value) {
    m_checkNameHasBeenSet = true;
    m_checkName = std::forward<CheckNameT>(value);
  }
  template <typename CheckNameT = Aws::String>
  ListAuditFindingsRequest& WithCheckName(CheckNameT&& value) {
    SetCheckName(std::forward<CheckNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information identifying the noncompliant resource.</p>
   */
  inline const ResourceIdentifier& GetResourceIdentifier() const { return m_resourceIdentifier; }
  inline bool ResourceIdentifierHasBeenSet() const { return m_resourceIdentifierHasBeenSet; }
  template <typename ResourceIdentifierT = ResourceIdentifier>
  void SetResourceIdentifier(ResourceIdentifierT&& value) {
    m_resourceIdentifierHasBeenSet = true;
    m_resourceIdentifier = std::forward<ResourceIdentifierT>(value);
  }
  template <typename ResourceIdentifierT = ResourceIdentifier>
  ListAuditFindingsRequest& WithResourceIdentifier(ResourceIdentifierT&& value) {
    SetResourceIdentifier(std::forward<ResourceIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return at one time. The default is 25.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListAuditFindingsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next set of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListAuditFindingsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter to limit results to those found after the specified time. You must
   * specify either the startTime and endTime or the taskId, but not both.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  ListAuditFindingsRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter to limit results to those found before the specified time. You must
   * specify either the startTime and endTime or the taskId, but not both.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  ListAuditFindingsRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Boolean flag indicating whether only the suppressed findings or the
   * unsuppressed findings should be listed. If this parameter isn't provided, the
   * response will list both suppressed and unsuppressed findings. </p>
   */
  inline bool GetListSuppressedFindings() const { return m_listSuppressedFindings; }
  inline bool ListSuppressedFindingsHasBeenSet() const { return m_listSuppressedFindingsHasBeenSet; }
  inline void SetListSuppressedFindings(bool value) {
    m_listSuppressedFindingsHasBeenSet = true;
    m_listSuppressedFindings = value;
  }
  inline ListAuditFindingsRequest& WithListSuppressedFindings(bool value) {
    SetListSuppressedFindings(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_taskId;
  bool m_taskIdHasBeenSet = false;

  Aws::String m_checkName;
  bool m_checkNameHasBeenSet = false;

  ResourceIdentifier m_resourceIdentifier;
  bool m_resourceIdentifierHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  Aws::Utils::DateTime m_startTime{};
  bool m_startTimeHasBeenSet = false;

  Aws::Utils::DateTime m_endTime{};
  bool m_endTimeHasBeenSet = false;

  bool m_listSuppressedFindings{false};
  bool m_listSuppressedFindingsHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoT
}  // namespace Aws
