﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codecommit/CodeCommitRequest.h>
#include <aws/codecommit/CodeCommit_EXPORTS.h>
#include <aws/codecommit/model/ConflictDetailLevelTypeEnum.h>
#include <aws/codecommit/model/ConflictResolution.h>
#include <aws/codecommit/model/ConflictResolutionStrategyTypeEnum.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodeCommit {
namespace Model {

/**
 */
class MergeBranchesByThreeWayRequest : public CodeCommitRequest {
 public:
  AWS_CODECOMMIT_API MergeBranchesByThreeWayRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "MergeBranchesByThreeWay"; }

  AWS_CODECOMMIT_API Aws::String SerializePayload() const override;

  AWS_CODECOMMIT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the repository where you want to merge two branches.</p>
   */
  inline const Aws::String& GetRepositoryName() const { return m_repositoryName; }
  inline bool RepositoryNameHasBeenSet() const { return m_repositoryNameHasBeenSet; }
  template <typename RepositoryNameT = Aws::String>
  void SetRepositoryName(RepositoryNameT&& value) {
    m_repositoryNameHasBeenSet = true;
    m_repositoryName = std::forward<RepositoryNameT>(value);
  }
  template <typename RepositoryNameT = Aws::String>
  MergeBranchesByThreeWayRequest& WithRepositoryName(RepositoryNameT&& value) {
    SetRepositoryName(std::forward<RepositoryNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The branch, tag, HEAD, or other fully qualified reference used to identify a
   * commit (for example, a branch name or a full commit ID).</p>
   */
  inline const Aws::String& GetSourceCommitSpecifier() const { return m_sourceCommitSpecifier; }
  inline bool SourceCommitSpecifierHasBeenSet() const { return m_sourceCommitSpecifierHasBeenSet; }
  template <typename SourceCommitSpecifierT = Aws::String>
  void SetSourceCommitSpecifier(SourceCommitSpecifierT&& value) {
    m_sourceCommitSpecifierHasBeenSet = true;
    m_sourceCommitSpecifier = std::forward<SourceCommitSpecifierT>(value);
  }
  template <typename SourceCommitSpecifierT = Aws::String>
  MergeBranchesByThreeWayRequest& WithSourceCommitSpecifier(SourceCommitSpecifierT&& value) {
    SetSourceCommitSpecifier(std::forward<SourceCommitSpecifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The branch, tag, HEAD, or other fully qualified reference used to identify a
   * commit (for example, a branch name or a full commit ID).</p>
   */
  inline const Aws::String& GetDestinationCommitSpecifier() const { return m_destinationCommitSpecifier; }
  inline bool DestinationCommitSpecifierHasBeenSet() const { return m_destinationCommitSpecifierHasBeenSet; }
  template <typename DestinationCommitSpecifierT = Aws::String>
  void SetDestinationCommitSpecifier(DestinationCommitSpecifierT&& value) {
    m_destinationCommitSpecifierHasBeenSet = true;
    m_destinationCommitSpecifier = std::forward<DestinationCommitSpecifierT>(value);
  }
  template <typename DestinationCommitSpecifierT = Aws::String>
  MergeBranchesByThreeWayRequest& WithDestinationCommitSpecifier(DestinationCommitSpecifierT&& value) {
    SetDestinationCommitSpecifier(std::forward<DestinationCommitSpecifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The branch where the merge is applied. </p>
   */
  inline const Aws::String& GetTargetBranch() const { return m_targetBranch; }
  inline bool TargetBranchHasBeenSet() const { return m_targetBranchHasBeenSet; }
  template <typename TargetBranchT = Aws::String>
  void SetTargetBranch(TargetBranchT&& value) {
    m_targetBranchHasBeenSet = true;
    m_targetBranch = std::forward<TargetBranchT>(value);
  }
  template <typename TargetBranchT = Aws::String>
  MergeBranchesByThreeWayRequest& WithTargetBranch(TargetBranchT&& value) {
    SetTargetBranch(std::forward<TargetBranchT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The level of conflict detail to use. If unspecified, the default FILE_LEVEL
   * is used, which returns a not-mergeable result if the same file has differences
   * in both branches. If LINE_LEVEL is specified, a conflict is considered not
   * mergeable if the same file in both branches has differences on the same
   * line.</p>
   */
  inline ConflictDetailLevelTypeEnum GetConflictDetailLevel() const { return m_conflictDetailLevel; }
  inline bool ConflictDetailLevelHasBeenSet() const { return m_conflictDetailLevelHasBeenSet; }
  inline void SetConflictDetailLevel(ConflictDetailLevelTypeEnum value) {
    m_conflictDetailLevelHasBeenSet = true;
    m_conflictDetailLevel = value;
  }
  inline MergeBranchesByThreeWayRequest& WithConflictDetailLevel(ConflictDetailLevelTypeEnum value) {
    SetConflictDetailLevel(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies which branch to use when resolving conflicts, or whether to attempt
   * automatically merging two versions of a file. The default is NONE, which
   * requires any conflicts to be resolved manually before the merge operation is
   * successful.</p>
   */
  inline ConflictResolutionStrategyTypeEnum GetConflictResolutionStrategy() const { return m_conflictResolutionStrategy; }
  inline bool ConflictResolutionStrategyHasBeenSet() const { return m_conflictResolutionStrategyHasBeenSet; }
  inline void SetConflictResolutionStrategy(ConflictResolutionStrategyTypeEnum value) {
    m_conflictResolutionStrategyHasBeenSet = true;
    m_conflictResolutionStrategy = value;
  }
  inline MergeBranchesByThreeWayRequest& WithConflictResolutionStrategy(ConflictResolutionStrategyTypeEnum value) {
    SetConflictResolutionStrategy(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the author who created the commit. This information is used as
   * both the author and committer for the commit.</p>
   */
  inline const Aws::String& GetAuthorName() const { return m_authorName; }
  inline bool AuthorNameHasBeenSet() const { return m_authorNameHasBeenSet; }
  template <typename AuthorNameT = Aws::String>
  void SetAuthorName(AuthorNameT&& value) {
    m_authorNameHasBeenSet = true;
    m_authorName = std::forward<AuthorNameT>(value);
  }
  template <typename AuthorNameT = Aws::String>
  MergeBranchesByThreeWayRequest& WithAuthorName(AuthorNameT&& value) {
    SetAuthorName(std::forward<AuthorNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The email address of the person merging the branches. This information is
   * used in the commit information for the merge.</p>
   */
  inline const Aws::String& GetEmail() const { return m_email; }
  inline bool EmailHasBeenSet() const { return m_emailHasBeenSet; }
  template <typename EmailT = Aws::String>
  void SetEmail(EmailT&& value) {
    m_emailHasBeenSet = true;
    m_email = std::forward<EmailT>(value);
  }
  template <typename EmailT = Aws::String>
  MergeBranchesByThreeWayRequest& WithEmail(EmailT&& value) {
    SetEmail(std::forward<EmailT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The commit message to include in the commit information for the merge.</p>
   */
  inline const Aws::String& GetCommitMessage() const { return m_commitMessage; }
  inline bool CommitMessageHasBeenSet() const { return m_commitMessageHasBeenSet; }
  template <typename CommitMessageT = Aws::String>
  void SetCommitMessage(CommitMessageT&& value) {
    m_commitMessageHasBeenSet = true;
    m_commitMessage = std::forward<CommitMessageT>(value);
  }
  template <typename CommitMessageT = Aws::String>
  MergeBranchesByThreeWayRequest& WithCommitMessage(CommitMessageT&& value) {
    SetCommitMessage(std::forward<CommitMessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If the commit contains deletions, whether to keep a folder or folder
   * structure if the changes leave the folders empty. If true, a .gitkeep file is
   * created for empty folders. The default is false.</p>
   */
  inline bool GetKeepEmptyFolders() const { return m_keepEmptyFolders; }
  inline bool KeepEmptyFoldersHasBeenSet() const { return m_keepEmptyFoldersHasBeenSet; }
  inline void SetKeepEmptyFolders(bool value) {
    m_keepEmptyFoldersHasBeenSet = true;
    m_keepEmptyFolders = value;
  }
  inline MergeBranchesByThreeWayRequest& WithKeepEmptyFolders(bool value) {
    SetKeepEmptyFolders(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If AUTOMERGE is the conflict resolution strategy, a list of inputs to use
   * when resolving conflicts during a merge.</p>
   */
  inline const ConflictResolution& GetConflictResolution() const { return m_conflictResolution; }
  inline bool ConflictResolutionHasBeenSet() const { return m_conflictResolutionHasBeenSet; }
  template <typename ConflictResolutionT = ConflictResolution>
  void SetConflictResolution(ConflictResolutionT&& value) {
    m_conflictResolutionHasBeenSet = true;
    m_conflictResolution = std::forward<ConflictResolutionT>(value);
  }
  template <typename ConflictResolutionT = ConflictResolution>
  MergeBranchesByThreeWayRequest& WithConflictResolution(ConflictResolutionT&& value) {
    SetConflictResolution(std::forward<ConflictResolutionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_repositoryName;

  Aws::String m_sourceCommitSpecifier;

  Aws::String m_destinationCommitSpecifier;

  Aws::String m_targetBranch;

  ConflictDetailLevelTypeEnum m_conflictDetailLevel{ConflictDetailLevelTypeEnum::NOT_SET};

  ConflictResolutionStrategyTypeEnum m_conflictResolutionStrategy{ConflictResolutionStrategyTypeEnum::NOT_SET};

  Aws::String m_authorName;

  Aws::String m_email;

  Aws::String m_commitMessage;

  bool m_keepEmptyFolders{false};

  ConflictResolution m_conflictResolution;
  bool m_repositoryNameHasBeenSet = false;
  bool m_sourceCommitSpecifierHasBeenSet = false;
  bool m_destinationCommitSpecifierHasBeenSet = false;
  bool m_targetBranchHasBeenSet = false;
  bool m_conflictDetailLevelHasBeenSet = false;
  bool m_conflictResolutionStrategyHasBeenSet = false;
  bool m_authorNameHasBeenSet = false;
  bool m_emailHasBeenSet = false;
  bool m_commitMessageHasBeenSet = false;
  bool m_keepEmptyFoldersHasBeenSet = false;
  bool m_conflictResolutionHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeCommit
}  // namespace Aws
