/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.netbeans.modules.websvc.spi.support;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import javax.swing.tree.TreeSelectionModel;
import org.openide.explorer.ExplorerManager;
import org.openide.explorer.view.BeanTreeView;
import org.openide.filesystems.FileObject;
import org.openide.nodes.Node;
import org.openide.util.Lookup;
import org.openide.util.NbBundle;
import org.netbeans.modules.websvc.api.support.InvokeOperationCookie;

/** Default ClientSelection panel.
 *
 * @authorMilan Kuchtiak
 */
public abstract class DefaultClientSelectionPanel
        extends InvokeOperationCookie.ClientSelectionPanel
        implements ExplorerManager.Provider {

    private ExplorerManager manager;
    private FileObject targetFile;
    private Node selectedMethod;
    private BeanTreeView treeView;
    private NodeChangeListener nodeChangeListener;

    /** Constructor for ClientSelectioPanel.
     *
     * @param targetFile fileObject representing target file
     */
    public DefaultClientSelectionPanel(FileObject targetFile) {
        this.targetFile = targetFile;
        manager = new ExplorerManager();
        initComponents();
        initUserComponents();
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        jLblTreeView = new javax.swing.JLabel();

        setLayout(new java.awt.GridBagLayout());

        org.openide.awt.Mnemonics.setLocalizedText(jLblTreeView, NbBundle.getMessage(DefaultClientSelectionPanel.class, "LBL_AvailableWebServices")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(11, 11, 0, 11);
        add(jLblTreeView, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JLabel jLblTreeView;
    // End of variables declaration//GEN-END:variables

    private void initUserComponents() {
        treeView = new BeanTreeView();
        treeView.setRootVisible(false);
        treeView.setPopupAllowed(false);

        java.awt.GridBagConstraints gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.insets = new java.awt.Insets(11, 11, 0, 11);
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        add(treeView, gridBagConstraints);
        jLblTreeView.setLabelFor(treeView.getViewport().getView());
        treeView.getAccessibleContext().setAccessibleName(
                NbBundle.getMessage(DefaultClientSelectionPanel.class, "ACSD_AvailableWebServicesTree"));
        treeView.getAccessibleContext().setAccessibleDescription(
                NbBundle.getMessage(DefaultClientSelectionPanel.class, "ACSD_AvailableWebServicesTree"));
        treeView.setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
    }

    @Override
    public ExplorerManager getExplorerManager() {
        return manager;
    }

    @Override
    public void addNotify() {
        super.addNotify();
        if (nodeChangeListener == null) {
            nodeChangeListener = new NodeChangeListener();
        }
        manager.addPropertyChangeListener(nodeChangeListener);
        manager.setRootContext(getRootContext());
        treeView.expandAll();
    }

    @Override
    public void removeNotify() {
        manager.removePropertyChangeListener(nodeChangeListener);
        super.removeNotify();
    }

    public void propertyChange(PropertyChangeEvent evt) {
        if (evt.getSource() == manager) {
            if (ExplorerManager.PROP_SELECTED_NODES.equals(evt.getPropertyName())) {
                Node[] nodes = manager.getSelectedNodes();
                if (nodes != null && nodes.length > 0) {
                    Node node = nodes[0];
                    if (isClientNode(node)) {
                        // This is a method node.
                        selectedMethod = node;
                        setSelectionValid(true);
                    } else {
                        // This is not a method node.
                        selectedMethod = null;
                        setSelectionValid(false);
                    }
                }
            }
        }
    }

    @Override
    public Lookup getSelectedClient() {
        return selectedMethod.getLookup();
    }

    public final FileObject getTargetFile() {
        return targetFile;
    }

    /** Give root node for bean tree view.
     *
     * @return root node
     */
    protected abstract Node getRootContext();

    /** Check if selected node represents a client.
     *
     * @param node selected node
     * @return triue or false
     */
    protected abstract boolean isClientNode(Node node);

    private class NodeChangeListener implements PropertyChangeListener {

        public void propertyChange(PropertyChangeEvent evt) {
            if (evt.getSource() == manager) {
                if (ExplorerManager.PROP_SELECTED_NODES.equals(evt.getPropertyName())) {
                    Node[] nodes = manager.getSelectedNodes();
                    if (nodes != null && nodes.length > 0) {
                        Node node = nodes[0];
                        if (isClientNode(node)) {
                            // This is a method node.
                            selectedMethod = node;
                            setSelectionValid(true);
                        } else {
                            // This is not a method node.
                            selectedMethod = null;
                            setSelectionValid(false);
                        }
                    }
                }
            }
        }

    }

    protected BeanTreeView getTreeView() {
        return treeView;
    }

}
