////////////////////////////////////////////////////////////////////////////////
//
//  Licensed to the Apache Software Foundation (ASF) under one or more
//  contributor license agreements.  See the NOTICE file distributed with
//  this work for additional information regarding copyright ownership.
//  The ASF licenses this file to You under the Apache License, Version 2.0
//  (the "License"); you may not use this file except in compliance with
//  the License.  You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
////////////////////////////////////////////////////////////////////////////////

package spark.accessibility
{

import mx.accessibility.AccConst;
import mx.core.UIComponent;
import mx.core.mx_internal;

import spark.components.RadioButton;
import spark.components.RadioButtonGroup;

use namespace mx_internal;

/**
 *  RadioButtonAccImpl is the accessibility implementation class
 *  for spark.components.RadioButton.
 *
 *  <p>When a Spark RadioButton is created,
 *  its <code>accessibilityImplementation</code> property
 *  is set to an instance of this class.
 *  The Flash Player then uses this class to allow MSAA clients
 *  such as screen readers to see and manipulate the RadioButton.
 *  See the mx.accessibility.AccImpl and
 *  flash.accessibility.AccessibilityImplementation classes
 *  for background information about accessibility implementatio
 *   classes and MSAA.</p>
 *
 *  <p><b>Children</b></p>
 *
 *  <p>A RadioButton has no MSAA children.</p>
 *
 *  <p><b>Role</b></p>
 *
 *  <p>The MSAA Role of a RadioButton is ROLE_SYSTEM_RADIOBUTTON.</p>
 *
 *  <p><b>Name</b></p>
 *
 *  <p>The MSAA Name of a RadioButton is, by default,
 *  the label that it displays.
 *  When wrapped in a FormItem,
 *  this label will be combined with the FormItem's label.
 *  To override this behavior,
 *  set the RadioButton's <code>accessibilityName</code> property.</p>
 *
 *  <p>When the Name changes,
 *  a RadioButton dispatches the MSAA event EVENT_OBJECT_NAMECHANGE.</p>
 *
 *  <p><b>Description</b></p>
 *
 *  <p>The MSAA Description of a RadioButton is, by default, the empty string,
 *  but you can set the RadioButton's <code>accessibilityDescription</code>
 *  property.</p>
 *
 *  <p><b>State</b></p>
 *
 *  <p>The MSAA state of a RadioButton is a combination of: 
 *  <ul>
 *    <li>STATE_SYSTEM_UNAVAILABLE (when enabled is false)</li>
 *    <li>STATE_SYSTEM_FOCUSABLE (when enabled is true)</li>
 *    <li>STATE_SYSTEM_FOCUSED (when enabled is true
 *    and the RadioButton has focus)</li>
 *    <li>STATE_SYSTEM_CHECKED (when selected is true)</li>
 *  </ul></p>
 *
 *  <p>When the State changes,
 *  a RadioButton dispatches the MSAA event EVENT_OBJECT_STATECHANGE.</p>
 *
 *  <p><b>Value</b></p>
 *
 *  <p>A RadioButton does not have an MSAA Value.</p>
 *
 *  <p><b>Location</b></p>
 *
 *  <p>The MSAA Location of a RadioButton is its bounding rectangle.</p>
 *
 *  <p><b>Default Action</b></p>
 *
 *  <p>The MSAA DefaultAction of a RadioButton is "Check".</p>
 *
 *  <p>When an MSAA client tells the RadioButton to perform this action,
 *  KEY_DOWN and KEY_UP MouseEvents for the SPACE key are generated,
 *  to simulate pressing the RadioButton via the keyboard,
 *  if the RadioButton is enabled.</p>
 *
 *  <p><b>Focus</b></p>
 *
 *  <p>A RadioButton accepts focus.
 *  When it does so, it dispatches the MSAA event EVENT_OBJECT_FOCUS.</p>
 *
 *  <p><b>Selection</b></p>
 *
 *  <p>A RadioButton does not support selection in the MSAA sense.</p>
 *
 *  @langversion 3.0
 *  @playerversion Flash 10
 *  @playerversion AIR 1.5
 *  @productversion Flex 4
 */
public class RadioButtonAccImpl extends CheckBoxAccImpl
{
    include "../core/Version.as";

    //--------------------------------------------------------------------------
    //
    //  Class methods
    //
    //--------------------------------------------------------------------------

    /**
     *  Enables accessibility in the RadioButton class.
     *
     *  <p>This method is called by application startup code
     *  that is autogenerated by the MXML compiler.
     *  Afterwards, when instances of RadioButton are initialized,
     *  their <code>accessibilityImplementation</code> property
     *  will be set to an instance of this class.</p>
     *
     *  @langversion 3.0
     *  @playerversion Flash 10
     *  @playerversion AIR 1.5
     *  @productversion Flex 4
     */
    public static function enableAccessibility():void
    {
        RadioButton.createAccessibilityImplementation =
            createAccessibilityImplementation;
    }

    /**
     *  @private
     *  Creates a RadioButton's AccessibilityImplementation object.
     *  This method is called from UIComponent's
     *  initializeAccessibility() method.
     */
    mx_internal static function createAccessibilityImplementation(
                                component:UIComponent):void
    {
        component.accessibilityImplementation =
            new RadioButtonAccImpl(component);
    }

    //--------------------------------------------------------------------------
    //
    //  Constructor
    //
    //--------------------------------------------------------------------------

    /**
     *  Constructor.
     *
     *  @param master The UIComponent instance that this AccImpl instance
     *  is making accessible.
     *
     *  @langversion 3.0
     *  @playerversion Flash 10
     *  @playerversion AIR 1.5
     *  @productversion Flex 4
     */
    public function RadioButtonAccImpl(master:UIComponent)
    {
        super(master);

        role = AccConst.ROLE_SYSTEM_RADIOBUTTON;
    }

    //--------------------------------------------------------------------------
    //
    //  Overridden methods: AccImpl
    //
    //--------------------------------------------------------------------------

     /**
     *  @private
     *  method for returning the name of the radio button
     *  which is spoken out by the screen reader.
     *  The RadioButton should return the label as the name
     *  with m of n string and ListBox should return the name
     *  specified in the Accessibility Panel.
     *
     *  @param childID uint
     *
     *  @return Name String
     */
    override protected function getName(childID:uint):String
    {

        var name:String = RadioButton(master).label;
        var radioGroup:RadioButtonGroup = RadioButton(master).group;
        if (radioGroup)
        {
           var index:int = 0;
           var n:int = radioGroup.numRadioButtons;
           for (var i:int = 0; i < n; i++)
           {
               var radioButton:RadioButton = radioGroup.getRadioButtonAt(i);
               if (radioButton === master)
               {
                   index = i + 1;
                   break;
               }
           }
           name += " " + index + " of " + n;
        }
        return name;
    }
}

}
