/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.discover;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.Executor;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import org.apache.bookkeeper.client.BKException;
import org.apache.bookkeeper.common.concurrent.FutureUtils;
import org.apache.bookkeeper.common.util.SafeRunnable;
import org.apache.bookkeeper.discover.BookieServiceInfo;
import org.apache.bookkeeper.discover.BookieServiceInfoUtils;
import org.apache.bookkeeper.discover.RegistrationClient;
import org.apache.bookkeeper.net.BookieSocketAddress;
import org.apache.bookkeeper.proto.DataFormats;
import org.apache.bookkeeper.versioning.LongVersion;
import org.apache.bookkeeper.versioning.Version;
import org.apache.bookkeeper.versioning.Versioned;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZKRegistrationClient
implements RegistrationClient {
    private static final Logger log = LoggerFactory.getLogger(ZKRegistrationClient.class);
    static final int ZK_CONNECT_BACKOFF_MS = 200;
    private final ZooKeeper zk;
    private final ScheduledExecutorService scheduler;
    private WatchTask watchWritableBookiesTask = null;
    private WatchTask watchReadOnlyBookiesTask = null;
    private final String bookieRegistrationPath;
    private final String bookieAllRegistrationPath;
    private final String bookieReadonlyRegistrationPath;

    public ZKRegistrationClient(ZooKeeper zk, String ledgersRootPath, ScheduledExecutorService scheduler) {
        this.zk = zk;
        this.scheduler = scheduler;
        this.bookieRegistrationPath = ledgersRootPath + "/" + "available";
        this.bookieAllRegistrationPath = ledgersRootPath + "/" + "cookies";
        this.bookieReadonlyRegistrationPath = this.bookieRegistrationPath + "/" + "readonly";
    }

    @Override
    public void close() {
    }

    public ZooKeeper getZk() {
        return this.zk;
    }

    @Override
    public CompletableFuture<Versioned<Set<BookieSocketAddress>>> getWritableBookies() {
        return this.getChildren(this.bookieRegistrationPath, null);
    }

    @Override
    public CompletableFuture<Versioned<Set<BookieSocketAddress>>> getAllBookies() {
        return this.getChildren(this.bookieAllRegistrationPath, null);
    }

    @Override
    public CompletableFuture<Versioned<Set<BookieSocketAddress>>> getReadOnlyBookies() {
        return this.getChildren(this.bookieReadonlyRegistrationPath, null);
    }

    @Override
    public CompletableFuture<Versioned<BookieServiceInfo>> getBookieServiceInfo(String bookieId) {
        String pathAsWritable = this.bookieRegistrationPath + "/" + bookieId;
        String pathAsReadonly = this.bookieReadonlyRegistrationPath + "/" + bookieId;
        CompletableFuture<Versioned<BookieServiceInfo>> promise = new CompletableFuture<Versioned<BookieServiceInfo>>();
        this.zk.getData(pathAsWritable, false, (rc, path, o, bytes, stat) -> {
            if (KeeperException.Code.OK.intValue() == rc) {
                try {
                    BookieServiceInfo bookieServiceInfo = ZKRegistrationClient.deserializeBookieServiceInfo(bookieId, bytes);
                    promise.complete(new Versioned<BookieServiceInfo>(bookieServiceInfo, new LongVersion(stat.getCversion())));
                }
                catch (IOException ex) {
                    promise.completeExceptionally(KeeperException.create((KeeperException.Code)KeeperException.Code.get((int)rc), (String)path));
                    return;
                }
            } else if (KeeperException.Code.NONODE.intValue() == rc) {
                this.zk.getData(pathAsReadonly, false, (rc2, path2, o2, bytes2, stat2) -> {
                    if (KeeperException.Code.OK.intValue() == rc2) {
                        try {
                            BookieServiceInfo bookieServiceInfo = ZKRegistrationClient.deserializeBookieServiceInfo(bookieId, bytes2);
                            promise.complete(new Versioned<BookieServiceInfo>(bookieServiceInfo, new LongVersion(stat2.getCversion())));
                        }
                        catch (IOException ex) {
                            promise.completeExceptionally(KeeperException.create((KeeperException.Code)KeeperException.Code.get((int)rc2), (String)path2));
                            return;
                        }
                    } else {
                        promise.completeExceptionally(BKException.create(-3));
                    }
                }, null);
            } else {
                promise.completeExceptionally(KeeperException.create((KeeperException.Code)KeeperException.Code.get((int)rc), (String)path));
            }
        }, null);
        return promise;
    }

    @VisibleForTesting
    static BookieServiceInfo deserializeBookieServiceInfo(String bookieId, byte[] bookieServiceInfo) throws IOException {
        if (bookieServiceInfo == null || bookieServiceInfo.length == 0) {
            return BookieServiceInfoUtils.buildLegacyBookieServiceInfo(bookieId);
        }
        DataFormats.BookieServiceInfoFormat builder = DataFormats.BookieServiceInfoFormat.parseFrom((byte[])bookieServiceInfo);
        BookieServiceInfo bsi = new BookieServiceInfo();
        List<BookieServiceInfo.Endpoint> endpoints = builder.getEndpointsList().stream().map(e -> {
            BookieServiceInfo.Endpoint endpoint = new BookieServiceInfo.Endpoint();
            endpoint.setId(e.getId());
            endpoint.setPort(e.getPort());
            endpoint.setHost(e.getHost());
            endpoint.setProtocol(e.getProtocol());
            endpoint.setAuth((List<String>)e.getAuthList());
            endpoint.setExtensions((List<String>)e.getExtensionsList());
            return endpoint;
        }).collect(Collectors.toList());
        bsi.setEndpoints(endpoints);
        bsi.setProperties(builder.getPropertiesMap());
        return bsi;
    }

    private CompletableFuture<Versioned<Set<BookieSocketAddress>>> getChildren(String regPath, Watcher watcher) {
        CompletableFuture future = FutureUtils.createFuture();
        this.zk.getChildren(regPath, watcher, (rc, path, ctx, children, stat) -> {
            if (0 != rc) {
                BKException.ZKException zke = new BKException.ZKException(KeeperException.create((KeeperException.Code)KeeperException.Code.get((int)rc), (String)path));
                future.completeExceptionally(zke.fillInStackTrace());
                return;
            }
            LongVersion version = new LongVersion(stat.getCversion());
            HashSet<BookieSocketAddress> bookies = ZKRegistrationClient.convertToBookieAddresses(children);
            future.complete(new Versioned<HashSet<BookieSocketAddress>>(bookies, version));
        }, null);
        return future;
    }

    @Override
    public synchronized CompletableFuture<Void> watchWritableBookies(RegistrationClient.RegistrationListener listener) {
        CompletionStage f;
        if (null == this.watchWritableBookiesTask) {
            f = new CompletableFuture();
            this.watchWritableBookiesTask = new WatchTask(this.bookieRegistrationPath, (CompletableFuture<Void>)f);
            f = f.whenComplete((value, cause) -> {
                if (null != cause) {
                    this.unwatchWritableBookies(listener);
                }
            });
        } else {
            f = this.watchWritableBookiesTask.firstRunFuture;
        }
        this.watchWritableBookiesTask.addListener(listener);
        if (this.watchWritableBookiesTask.getNumListeners() == 1) {
            this.watchWritableBookiesTask.watch();
        }
        return f;
    }

    @Override
    public synchronized void unwatchWritableBookies(RegistrationClient.RegistrationListener listener) {
        if (null == this.watchWritableBookiesTask) {
            return;
        }
        this.watchWritableBookiesTask.removeListener(listener);
        if (this.watchWritableBookiesTask.getNumListeners() == 0) {
            this.watchWritableBookiesTask.close();
            this.watchWritableBookiesTask = null;
        }
    }

    @Override
    public synchronized CompletableFuture<Void> watchReadOnlyBookies(RegistrationClient.RegistrationListener listener) {
        CompletionStage f;
        if (null == this.watchReadOnlyBookiesTask) {
            f = new CompletableFuture();
            this.watchReadOnlyBookiesTask = new WatchTask(this.bookieReadonlyRegistrationPath, (CompletableFuture<Void>)f);
            f = f.whenComplete((value, cause) -> {
                if (null != cause) {
                    this.unwatchReadOnlyBookies(listener);
                }
            });
        } else {
            f = this.watchReadOnlyBookiesTask.firstRunFuture;
        }
        this.watchReadOnlyBookiesTask.addListener(listener);
        if (this.watchReadOnlyBookiesTask.getNumListeners() == 1) {
            this.watchReadOnlyBookiesTask.watch();
        }
        return f;
    }

    @Override
    public synchronized void unwatchReadOnlyBookies(RegistrationClient.RegistrationListener listener) {
        if (null == this.watchReadOnlyBookiesTask) {
            return;
        }
        this.watchReadOnlyBookiesTask.removeListener(listener);
        if (this.watchReadOnlyBookiesTask.getNumListeners() == 0) {
            this.watchReadOnlyBookiesTask.close();
            this.watchReadOnlyBookiesTask = null;
        }
    }

    private static HashSet<BookieSocketAddress> convertToBookieAddresses(List<String> children) {
        HashSet newBookieAddrs = Sets.newHashSet();
        for (String bookieAddrString : children) {
            BookieSocketAddress bookieAddr;
            if ("readonly".equals(bookieAddrString)) continue;
            try {
                bookieAddr = new BookieSocketAddress(bookieAddrString);
            }
            catch (IOException e) {
                log.error("Could not parse bookie address: " + bookieAddrString + ", ignoring this bookie");
                continue;
            }
            newBookieAddrs.add(bookieAddr);
        }
        return newBookieAddrs;
    }

    WatchTask getWatchWritableBookiesTask() {
        return this.watchWritableBookiesTask;
    }

    WatchTask getWatchReadOnlyBookiesTask() {
        return this.watchReadOnlyBookiesTask;
    }

    class WatchTask
    implements SafeRunnable,
    Watcher,
    BiConsumer<Versioned<Set<BookieSocketAddress>>, Throwable>,
    AutoCloseable {
        private final String regPath;
        private final Set<RegistrationClient.RegistrationListener> listeners;
        private volatile boolean closed = false;
        private Set<BookieSocketAddress> bookies = null;
        private Version version = Version.NEW;
        private final CompletableFuture<Void> firstRunFuture;

        WatchTask(String regPath, CompletableFuture<Void> firstRunFuture) {
            this.regPath = regPath;
            this.listeners = new CopyOnWriteArraySet<RegistrationClient.RegistrationListener>();
            this.firstRunFuture = firstRunFuture;
        }

        public int getNumListeners() {
            return this.listeners.size();
        }

        public boolean addListener(RegistrationClient.RegistrationListener listener) {
            if (this.listeners.add(listener) && null != this.bookies) {
                ZKRegistrationClient.this.scheduler.execute(() -> listener.onBookiesChanged(new Versioned<Set<BookieSocketAddress>>(this.bookies, this.version)));
            }
            return true;
        }

        public boolean removeListener(RegistrationClient.RegistrationListener listener) {
            return this.listeners.remove(listener);
        }

        void watch() {
            this.scheduleWatchTask(0L);
        }

        private void scheduleWatchTask(long delayMs) {
            try {
                ZKRegistrationClient.this.scheduler.schedule((Runnable)((Object)this), delayMs, TimeUnit.MILLISECONDS);
            }
            catch (RejectedExecutionException ree) {
                log.warn("Failed to schedule watch bookies task", (Throwable)ree);
            }
        }

        public void safeRun() {
            if (this.isClosed()) {
                return;
            }
            ZKRegistrationClient.this.getChildren(this.regPath, this).whenCompleteAsync((BiConsumer)this, (Executor)ZKRegistrationClient.this.scheduler);
        }

        @Override
        public void accept(Versioned<Set<BookieSocketAddress>> bookieSet, Throwable throwable) {
            if (throwable != null) {
                if (this.firstRunFuture.isDone()) {
                    this.scheduleWatchTask(200L);
                } else {
                    this.firstRunFuture.completeExceptionally(throwable);
                }
                return;
            }
            if (this.version.compare(bookieSet.getVersion()) == Version.Occurred.BEFORE) {
                this.version = bookieSet.getVersion();
                this.bookies = bookieSet.getValue();
                for (RegistrationClient.RegistrationListener listener : this.listeners) {
                    listener.onBookiesChanged(bookieSet);
                }
            }
            FutureUtils.complete(this.firstRunFuture, null);
        }

        public void process(WatchedEvent event) {
            if (Watcher.Event.EventType.None == event.getType()) {
                if (Watcher.Event.KeeperState.Expired == event.getState()) {
                    this.scheduleWatchTask(200L);
                }
                return;
            }
            this.scheduleWatchTask(0L);
        }

        boolean isClosed() {
            return this.closed;
        }

        @Override
        public void close() {
            this.closed = true;
        }
    }
}

