#!/bin/sh

# #!/bin/dash
# #!/bin/sh
# #!/bin/bash
# #!/bin/ksh
# #!/bin/zsh

# Audio Function Generator.
# AFG.sh
# Author:- (C)2023 Barry Walker, G0LCU. Issued as GPLv3.

# For Linux flavours, APPLE OSX 10.12.x and higher, *NIX, AMIGA A1200(HD)
# and ADE combination, and possibly, CygWin with its builtin 'paplay'.
# There may be a chance that this will work inside WSL with a Linux install and a command line audio player.

# It is fully POSIX compliant and designed to run under 'dash'.
# External utilities used must use POSIX only options, etc...

VERSION='0.95.00'

# Unusual functions needed.
# Reset the terminal and clear the screen. 
clrscn()
{
	printf "%b" "\033c\033[0m\033[2J\033[H";
}

# A timeout function for OSes that don't have it, like APPLE OSX for example.
# time_out 4 'paplay [/full/path/to/]WAVEFILE.wav'
time_out()
{
	time=${1}
	cmd=${2}
	eval "${cmd} &"; sleep "${time}"; kill $! > /dev/null 2>&1;
}

# Clear the terminal window...
clrscn

PLATFORM=$( uname )
# AMIGA A1200 OS3.0x, using ADE the *NIX environment.
if [ "${PLATFORM}" = "AmigaOS" ]
then
	HOME="/ADE/home/"
fi

# Create everything that is needed for all other platforms than the AMIGA!
# This part is fully ignored for the AMIGA as it has already been done in the AMINET archive.
# Firstly create the directory tree inside the "${HOME}" directory.
if [ ! -d "${HOME}/AFG/64/" ]
then
	echo "Creating directory tree!"
	mkdir ${HOME}/AFG
	mkdir ${HOME}/AFG/64
	mkdir ${HOME}/AFG/16
	mkdir ${HOME}/AFG/4
	echo "Directroy tree, DONE!"
	# Now create the required files. All files are exactly 524288 bytes in size.
	# IMPORTANT NOTE:- ALL waveforms use the ASCII characters from 'SPACE' to '~' ONLY!
	# This means you can edit a waveform of your own in a basic text editor.
	# Read the manual for more information...
	echo "Creating required files!"
	SINE64='OTX]aeimpsvxz|}~~~}|zxvspmiea]XTOJFA=951.+(&$"!   !"$&(+.159=AFJ'
	SINE16='Oapz~zpaO=.$ $.='
	SINE4='O~O '
	SQUARE64='~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                                '
	SQUARE16='~~~~~~~~        '
	SQUARE4='~~  '
	PULSE64='~~~~                                                            '
	PULSE16='~~              '
	PULSE4='~   '
	PULSE_INVERSE64='    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~'
	PULSE_INVERSE16='  ~~~~~~~~~~~~~~'
	PULSE_INVERSE4=' ~~~'
	TRIANGLE64='ORUX[^adfilorux{~{xurolifda^[XUROLIFC@=:852/,)&# #&),/258:=@CFIL'
	TRIANGLE16='O[fr~rf[OC8, ,8C'
	TRIANGLE4=' O~O'
	SAWTOOTH64=' !#$&'"'"')*,-/0235689;<>?ABDEGHJKMNPQRTUWXZ[]^`acdfgijlmoprsuvxy{|~'
	SAWTOOTH16=' &,39?FLRX^ekqx~'
	SAWTOOTH4=' ?_~'
	SAWTOOTH_INVERSE64='~|{yxvusrpomljigfdca`^][ZXWUTRQPNMKJHGEDBA?><;9865320/-,*)'"'"'&$#! '
	SAWTOOTH_INVERSE16='~xqke^XRLF?93,& '
	SAWTOOTH_INVERSE4='~_? '
	for n in $( seq 1 1 13 )
	do
		SINE64="${SINE64}${SINE64}"
		SQUARE64="${SQUARE64}${SQUARE64}"
		PULSE64="${PULSE64}${PULSE64}"
		PULSE_INVERSE64="${PULSE_INVERSE64}${PULSE_INVERSE64}"
		TRIANGLE64="${TRIANGLE64}${TRIANGLE64}"
		SAWTOOTH64="${SAWTOOTH64}${SAWTOOTH64}"
		SAWTOOTH_INVERSE64="${SAWTOOTH_INVERSE64}${SAWTOOTH_INVERSE64}"
	done
	for n in $( seq 1 1 15 )
	do
		SINE16="${SINE16}${SINE16}"
		SQUARE16="${SQUARE16}${SQUARE16}"
		PULSE16="${PULSE16}${PULSE16}"
		PULSE_INVERSE16="${PULSE_INVERSE16}${PULSE_INVERSE16}"
		TRIANGLE16="${TRIANGLE16}${TRIANGLE16}"
		SAWTOOTH16="${SAWTOOTH16}${SAWTOOTH16}"
		SAWTOOTH_INVERSE16="${SAWTOOTH_INVERSE16}${SAWTOOTH_INVERSE16}"
	done
	for n in $( seq 1 1 17 )
	do
		SINE4="${SINE4}${SINE4}"
		SQUARE4="${SQUARE4}${SQUARE4}"
		PULSE4="${PULSE4}${PULSE4}"
		PULSE_INVERSE4="${PULSE_INVERSE4}${PULSE_INVERSE4}"
		TRIANGLE4="${TRIANGLE4}${TRIANGLE4}"
		SAWTOOTH4="${SAWTOOTH4}${SAWTOOTH4}"
		SAWTOOTH_INVERSE4="${SAWTOOTH_INVERSE4}${SAWTOOTH_INVERSE4}"
	done
	printf "%b" "${SINE64}" > ${HOME}/AFG/64/SINE
	printf "%b" "${SINE16}" > ${HOME}/AFG/16/SINE
	printf "%b" "${SINE4}" > ${HOME}/AFG/4/SINE
	printf "%b" "${SQUARE64}" > ${HOME}/AFG/64/SQUARE
	printf "%b" "${SQUARE16}" > ${HOME}/AFG/16/SQUARE
	printf "%b" "${SQUARE4}" > ${HOME}/AFG/4/SQUARE
	printf "%b" "${PULSE64}" > ${HOME}/AFG/64/PULSE
	printf "%b" "${PULSE16}" > ${HOME}/AFG/16/PULSE
	printf "%b" "${PULSE4}" > ${HOME}/AFG/4/PULSE
	printf "%b" "${PULSE_INVERSE64}" > ${HOME}/AFG/64/IPULSE
	printf "%b" "${PULSE_INVERSE16}" > ${HOME}/AFG/16/IPULSE
	printf "%b" "${PULSE_INVERSE4}" > ${HOME}/AFG/4/IPULSE
	printf "%b" "${TRIANGLE64}" > ${HOME}/AFG/64/TRIANGLE
	printf "%b" "${TRIANGLE16}" > ${HOME}/AFG/16/TRIANGLE
	printf "%b" "${TRIANGLE4}" > ${HOME}/AFG/4/TRIANGLE
	printf "%b" "${SAWTOOTH64}" > ${HOME}/AFG/64/SAWTOOTH
	printf "%b" "${SAWTOOTH16}" > ${HOME}/AFG/16/SAWTOOTH
	printf "%b" "${SAWTOOTH4}" > ${HOME}/AFG/4/SAWTOOTH
	printf "%b" "${SAWTOOTH_INVERSE64}" > ${HOME}/AFG/64/ISAWTOOTH
	printf "%b" "${SAWTOOTH_INVERSE16}" > ${HOME}/AFG/16/ISAWTOOTH
	printf "%b" "${SAWTOOTH_INVERSE4}" > ${HOME}/AFG/4/ISAWTOOTH
	echo "Required files, DONE!"
	echo "Finally create the Manual.txt and other parts..."
cat << 'MANUAL' > ${HOME}/AFG/Manual.txt

################ Audio Function Generator, Version X.XX.XX. #################
################### Author: '(C)_2023,_B.Walker,_G0LCU.' ####################

 Preface:-
 ---------

        Fully POSIX Compliant, Dash Shell, Audio Function Generator.
        ------------------------------------------------------------

 (Apologies for any typos, etc...)

 Well, we can kiss goodbye to the UNIX ethos that everything is a file!
 We no longer have /dev/adsp, /dev/audio, and /dev/dsp, so, this is an
 alternative method of generating an audio signal for basic audio testing
 using the default, standard, _builtin_, OS command line audio players.

 This code snippet generates 7 different waveforms, SINE, SQUARE, TRIANGLE,
 PULSE, INVERSE PULSE, SAWTOOTH, and INVERSE SAWTOOTH.
 There is much to be desired in, frequency accuracy, waveform _purity_, and
 duration for each burst as it is not possible to have a continuous signal
 using 'aplay' for ALSA, 'paplay' for PulseAudio, 'afplay' for OSX 10.x.x,
 and 'Play16' for the AMIGA A1200(HD) and ADE combination.

 The ABSOLUTE limits are, (dependent upon the sound system present):-
 1) 2000 sps minimum, 64000 sps maximum, (56000 sps for the AMIGA A1200(HD)).
 2) 32Hz minimum, 16000Hz maximum, (14000Hz maximum for the AMIGA A1200(HD)).
 3) 1 second to 8 seconds duration for each burst.
 4) The longest possible. Completely dependent upon the frequency decided
    upon with 9 seconds selected, but NEVER less than 8 seconds.

 PRACTICAL, USEFUL, reasonably accurate limits are:-
 1) 8000 sps minimum, 28000 sps maximum, corresponding to AMIGA A1200(HD)
    specification[s].
 2) 125Hz minimum, 7000Hz maximum.
 3) Pseudo-variable duration burst times.

 This manual is centred around the AMIGA A1200(HD) and ADE combination so
 all you *NIX, GNU/Linux, CygWin, Android, and perhaps even WSL users will
 have to _translate_ this manual to suit your platform of choice.

 The whole script was written for a 'dash' shell but has a shebang of
 '#!/bin/sh' for generality including for the AMIGA A1200(HD) and ADE.
 It is FULLY POSIX compliant and a little naive in places, mainly to suit
 the AMIGA, my platform of choice.

 Although readily available, audio players sox, mplayer, music123, vlc,
 audioplay, ffplay and other command line AF players are not catered for as
 they are not needed for this project, just the default players for each
 platform. That does not mean to say that you can't use them, just that they
 are an unneccessary addition to have to install.

        Do's and Don'ts:-
        -----------------

 1) There is little to no error checking in this script so therefore.......
 2) All numerical entries MUST be integers only. Decimal point numbers WILL
    throw errors and may even crash out of the program.
 3) Do not use Ctrl-C to stop as it is being trapped. You MUST wait until a
    burst has finished and follow the bold COLOURED, (YELLOW for other
    platforms), prompt when it appears.
 4) If you press Ctrl-C at any time it WILL mess up the text on screen. This
    WILL clear on the next MAJOR loop but will set everything to the default
    settings, SINE, 1000Hz, 1 Second Burst inside the continuous burst loop.
 5) Certain visual things require terminal escape codes! The terminal size
    MUST be 80 Columns x 24 Lines for all platforms, except, the AMIGA shell
    MUST be opened to the maximum window size allowed, and be able to
    support the terminal escape codes used.
 6) If your OS has both 'aplay' AND 'paplay' you will get 2 bursts in close
    succession per loop. This is NOT a bug, as the general case is one OR
    the other could be used depending upon the platform in use. I had to
    install the PulseAudio 'paplay' so as to develop this script.
 7) Aliasing is a possibility towards the extremes of the frequencies
    entered. This is something you will have to put up with as the
    limitations are mentioned earlier.
 8) Do, take notice of the _on-screen_ prompts as they are to be as helpful
    as possible. Pressing the ENTER/RETURN key will take you straight to the
    default, SINE, 1000Hz, 1 Second Burst for practicing and learning how to
    use this audio signal generator.
 9) And FINALLY, you DO have permission to remove any crud that will not be
    used on your OS/system, but ONLY for your personal usage.
    Otherwise GPL-V3 applies...
                https://www.gnu.org/licenses/gpl-3.0.en.html

#############################################################################

 Requirements:-
 --------------

        Architectures: FS-UAE, OSX 10.15.7, GNU/Linux Mint 20.3:-
        ---------------------------------------------------------

        MINIMUM Requirements Are:-
        --------------------------

  1) AMIGA A1200(HD) environment:-
     FS-UAE on both OSX 10.15.7 and Linux Mint 20.3.
     AMIGA OS_3.0.x install or better with 4MB Fast RAM.
     Emulated 68020 CPU, (OS 3.1.x, and much better HW emulation will be a
     massive advantage).

     Play16, two versions, the lower one includes a 68000 version,
     and both also supports AHI and others if need be but Paula8 is the
     default and the one actually used!
       http://aminet.net/package/mus/play/Play16
       http://aminet.net/package/mus/play/Play16_v1.9
       http://aminet.net/package/dev/c/AsyncIO <- This is already inside
     the archive of the Play16_v1.9 install.
     ADE the *NIX emulation for the AMIGA A1200(HD).
       http://aminet.net/package/dev/gcc/ADE
     Library file ixnet.library version 48.0
     Library file ixemul.library version 48.0
     Both files are here if you have NOT got them:
       http://aminet.net/package/util/libs/ixemul-48.0

  2) Other platforms:-
     Apple OSX 10.12.x and later using the default shell.
        (NOTE: Used to be 'bash', but is now 'zsh'.)
     GNU/Linux Mint 20.3 and later using the default shell.
     No other dependencies required.

  3) !!! IMPORTANT !!!
     This script has NOT been tested on real AMIGA A1200(HD) hardware,
     WinUAE, Windows Subsystem for Linux and Linux install, Android, or
     CygWin but they might just work fine.
     As far as I know, Android and CygWin both use the PulseAudio 'paplay'
     audio player, and that CygWin still has '/dev/dsp' if not.
     You therefore try it out ENTIRELY at your own risk.

#############################################################################

 Installation and setting up:-
 -----------------------------

  1) From now on all AMIGA references refer to an AMIGA A1200(HD).
     And, <CR> is the RETURN/ENTER key.

  2) The code is FULL of comments some of which are duplicated here,
     BUT some of which are not mentioned at all so read that file too.

  3) There is almost NO error checking and correction at all in the
     code at present but IF you adhere to the setting up below you
     probably will not need any error checking.

  4) You have permission to remove any _crud_ you don't need for your
     own personal use; that is, comments and platforms you will not
     use, otherwise licence GPL-V3 fully applies, see above for the URL.

-----------------------------------------------------------------------------

  1) The minimum requirements ARE needed for the AMIGA.

     !!! IMPORTANT:- FOR OTHER PLATFORMS. !!!
     For other platforms just use the AFG.sh file ONLY, nothing else is
     required and place it in the '${HOME}' drawer|folder|directory.
     Start up a terminal, go to the '${HOME}' drawer, change user rights
     and run ./AFG.sh in the usual way. Everything needed, will be created
     on the very first run! It will exit immediately and you will have to
     restart the script from inside the ${HOME}/AFG/ drawer.
     !!! END OF IMPORTANT. !!!

     !!! Back to the AMIGA. !!!
  2) Install the Play16 package and its dependencies wherever you please,
     and once installed place the executable Play16 inside the SYS:C drawer
     or the C: Volume.
  3) Install the ADE *NIX package and its dependencies, and go to the
     'DRIVE:ADE/home/' drawer. Place this archive into this drawer and
     unarchive it. Everything is in the archive and now ready to go.
  4) You should now have the drawer 'DRIVE:ADE/home/AFG' ready to go!
  5) Start up a default standard AMIGA shell and open it up to its maximum
     window size possible.
  6) Start ADE itself from the DRIVE:ADE drawer as 'Execute ADE-Startup'.
  7) In case it does NOT run straight into 'ksh'[88] or 'sh', start 'sh'
     inside the AMIGA shell.
  8) You should now be in the '/DRIVE/ADE' drawer in *NIX emulation mode.
  9) Change the drawer to 'cd home/AFG'<CR>
 10) Do the command 'chmod 755 AFG.sh'<CR>
 11) Start AFG.sh as './AFG.sh'<CR>
 12) You are now starting to use the Audio Function Generator.

-----------------------------------------------------------------------------

 Using the Audio Function Generator:-
 ------------------------------------

  1) FOLLOW THE IN-SCREEN PROMPTS!
  2) The first screen is a very simple circuit diagram of a stereo jack plug
     output source for this UNCALIBRATED level audio signal generator.
     This also gives the version and other info too.
     Just press <CR> to continue...

  3) The next screen starts off like this, 750Hz is being entered:-
        Follow the on-screen prompts...
        -------------------------------

        Enter the frequency in Hz, 100 to 10000 or ENTER for default:- 750

  4) Next comes the waveforms, ENTER defaults to SINE:-
        Follow the on-screen prompts...
        -------------------------------

        Enter the frequency in Hz, 100 to 10000 or ENTER for default:- 750
        Waveforms available...
          's' for a SINE wave.
          'S' for a SQUARE wave.
          'p' for a PULSE wave.
          'P' for an INVERSE PULSE wave.
          't' for a TRIANGLE wave.
          'w' for a SAWTOOTH wave.
          'W' for an INVERSE SAWTOOTH wave.
        Enter the letter for your waveform or ENTER for default:- 

  5) Next comes the burst time, 9 will be used here for the maximum time:-
        Follow the on-screen prompts...
        -------------------------------

        Enter the frequency in Hz, 100 to 10000 or ENTER for default:- 750
        Waveforms available...
          's' for a SINE wave.
          'S' for a SQUARE wave.
          'p' for a PULSE wave.
          'P' for an INVERSE PULSE wave.
          't' for a TRIANGLE wave.
          'w' for a SAWTOOTH wave.
          'W' for an INVERSE SAWTOOTH wave.
        Enter the letter for your waveform or ENTER for default:- 
        Enter period to play in seconds, 1 to 8 or ENTER for default:- 9
        Playing a 750Hz SINE waveform at 48000 sps.

        You have chosen a value of 9 seconds!
        Each loop duration will now last for 10 seconds.

  6) It now plays a 750Hz SINEWAVE for 10 seconds with a line in bold BLACK,
     (RED for other platforms), of what is playing, AND, 2 extra lines of
     information in normal colours, this carries on and on until:-

  7) A bold COLOURED prompt appears, (YELLOW for other platforms), asking
     for 'Ctrl-C' IF you want it. The colour depends entirely on your Classic
     Workbench colours. You have 2 seconds in which to acknowledge it:-
        Follow the on-screen prompts...
        -------------------------------

        Enter the frequency in Hz, 100 to 10000 or ENTER for default:- 750
        Waveforms available...
          's' for a SINE wave.
          'S' for a SQUARE wave.
          'p' for a PULSE wave.
          'P' for an INVERSE PULSE wave.
          't' for a TRIANGLE wave.
          'w' for a SAWTOOTH wave.
          'W' for an INVERSE SAWTOOTH wave.
        Enter the letter for your waveform or ENTER for default:- 
        Enter period to play in seconds, 1 to 8 or ENTER for default:- 9
        Playing a 750Hz SINE waveform at 48000 sps.

        Press Ctrl-C momentarily to EXIT the loop:-

  8) When you press 'Ctrl-C' a bold WHITE, (GREEN for other platforms), line
     acknowledging your request appears along with the final line in normal
     colours awaiting your final input request:-
        Follow the on-screen prompts...
        -------------------------------

        Enter the frequency in Hz, 100 to 10000 or ENTER for default:- 750
        Waveforms available...
          's' for a SINE wave.
          'S' for a SQUARE wave.
          'p' for a PULSE wave.
          'P' for an INVERSE PULSE wave.
          't' for a TRIANGLE wave.
          'w' for a SAWTOOTH wave.
          'W' for an INVERSE SAWTOOTH wave.
        Enter the letter for your waveform or ENTER for default:- 
        Enter period to play in seconds, 1 to 8 or ENTER for default:- 9
        Playing a 750Hz SINE waveform at 48000 sps.

        Pressing Ctrl-C has successfully exited the loop!      
                                                               
        Enter 'Q' to QUIT, 'M' for the Manual or ENTER only to continue:- 

  9) Pressing Q or q quits the program with a return code of 0:-
        Follow the on-screen prompts...
        -------------------------------

        Enter the frequency in Hz, 100 to 10000 or ENTER for default:- 750
        Waveforms available...
          's' for a SINE wave.
          'S' for a SQUARE wave.
          'p' for a PULSE wave.
          'P' for an INVERSE PULSE wave.
          't' for a TRIANGLE wave.
          'w' for a SAWTOOTH wave.
          'W' for an INVERSE SAWTOOTH wave.
        Enter the letter for your waveform or ENTER for default:- 
        Enter period to play in seconds, 1 to 8 or ENTER for default:- 9
        Playing a 750Hz SINE waveform at 48000 sps.

        Pressing Ctrl-C has successfully exited the loop!      
                                                               
        Enter 'Q' to QUIT, 'M' for the Manual or ENTER only to continue:- q
        Quitting with a return code of 0!

 10) Pressing M or m opens up this Manual for reference.
 11) Pressing <CR> reruns the program for another set of parameters WITHOUT
     the first circuit screen.
 12) And FINALLY there is more information inside 'AFG.sh' script itself via
     all of the comments, so read that also.
 13) There is no physical output level control on a real Classic AMIGA A1200
     so you will have to build an attenuator of your own to alter the output
     level from the Phono/RCA sockets.
 14) For ALL other platforms, including AMIGA emulation, use the computer's
     volume control to adjust the output.

#############################################################################
#
# Simple circuit diagram for obtaining the audio output[s] for FS-UAE.
# --------------------------------------------------------------------
#
# It is just as easy to replace the 4 pole 3.5mm Jack Plug with a 3.5mm
# Stereo Jack Plug as Ring 2 is already part of _GND_.
#
# Tip ----->  O  <----------------o--------------0 <--- Output 1.
# Ring 1 -->  H  <----------o-----)--------------0 <--- Output 2.
# Ring 2 -->  H  <----o     |     |
# _Gnd_ --->  H  <----o     |     |     +--------0 <--- Pseudo-Ground.
#           +===+     |     \     \     |
#           |   |     |     /     /     |
#        P1 |   |     |     \     \     |
#           |   |     |  R1 /  R2 /     |
#            \ /      |     \     \     |
#             H       |     /     /     |
#          (CABLE)    |     |     |     |
#                     o-----o-----o-----+ Connect Ring 2 and _GND_ together.
# Pseudo-Ground. -> __|__
#            _GND_  /////
#
# Parts List:-
# P1 ......... 3.5mm, 4 pole jack plug, OR, 3.5mm standard jack plug.
# R1, R2 ..... 33R, 1/8W, 5% tolerance resistor.
#
#############################################################################
#
# Ultra simple attenuator circuit for Classic AMIGAs.
# ---------------------------------------------------
#
#  +-----------o-----+  
#  | RCA1      |     |  
# (O)-----+    |     \  
#    RCA2 |    |     / RV1
# (O)-----o    |     \<---------------------O
#  |      |    |     /               Variable Output.
#  +------)----+     \                      O
#         |          /                      |
#         |          |                      |
#         o----------o----------------------+
#       __|__ <- Pseudo-Ground.
#       ///// _GND_
#
# Parts List:-
# RCA1, RCA2 ......... Phono/RCA plugs.
# RV1 ................ 10K potentiometer.
#
#############################################################################
#############################################################################

The directories and files generated from the script for NON-AMIGA machines:-
----------------------------------------------------------------------------

amiga@AMIGA:~$ cd AFG
amiga@AMIGA:~/AFG$ ls -lR
.:
total 580
drwxrwxr-x 2 amiga amiga   4096 Jul 15 19:33 16
drwxrwxr-x 2 amiga amiga   4096 Jul 15 19:33 4
drwxrwxr-x 2 amiga amiga   4096 Jul 15 19:33 64
-rwxr-xr-x 1 amiga amiga  30297 Jul 15 19:33 AFG.sh
-rw-rw-r-- 1 amiga amiga  16572 Jul 15 19:33 Manual.txt
-rw-rw-r-- 1 amiga amiga 524332 Jul 16 12:47 WAVEFORM.wav

./16:
total 3584
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 IPULSE
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 ISAWTOOTH
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 PULSE
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 SAWTOOTH
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 SINE
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 SQUARE
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 TRIANGLE

./4:
total 3584
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 IPULSE
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 ISAWTOOTH
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 PULSE
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 SAWTOOTH
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 SINE
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 SQUARE
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 TRIANGLE

./64:
total 3584
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 IPULSE
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 ISAWTOOTH
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 PULSE
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 SAWTOOTH
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 SINE
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 SQUARE
-rw-rw-r-- 1 amiga amiga 524288 Jul 15 19:33 TRIANGLE
amiga@AMIGA:~/AFG$

#############################################################################

Enjoy...

Bazza, G0LCU.

#############################################################################

MANUAL

	AFG_Folder=$( pwd )
	cp ${AFG_Folder}/AFG.sh ${HOME}/AFG/AFG.sh
	cd ${HOME}/AFG/
	chmod 755 AFG.sh
	echo "Everything is ready..."
	echo "Restart AFG.sh from the '~/AFG/' folder."
	exit 0
fi

# More functions for the main running section.
#
# Set up the wavefile header.
# This header is a single channel MONO, unsigned 8 bit integer, VARIABLE
# sample rate, 44 byte '.wav' header, ONLY.
header()
{
	# Create the filename in the ' ${HOME}/AFG/' directory.
	: > ${HOME}/AFG/WAVEFORM.wav
	# Enter your required frequency, integer values only.
	printf "        Enter the frequency in Hz, 100 to 10000 or ENTER for default:- "
	read -r FREQ
	# Allow only ASCII numeric characters and set to '1000'Hz by default.
	case ${FREQ} in
	[3-9][0-9]|[1-9][0-9][0-9]|[1-9][0-9][0-9][0-9]|1[0-6][0-9][0-9][0-9])
		FREQ=${FREQ};;
	*)
		FREQ=1000;;
	esac
	# Set the absolute limits for ALL cases...
	if [ ${FREQ} -gt 16000 ] || [ ${FREQ} -lt 32 ]
	then
		# The default frequency, 1000Hz.
		FREQ=1000
	fi
	# The AMIGA's internal 'Paula' audio upper frequency limit taken care of too...
	if [ "${PLATFORM}" = "AmigaOS" ] && [ ${FREQ} -gt 14000 ]
	then
		FREQ=1000
	fi
	# Switch to the correct directory, dependent upon the frequency.
	cd ${HOME}/AFG/64/
	BYTES=64
	if [ ${FREQ} -ge 3001 ]
	then
		cd ${HOME}/AFG/4/
		BYTES=4
	fi
	# '16' is the default number of bytes for the default 1000Hz waveform.
	if [ ${FREQ} -ge 751 ] && [ ${FREQ} -le 3000 ]
	then
		cd ${HOME}/AFG/16/
		BYTES=16
	fi
	# Determine the sample rate from the two variables, FREQ and BYTES.
	RATE=$(( FREQ*BYTES ))
	if [ ${RATE} -gt 64000 ] || [ ${RATE} -lt 2000 ]
	then
		# This is the default sample RATE.
		RATE=16000
	fi
	# Create the two bytes in octal form for the WAVE header.
	RATE_HIGH=$( printf "%04o" $(( RATE/256 )) )
	RATE_LOW=$( printf "%04o" $(( RATE%256 )) )
	# Default RATE_HIGH="076", RATE_LOW="0200".
	# IMPORTANT NOTE! ALL octal numbers must start wit a ZERO, 0, in full POSIX mode.
	printf "%b" "\0122\0111\0106\0106\044\000\010\000\0127\0101\0126\0105\0146\0155\0164\040\020\000\000\000\001\000\001\000" >> ${HOME}/AFG/WAVEFORM.wav
	printf "%b" "\\${RATE_LOW}\\${RATE_HIGH}" >> ${HOME}/AFG/WAVEFORM.wav
	printf "%b" "\000\000" >> ${HOME}/AFG/WAVEFORM.wav
	printf "%b" "\\${RATE_LOW}\\${RATE_HIGH}" >> ${HOME}/AFG/WAVEFORM.wav
	printf "%b" "\000\000\001\000\010\000\0144\0141\0164\0141\000\000\010\000" >> ${HOME}/AFG/WAVEFORM.wav
}

# Select the wave shape required.
waveform()
{
	KB="SINE"
	echo "        Waveforms available..."
	echo "          's' for a SINE wave."
	echo "          'S' for a SQUARE wave."
	echo "          'p' for a PULSE wave."
	echo "          'P' for an INVERSE PULSE wave."
	echo "          't' for a TRIANGLE wave."
	echo "          'w' for a SAWTOOTH wave."
	echo "          'W' for an INVERSE SAWTOOTH wave."
	printf "        Enter the letter for your waveform or ENTER for default:- "
	read -r KB
	case ${KB} in
		[sSpPtwW])
			KB=${KB};;
		*)
			KB="s";;
	esac
	# Only allow shown characters and set to the default 'SINE' if <CR> is entered...
	if [ "${KB}" = "" ] || [ "${KB}" = "s" ]
	then
		KB="SINE"
	fi
	if [ "${KB}" = "S" ]
	then
		KB="SQUARE"
	fi
	if [ "${KB}" = "p" ]
	then
		KB="PULSE"
	fi
	if [ "${KB}" = "P" ]
	then
		KB="IPULSE"
	fi
	if [ "${KB}" = "t" ]
	then
		KB="TRIANGLE"
	fi
	if [ "${KB}" = "w" ]
	then
		KB="SAWTOOTH"
	fi
	if [ "${KB}" = "W" ]
	then
		KB="ISAWTOOTH"
	fi
	# Once the header is created the waveform is appended.
	cat < "${KB}" >> ${HOME}/AFG/WAVEFORM.wav
}

# Call all the default *NIX players and use the one[s] available.
player()
{
	Esc='?'
	cd ${HOME}/AFG/
	printf "        Enter period to play in seconds, 1 to 8 or ENTER for default:- "
	read -r PLAYTIME
	# The line below printed in bright red!
	printf "%b" "\033[1;31m        Playing a ${FREQ}Hz ${KB} waveform at ${RATE} sps.\033[0m\n"
	echo ""
	# Set to the default of a 1 second burst if <CR> is entered...
	case ${PLAYTIME} in
		[1-9])
			PLAYTIME=${PLAYTIME};;
		*)
			PLAYTIME=1;;
	esac
	# If 9 is selected then the PLAYTIME becomes the maximum possible for that particular frequency.
	if [ ${PLAYTIME} -eq 9 ]
	then
		echo '        You have chosen a value of 9 seconds!'
		PLAYTIME=$(( 524288/RATE ))
		echo "        Each loop duration will now last for ${PLAYTIME} seconds."
	fi
	if [ -e WAVE.wav ]
	then
		rm WAVE.wav
	fi
	while true
	do

		# AMIGA OS3.0, 'Play16, and ADE combination':
		if [ -e /C/Play16 ]
		then
			if [ ! -e WAVE.wav ]
			then
				TIMER=$(( (RATE*PLAYTIME)/256 ))
				# AHEM! Bending the rules a little here... <wink>
				dd if=WAVEFORM.wav bs=256 count=${TIMER} of=WAVE.wav > /dev/null 2>&1
			fi
			# Using only Paula for the playback...
			/C/Play16 FILTER=AUTO QUIET WAVE.wav > /dev/null 2>&1
		fi

		# Linux ALSA, 'aplay':
		if [ -e /usr/bin/aplay ]
		then
			aplay -d ${PLAYTIME} ${HOME}/AFG/WAVEFORM.wav > /dev/null 2>&1
		fi

		# APPLE OS X 10.12.0 to at least 10.15.7, 'afplay':
		if [ -e /usr/bin/afplay ]
		then
			afplay -t ${PLAYTIME} ${HOME}/AFG/WAVEFORM.wav > /dev/null 2>&1
		fi

		# PulseAudio for various platforms, 'paplay':
		# 'paplay' does NOT have a time limit switch.
		#if [ -e /usr/bin/paplay ]
		#then
		# 	if [ -e /usr/bin/timeout ]
		# 	then
		# 		timeout --kill-after 0 ${PLAYTIME} paplay < ${HOME}/AFG/WAVEFORM.wav > /dev/null 2>&1
		# 	else
		# 		time_out ${PLAYTIME} 'paplay < ${HOME}/AFG/WAVEFORM.wav > /dev/null 2>&1'
		# 	fi
		#fi

		# Download 'music123' from the repository, and commented out unless needed:
		# 'music123' does NOT have a time limit switch.
		#if [ -e /usr/bin/music123 ]
		#then
		#	if [ -e /usr/bin/timeout ]
		#	then
		#		timeout --kill-after 0 ${PLAYTIME} music123 -i -q ${HOME}/AFG/WAVEFORM.wav > /dev/null 2>&1
		#	else
		#		time_out ${PLAYTIME} 'music123 -i -q ${HOME}/AFG/WAVEFORM.wav > /dev/null 2>&1'
		#	fi
		#fi

		# OSS, Open Sound System for various platforms, '/dev/dsp':
		#if [ -e /dev/dsp ]
		#then
		#	if [ -e /usr/bin/timeout ]
		#	then
		#		timeout --kill-after 0 ${PLAYTIME} cat < ${HOME}/AFG/WAVEFORM.wav > /dev/null 2>&1
		#	else
		#		time_out ${PLAYTIME} 'cat < ${HOME}/AFG/WAVEFORM.wav > /dev/null 2>&1'
		#	fi
		#fi

		# General FFMPEG for ?ALL? current platforms, 'ffplay':
		# As of 01-01-2025, for all CURRENT platforms, INSTALL the FFMPEG suite with 'ffpplay' inside the path.
		# Using the 'timeout' command only!
		#timeout --kill-after 0 ${PLAYTIME} ffplay -autoexit -nodisp -loglevel quiet -stats ${HOME}/AFG/WAVEFORM.wav > /dev/null 2>&1

		Esc='?'
		# <trap 'Esc="Ctrl_C"' INT> elsewhere in this code, is the ONLY breakout method that works for the AMIGA A1200(HD).
		# It also works for ALL of the other *NIX platfrorms in this code too!
		# The line below printed in bright yellow!
		printf "%b" "\033[1;33m\033[17;1f        Press Ctrl-C momentarily to EXIT the loop:-            \033[17;53f\033[0m"
		printf "%b" "\033[18;1f                                                               \033[17;1f"
		# Allow the user time to respond with Ctrl-C to exit loop!
		sleep 2 > /dev/null 2>&1
		if [ "${Esc}" = "Ctrl_C" ]
		then
			# The line below printed in bright green!
			printf "%b" "\033[1;32m\033[17;1f        Pressing Ctrl-C has successfully exited the loop!     \033[0m"
			echo ''
			Esc='?'
			break
		fi
		# End of ALL *NIX plaforms and the AMIGA A1200 and ADE compination breakout method.
		printf "%b" "\033[17;1f                                                               \033[17;1f"
	done
}

# The startup page showing a simple circuit for a test cable wiring.
circuit()
{
	clrscn
	echo '################ Audio Function Generator, Version '"${VERSION}"'. #################'
	echo '                          Simple curcuit diagram.'
	echo ' Simple circuit diagram for obtaining the audio output[s].'
	echo ' It is just as easy to replace the 4 pole 3.5mm Jack Plug with a 3.5mm'
	echo ' Stereo Jack Plug as Ring 2 is already part of _GND_.'
	echo ' Tip ----->  O  <----------------o--------------0 <--- Output 1.'
	echo ' Ring 1 -->  H  <----------o-----)--------------0 <--- Output 2.'
	echo ' Ring 2 -->  H  <----o     |     |'
	echo ' _Gnd_ --->  H  <----o     |     |     +--------0 <--- Pseudo-Ground.'
	echo '           .===.     |     \     \     |'
	echo '           |   |     |     /     /     |'
	echo '        P1 |   |     |     \     \     |'
	echo '           |   |     |  R1 /  R2 /     |'
	echo '            \ /      |     \     \     |'
	echo '             H       |     /     /     |'
	echo '          (CABLE)    |     |     |     |'
	echo '                     o-----o-----o-----+ Connect Ring 2 and _GND_ together.'
	echo ' Pseudo-Ground. -> __|__'
	echo '            _GND_  /////'
	echo ' Parts List:-'
	echo ' P1 ......... 3.5mm, 4 pole jack plug, OR, 3.5mm standard jack plug.'
	echo ' R1, R2 ..... 33R, 1/8W, 5% tolerance resistor.'
	echo '########################## (C)2023 B.Walker, G0LCU. #########################'
	printf '                          Press ENTER to continue:- '
	read -r KB
}

# Default values.
# FREQ=1000
# RATE=16000
# BYTES=16
# RATE_LOW="0200"
# RATE_HIGH="076"
# KB="SINE"
# PLAYTIME=1
# Main loop, starts with a simple circuit before the first loop.

: "Running program officially starts here!"
# Copytight page in basic I/O circuit diagram!
circuit

# The ONLY breakout of loop method that works for the AMIGA A1200 and ADE combination!
# All the other tested *NIX flavours work with this method too!
Esc='?'
trap 'Esc="Ctrl_C"' INT

while true
do
	clrscn
	echo "        Follow the on-screen prompts..."
	echo "        -------------------------------"
	echo ""
	header
	waveform
	player
	echo ""
	printf "        Enter 'Q' to QUIT, 'M' for the Manual or ENTER only to continue:- "
	read -r KB
	if [ "${KB}" = "Q" ] || [ "${KB}" = "q" ]
	then
		break
	fi
	if [ "${KB}" = "M" ] || [ "${KB}" = "m" ]
	then
		clrscn
		more ${HOME}/AFG/Manual.txt
	fi
	# NOTE:- These two keys is NOT mentioned in the manual at all.
	if [ "${KB}" = "C" ] || [ "${KB}" = "c" ]
	then
		circuit
	fi
done
echo "        Quitting with a return code of 0!"
echo ""
exit 0

