#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "backend.h"
#include "cfghooks.h"
#include "tree.h"
#include "rtl.h"

#define USE_MOVQ(i)	((unsigned) ((i) + 128) <= 255)

/* Return the cost of moving constant I into a data register.  */

static int
const_int_cost (HOST_WIDE_INT i)
{
  switch (m68k_const_method (i))
    {
    case MOVQ:
      /* Constants between -128 and 127 are cheap due to moveq.  */
      return 0;
    case MVZ:
    case MVS:
    case NOTB:
    case NOTW:
    case NEGW:
    case SWAP:
      /* Constants easily generated by moveq + not.b/not.w/neg.w/swap.  */
      return 1;
    case MOVL:
      return 2;
    default:
      gcc_unreachable ();
    }
}


bool
m68k_68000_10_costs (rtx x, machine_mode mode, int outer_code,
		int opno,
		int *total, bool speed )
{
  int code = GET_CODE (x);

  /*
   *  SBF: defining a cost model for the 68000.
   *
   *  Use the cycles from MC68000UM.pdf.
   *
   */
  int total2 = 0;
  switch (code)
    {
    case CALL:
      {
	rtx a = XEXP(x, 0);
	if (MEM_P(a))
	  {
	    rtx b = XEXP(a, 0);
	    if (REG_P(b))
	      {
		*total = 16;
		return true;
	      }
	    if (GET_CODE(b) == PLUS)
	      {
		if (REG_P(XEXP(b, 0)))
		  {
		    *total = 18;
		    return true;
		  }
	      }
	    else if (SYMBOL_REF_P(b) || GET_CODE(b) == CONST_INT)
	      {
		tree decl = SYMBOL_REF_DECL (b);

		*total = 20;
		return true;
	      }
	  }
	*total = 22;
	return true;
      }
    case EQ:
    case NE:
    case GE:
    case GT:
    case GTU:
    case LE:
    case LT:
    case CC0:
      return false;
    case ZERO_EXTRACT:
      if (outer_code == COMPARE && GET_CODE (XEXP (x, 1)) == CONST_INT && INTVAL (XEXP (x, 1)) == 1)
      {
    	*total = 2;
    	return true;
      }
      return false;
    case POST_INC:
    case TRUNCATE:
      *total = 0;
      return true;
    case PRE_DEC:
      *total = 2;
      return true;
    case REG:
      *total = opno ? 4 : 0;
      return true;
    case SYMBOL_REF:
      *total = 12;
      return true;
    case IF_THEN_ELSE:
      *total = 10;
      return true;
    case SET:
      {
	rtx a = XEXP(x, 0);
	rtx b = XEXP(x, 1);
	if (m68k_68000_10_costs (a, mode, code, 0, total, speed)
	    && m68k_68000_10_costs (b, mode, code, 1, &total2, speed))
	  {
	    *total += total2;
	    if (MEM_P(a) && GET_CODE(b) == CONST_INT && INTVAL(b) == 0)
	      {
		// penalty for clr
		*total += GET_MODE_SIZE(mode) > 2 ? 8 : 4;
		if (GET_CODE(XEXP(a, 0)) == PRE_DEC)
		  *total += 2;
	      }
	    return true;
	  }
	break;
      }
    case CONST:
      {
	rtx a = XEXP(x, 0);
	if (GET_CODE(a) == PLUS && SYMBOL_REF_P(XEXP(a,0)) && GET_CODE(XEXP(a, 1)) == CONST_INT)
	  {
	    *total = 12;
	    return true;
	  }
	*total = 0;
	break;
      }
    case MULT:
      {
	/* umul, smul or call to __mulsi3? */
	rtx a = XEXP(x, 0);
	rtx b = XEXP(x, 1);

	if (GET_CODE(b) == CONST_INT)
	  {
	    int n = INTVAL(b);
	    if (n < 0) n = -n;
	    int x = 0;
	    unsigned y = 1;
	    for (; y < n; y += y, ++x)
	      ;
	    if (y == n)
	      {
		*total = (GET_MODE_SIZE(mode) > 2 ? 8 : 6);
		*total += 2 * x;
		return true;
	      }
	  }

	/* if there is an extended HImode, mul.w might be a candidate. */
	if (GET_CODE (a) == ZERO_EXTEND
	     || GET_CODE (a) == SIGN_EXTEND)
	  {
	    *total = 0;
	    mode = HImode;
	  }
	else
	if (!m68k_68000_10_costs (a, mode, code, 0, total, speed))
	  break;

	if (speed)
	  {
	    int f = GET_MODE_SIZE(mode) > 2 ? 150 : 50;
	    if (GET_CODE(b) == CONST_INT && GET_MODE_SIZE(mode) == 2)
	      {
		unsigned i = INTVAL(b);
		int bits = 0, l = 0;
		if (i > 0)
		  {
		    if (GET_CODE (a) == ZERO_EXTEND)
		      while (i)
			{
			  ++bits;
			  i>>=1;
			}
		    else // SIGN_EXTEND
		      while (i || l)
			{
			  if ((i&1) != l)
			    {
			      l = !l;
			      ++bits;
			    }
			  i >>= 1;
			}

		    f = 38 + 2 * bits;
		  }
	      }
	    *total += f;
	  }
	else
	  *total = GET_MODE_SIZE(mode) > 2 ? 48 : 16;

	return true;
      }
    case DIV:
    case UDIV:
    case MOD:
    case UMOD:
      {
	rtx a = XEXP(x, 0);
	if (m68k_68000_10_costs (a, mode, code, 0, total, speed))
	  {
	    if (speed)
	      *total += GET_MODE_SIZE(mode) > 2 ? 410 : 136;
	    else
	      *total += GET_MODE_SIZE(mode) > 2 ? 108 : 36;
	    return true;
	  }
	break;
      }
    case ASHIFT:
    case ASHIFTRT:
    case LSHIFTRT:
      {
	rtx a = XEXP(x, 0);
	rtx b = XEXP(x, 1);
	if (REG_P(a))
	  {
	    if (GET_CODE(b) == CONST_INT)
	      {
		*total = (GET_MODE_SIZE(mode) > 2 ? 8 : 6);
		*total += 2 * INTVAL(b);
		return true;
	      }
	    *total = (GET_MODE_SIZE(mode) > 2 ? 8 : 6) ;
	    if (speed)
	      *total += 16;
	    return true;
	  }
	if (m68k_68000_10_costs (XEXP(x, 0), mode, code, 0, total, speed))
	  {
	    *total += 8;
	    return true;
	  }
	break;
      }
    case SUBREG:
    case STRICT_LOW_PART:
      return m68k_68000_10_costs (XEXP(x, 0), GET_MODE(XEXP(x, 0)), code, 0, total, speed);
    case ZERO_EXTEND:
    case SIGN_EXTEND:
      if (m68k_68000_10_costs (XEXP(x, 0), GET_MODE(XEXP(x, 0)), code, 0, total, speed))
	{
	  *total += 8;
	  return true;
	}
      return false;
    case CONST_INT:
      {
	int kind = const_int_cost (INTVAL(x));
	if (kind == 0)
	  *total = 4;
	else if (kind == 1)
	  *total = GET_MODE_SIZE(mode) > 2 ? 10 : 8;
	else
	  *total = GET_MODE_SIZE(mode) > 2 ? 12 : 8;
	return true;
      }
    case NEG:
    case NOT:
      {
	rtx a = XEXP(x, 0);
	if (REG_P(a))
	  {
	    *total = GET_MODE_SIZE(mode) > 2 ? 6 : 4;
	    return true;
	  }
	if (m68k_68000_10_costs (a, mode, code, 0, total, speed))
	  {
	    *total += GET_MODE_SIZE(mode) > 2 ? 12 : 8;
	    return true;
	  }
	break;
      }
    case MINUS:
    case PLUS:
      {
	/* add const,reg, add ea,reg, add reg, mem */
	rtx a = XEXP(x, 0);
	rtx b = XEXP(x, 1);

	// handle lea n(ax,dy),az
	if (GET_CODE(a) == PLUS && GET_CODE(b) == CONST_INT && (unsigned)(INTVAL(b) + 128) < 255)
	  {
	    if (REG_P(XEXP(a,0)) && REG_P(XEXP(a,1)))
	      {
		*total = 12;
		return true;
	      }
	  }

	// handle lea n(ax),ay
	if (REG_P(a) && REGNO(a) >= 8 && REGNO(a) < 16) {
	    if (GET_CODE(b) == CONST && GET_CODE(XEXP(b, 0)) == UNSPEC)
	    {
		    *total = 8; // 12 - 4
		    return true;
	    }
	    if (GET_CODE(b) == CONST_INT && (unsigned)(INTVAL(b) + 32768) < 65535)
	    {
		    *total = 8;
		    return true;
	    }
	}

	// handle add reg,reg
	if (REG_P(a) && REG_P(b))
	  {
	    *total = GET_MODE_SIZE(mode) > 2 ? 8 : 4;
	    return true;
	  }

	if (GET_CODE(b) == CONST_INT)
	  {
	    int i = INTVAL(b);
	    if (REG_P(a) && (USE_MOVQ(i) || (REGNO(a) >= 8 && REGNO(a) < 16 && (unsigned)(INTVAL(b) + 32768) < 65535)))
	      *total = GET_MODE_SIZE(mode) > 2 ? 8 : 4;
	    else
	      *total = GET_MODE_SIZE(mode) > 2 ? 16 : 8;
	    return true;
	  }

	if (GET_CODE(b) == SYMBOL_REF ||
		    (GET_CODE(b) == CONST && GET_CODE(XEXP(b,0)) == PLUS && GET_CODE(XEXP(XEXP(b,0),0)) == SYMBOL_REF))
	{
	    *total = 16;
	    return true;
	}

	if (!m68k_68000_10_costs (a, mode, code, 0, total, speed))
	  break;
	if (!m68k_68000_10_costs (b, mode, code, 1, &total2, speed))
	  break;

	*total += total2 + (GET_MODE_SIZE(mode) > 2 ? 6 : 4);
	return true;
      }
    case COMPARE:
    case AND:
    case XOR:
    case IOR:
      {
	/* add const,reg, add ea,reg, add reg, mem */
	rtx a = XEXP(x, 0);
	rtx b = XEXP(x, 1);
	if (GET_CODE(b) == CONST_INT)
	  {
	    if (REG_P(a))
	      {
		*total = GET_MODE_SIZE(mode) > 2 ? (code == AND ? 14 : 16) : 8;
		return true;
	      }
	    if (m68k_68000_10_costs (a, mode, code, 0, total, speed))
	      {
		*total += GET_MODE_SIZE(mode) > 2 ? 20 : 12;
		return true;
	      }
	  }
	else
	if (REG_P(a))
	  {
	    if (m68k_68000_10_costs (b, mode, code, 1, total, speed))
	      {
		*total += GET_MODE_SIZE(mode) > 2 ? 4 : (code == XOR ? 8 : 6);
		return true;
	      }
	  }
	else
	if (REG_P(b))
	  if (m68k_68000_10_costs (a, mode, code, 1, total, speed))
	    {
	      *total += GET_MODE_SIZE(mode) > 2 ? 12 : 8;
	      return true;
	    }

	if (m68k_68000_10_costs (a, mode, code, 0, total, speed)
	    && m68k_68000_10_costs (b, mode, code, 1, &total2, speed))
	  {
	    *total += total2 + (GET_MODE_SIZE(mode) > 2 ? 4 : (code == XOR ? 8 : 6));
	    return true;
	  }
	break;
      }
    case MEM:
      {
      /* simple but not exact */
	rtx y = XEXP(x, 0);
	int yc = GET_CODE(y);
	if (yc == REG || yc == PRE_INC || yc == POST_INC || yc == POST_DEC || yc == PRE_DEC)
	  *total = 4;
	else if (yc == CONST_INT || yc == SYMBOL_REF || (yc == PLUS && GET_CODE(XEXP(y, 0)) == SYMBOL_REF))
	  *total = 12;
	else if (yc == PLUS && GET_CODE(XEXP(y, 0)) == PLUS)
	  *total = 10;
	else
	  *total = 8;

	if (mode != QImode && mode != HImode)
	  *total += 4;

	if (opno && yc == PRE_DEC)
	  *total += 2;

	return true;
      }
    default:
      break;
    }
  return false;
}
